;+
; NAME:
;    THRESHOLD_QUANTILE
;
; PURPOSE:
;    This function estimates the quantiles of the given thresholds in the input 
;    data array.
;
; CATEGORY:
;    Statistics
;
; CALLING SEQUENCE:
;    result = threshold_quantile( data, threshold )
;
; INPUTS:
;    DATA:  A numerical array of data, of size N_DATA.
;    THRESHOLD:  A numerical vector listing thresholds to find in DATA and for 
;        which to estimate their quantiles.  Of length N_THRESHOLD
;
; KEYWORD PARAMETERS:
;    PRESORTED:  If set then the function assumes that the values in DATA have 
;        already been sorted in ascending order, thus running more efficiently. 
;        The default is to assume that they have not been sorted and thus to 
;        spend some time sorting them.
;    WEIGHT:  An optional float vector of length N_DATA containing weighting 
;        factors to apply to the values in DATA when estimating the cumulative 
;        density function.  The default is to have no weights.  This vector 
;        does not need to be normalised on input.
;
; OUTPUTS:
;    RESULT:  A numerical array containing the N_THRESHOLD quantile values 
;        corresponding to the N_THRESHOLD thresholds in DATA as specified in 
;        THRESHOLD.
;
; USES:
;    -
;
; PROCEDURE:
;    This function essentially inverts the method used in 
;    quantile_threshold.pro.
;
; EXAMPLE:
;    data = randomn( 2, 100 )
;    threshold = [ -1., 1. ]
;    result = threshold_quantile( data, threshold )
;
; MODIFICATION HISTORY:
;    Written by:  Daithi Stone (dstone@lbl.gov), 2012-06-01
;    Modified:  DAS, 2012-07-05 (Added PRESORTED keyword)
;    Modified:  DAS, 2021-02-10 (Added WEIGHT keyword;  Switched to linear 
;        extrapolation for bottom and top empirical quantile cases)
;-

FUNCTION THRESHOLD_QUANTILE, $
    DATA, THRESHOLD, $
    WEIGHT=weight, $
    PRESORTED=presorted_opt

;***********************************************************************
; Constants and checks

; Count the thresholds
n_threshold = n_elements( threshold )

;***********************************************************************
; Estimate the quantiles

; Initialise the output vector of thresholds
quantile = fltarr( n_threshold )

; Identify valid values in DATA
id_data = where( finite( data ) eq 1, n_id_data )
if n_id_data eq 0 then stop
; Sort good values in data
data_sort = data[id_data]
if keyword_set( weight ) then weight_sort = weight[id_data]
if not( keyword_set( presorted_opt ) ) then begin
  id = sort( data_sort )
  data_sort = data_sort[id]
  if keyword_set( weight ) then weight_sort = weight_sort[id]
endif
; Normalise weights
if keyword_set( weight ) then weight_sort = weight_sort / total( weight_sort )

; Iterate through thresholds
for i_threshold = 0, n_threshold - 1 do begin
  ; Find the nearest data value lower than the threshold
  id_low = max( where( data_sort le threshold[i_threshold] ) )
  ; If there is no value lower
  if id_low eq -1 then begin
    ; Extrapolate linearly
    if keyword_set( weight ) then begin
      quantile[i_threshold] = weight_sort[0] / 2. $
          + ( threshold[i_threshold] - data_sort[0] ) * weight_sort[0]
    endif else begin
      quantile[i_threshold] = 1. / n_id_data / 2. $
          + ( threshold[i_threshold] - data_sort[0] ) / n_id_data
    endelse
    if quantile[i_threshold] lt 0. then quantile[i_threshold] = 0.
  ; If there is no higher value
  endif else if id_low eq n_id_data - 1 then begin
    ; Extrapolate linearly
    if keyword_set( weight ) then begin
      quantile[i_threshold] = 1. - weight_sort[n_id_data-1] / 2. $
          + ( threshold[i_threshold] - data_sort[n_id_data-1] ) $
          * weight_sort[n_id_data-1]
    endif else begin
      quantile[i_threshold] = 1. - 1. / n_id_data / 2. $
          + ( threshold[i_threshold] - data_sort[n_id_data-1] ) / n_id_data
    endelse
    if quantile[i_threshold] gt 1. then quantile[i_threshold] = 1.
  ; If we are not at the ends of the samples
  endif else begin
    ; Interpolate quantile
    if keyword_set( weight ) then begin
      quantile[i_threshold] = total( weight_sort[0:id_low] ) $
          - weight_sort[id_low] / 2. $
          + ( threshold[i_threshold] - data_sort[id_low] ) $
          / ( data_sort[id_low+1] - data_sort[id_low] ) * weight_sort[id_low+1]
    endif else begin
      quantile[i_threshold] = ( float( id_low ) + 0.5 $
          + ( threshold[i_threshold] - data_sort[id_low] ) $
          / ( data_sort[id_low+1] - data_sort[id_low] ) ) $
          / n_id_data
    endelse
  endelse
endfor

;***********************************************************************
; The end

return, quantile
END
