;+
; NAME:
;    LINE_LEGEND
;
; PURPOSE:
;    This procedure plots a legend for line plots.
;
; CATEGORY:
;    Graphics
;
; CALLING SEQUENCE:
;    LINE_LEGEND, Position, Label
;
; INPUTS:
;    Position:  A 2-element vector containing the position ([x,y]) in normal 
;        coordinates of the lower left corner of the box containing the legend.
;    Label:  A vector, of type string and length N_LABEL, containing the 
;        labels for each line.
;
; OUTPUTS:
;    LIMIT
;
; KEYWORD PARAMETERS:
;    BACKGROUND:  The colour index of a background for the legend box.  If not 
;        set, no background is drawn.
;    BORDER_COLOR:  An input integer giving the colour index of the border 
;        line.  The default is !p.color.
;    CHARSIZE:  The size of the label characters, of type floating point.  The 
;        default is the IDL default (!p.charsize).
;    CHARSIZ_TITLE:  An optional scalar float specifying the size of the 
;        characters in the title.  The default is CHARSIZE.
;    COLOR:  A vector, of type integer, containing the colour index values of 
;        the lines.  The default is the IDL default (!p.color).
;    FONT:  An integer specifying the graphics font to use.  The default is the 
;        IDL default (!p.font).
;    LENGTH:  The length of the lines in normal coordinates.
;    LIMIT:  An output floating point vector containing the coordinates of the 
;        edges of the legend box.  The format is [ left, bottom, right, top ].
;    LINESTYLE:  A vector, of type integer, containing the linestyle index 
;        value for each line.  The default is the IDL default (!p.linestyle).
;    NOBORDER:  If set then no border line is drawn around the legend.  The 
;        default is for a border box of colour BORDER_COLOR.
;    PSYM:  A vector, of type integer, containing the symbol codes.  The 
;        default is the IDL default (!p.psym).
;    THICK:  A vector, of type integer, containing the line thickness value for 
;        each line.  The default is the IDL default (!p.thick).
;    TITLE:  A string containing the title of the legend.
;    USERSYM_DEF:  An optional float array of size N_POINT,2,N_LABEL containing 
;        the definition of the symbol to use against each of the labels.  This 
;        is only used if PSYM=8.
;
; USES:
;    -
;
; PROCEDURE:
;    This procedure uses the input values to construct an appropriate box 
;    containing the legend for a line plot.
;
; EXAMPLE:
;    Create a legend for a two-line plot (colours red and green).
;      tek_color
;      line_legend, [0.2,0.2], ['Red','Green'], color=[2,3], title='Legend'
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (stoned@atm.ox.ac.uk), 2000-09-18.
;    Modified:  DAS, 2003-02-05 (added FONT and PSYM keywords, converted LINE 
;        keyword to LINESTYLE).
;    Modified:  DAS, 2003-06-25 (added BACKGROUND keyword).
;    Modified:  DAS, 2005-04-12 (fixed output bugs in CHARSIZE, COLOR, 
;        LINESTYLE, PSYM, THICK keywords)
;    Modified:  DAS, 2009-10-01 (fixed so only one symbol plotted per label 
;        instead of two when PSYM given)
;    Modified:  DAS, 2010-02-19 (fixed bug which always set PSYM)
;    Modified:  DAS, 2010-09-16 (added NOBORDER keyword option, LIMIT keyword 
;        output;  removed use of var_type.pro;  modified documentation format)
;    Modified:  DAS, 2010-09-26 (fixed bug in PSYM keyword implementation;  
;        added BORDER_COLOR keyword)
;    Modified:  DAS, 2013-02-01 (set text to print in BORDER_COLOR colour)
;    Modified:  DAS, 2015-01-08 (fixed bug in drawing border when BACKGROUND
;        is input)
;    Modified:  DAS, 2021-02-09 (Added USERSYM_DEF keyword input)
;    Modified:  DAS, 2024-01-15 (Added capability to deal with !C hard return 
;        symbols as hard returns;  Added CHARSIZ_TITLE keyword input)
;-

;***********************************************************************

PRO LINE_LEGEND, $
    Position, $
    Label, $
    BACKGROUND=background, BORDER_COLOR=border_color, NOBORDER=noborder_opt, $
    CHARSIZE=charsize, $
    COLOR=color, $
    FONT=font, $
    LENGTH=length, $
    LIMIT=limit, $
    LINESTYLE=linestyle, $
    PSYM=psym, $
    THICK=thick, $
    TITLE=title, CHARSIZ_TITLE=charsize_title, $
    USERSYM_DEF=usersym_def

;***********************************************************************
; Variables and Options

; Number of lines
n_label = n_elements( label )

; The inter-bullet vertical space
cr_bullet = 1.5
; The intra-bullet vertical space
cr_line = 1.2

; Line colours (COLOR keyword)
if not( keyword_set( color ) ) then begin
  color0 = !p.color + intarr( n_label )
endif else begin
  color0 = color
  ; Assure sufficient number defined
  if n_elements( color0 ) lt n_label then begin
    color0 = color0[0] + intarr( n_label )
  endif
endelse

; Graphics font (FONT keyword)
if n_elements( font ) eq 0 then font = !p.font

; Line styles (LINESTYLE keyword)
if not( keyword_set( linestyle ) ) then begin
  linestyle0 = !p.linestyle + intarr( n_label )
endif else begin
  linestyle0 = linestyle
  ; Assure sufficient number defined
  if n_elements( linestyle0 ) lt n_label then begin
    linestyle0 = linestyle0[0] + intarr( n_label )
  endif
endelse

; Line thicknesses (THICK keyword)
if not( keyword_set( thick ) ) then begin
  thick0 = !p.thick + intarr( n_label )
endif else begin
  thick0 = thick
  ; Assure sufficient number defined
  if n_elements( thick0 ) lt n_label then begin
    thick0 = thick0[0] + intarr( n_label )
  endif
endelse

; Symbols (PSYM keyword)
if not( keyword_set( psym ) ) then begin
  psym0 = !p.psym + intarr( n_label )
endif else begin
  psym0 = psym
  ; Assure sufficient number defined
  if n_elements( psym0 ) lt n_label then psym0 = psym0[0] + intarr( n_label )
  ; Ensure we have required user-defined symbol definition
  if max( psym0 eq 8 ) eq 1 then begin
    if not( keyword_set( usersym_def ) ) then stop
  endif
endelse

; Line lengths (LENGTH keyword)
if not( keyword_set( length ) ) then length = 0.02

; Determine the legend box corner positions
xpos = position[[0,0]]
ypos = position[[1,1]]
if !p.multi[1] ne 0 then begin
  if !p.multi[0] eq 0 then begin
    pmulti0 = !p.multi[1] * !p.multi[2]
  endif else begin
    pmulti0 = !p.multi[0]
  endelse
  xpos = ( pmulti0 + !p.multi[1] - 1 ) / !p.multi[1] $
      - pmulti0 / 1. / !p.multi[1] + xpos / !p.multi[1]
endif
if !p.multi[2] ne 0 then begin
  if !p.multi[0] eq 0 then begin
    pmulti0 = !p.multi[1] * !p.multi[2]
  endif else begin
    pmulti0 = !p.multi[0]
  endelse
  ypos = ( ypos + ( pmulti0 - 1 ) / !p.multi[1] ) / 1. / !p.multi[2]
endif

; Character scale
if not( keyword_set( charsize ) ) then begin
  charsize0 = !p.charsize
endif else begin
  charsize0 = charsize
endelse
if charsize0 eq 0. then charsize0 = 1.
if !p.multi[2] gt 1 then charsize0 = charsize0 / !p.multi[2]
spacing = 1. * charsize0 * !d.y_ch_size / !d.y_size
; The title's character size
if not( keyword_set( charsize_title ) ) then charsize_title = charsize0

; Determine the number of carriage returns for each label and maximum string 
; length
label_lenmax = 0
label_strmax = ''
label_return = intarr( n_label )
for i_label = 0, n_label - 1 do begin
  temp_str = strsplit( label[i_label], '!C', regex=1, extract=1, $
      count=n_temp_str )
  temp = max( strlen( temp_str ) )
  if temp gt label_lenmax then begin
    label_lenmax = temp
    id = where( strlen( temp_str ) eq temp )
    label_strmax = temp_str[id[0]]
  endif
  label_return[i_label] = n_temp_str - 1
endfor

;***********************************************************************
; Plot Legend

; Re-determine the legend box corner positions
ypos[1] = ypos[0] $
    + ( cr_bullet * n_label + cr_line * total( label_return ) + 0.5 ) * spacing
if n_elements( background ) ne 0 then begin
  xyouts, xpos[0]+length/2., ypos[0]+spacing/2., label_strmax, $
      normal=1, charsize=charsize0, width=str_width, font=font
  xpos[1] = xpos[0] + 2.5 * length + str_width
  ; Clear area for legend box
  polyfill, xpos[[0,1,1,0,0]], ypos[[0,0,1,1,0]]+[0,0,1,1,0]*2.*spacing, $
      color=background, normal=1
  ; Draw the legend border
  if not( keyword_set( noborder_opt ) ) then begin
    ;plots, xpos[[0,1,1,0,0]], ypos[[0,0,1,1,0]], color=!p.color, /normal
    plots, xpos[[0,1,1,0,0]], ypos[[0,0,1,1,0]], color=border_color, normal=1
  endif
endif

; Draw the bullets
temp_ypos = ypos[0] + ( 1. - cr_bullet ) * spacing
for i_label = 0, n_label - 1 do begin
  id_label = n_label - i_label - 1
  psym_opt = keyword_set( psym0[id_label] )
  if psym0[id_label] eq 8 then begin
    usersym, usersym_def[*,0,id_label], usersym_def[*,1,id_label], fill=1
  endif
  temp_ypos = temp_ypos $
      + ( cr_bullet + cr_line * label_return[id_label] ) * spacing
  plots, xpos[0]+length+[-1,1]*length/2.*(1-psym_opt), [temp_ypos,temp_ypos], $
      normal=1, color=color0[id_label], linestyle=linestyle0[id_label], $
      thick=thick0[id_label], psym=psym0[id_label]
endfor

; Label the bullets and determine the longest label
str_width = 0.
temp_ypos = ypos[0] + ( 0.75 - cr_bullet ) * spacing
for i_label = 0, n_label - 1 do begin
  temp_label = strsplit( label[n_label-1-i_label], '!C', extract=1, regex=1 )
  n_temp_label = label_return[n_label-1-i_label] + 1
  temp_ypos = temp_ypos + ( cr_bullet - cr_line ) * spacing
  for i_line = 0, n_temp_label - 1 do begin
    temp_ypos = temp_ypos + cr_line * spacing
    xyouts, xpos[0]+2*length, temp_ypos, temp_label[n_temp_label-1-i_line], $
        normal=1, charsize=charsize0, alignment=0, font=font, $
        color=border_color, width=str_width_1
    if str_width_1 gt str_width then str_width = str_width_1
  endfor
endfor

; Legend box
if n_elements( background ) eq 0 then begin
  ; Re-determine the legend box corner positions
  xpos[1] = xpos[0] + 2.5 * length + str_width
  ; Draw the legend border
  if not( keyword_set( noborder_opt ) ) then begin
    ;plots, xpos[[0,1,1,0,0]], ypos[[0,0,1,1,0]], color=!p.color, /normal
    plots, xpos[[0,1,1,0,0]], ypos[[0,0,1,1,0]], color=border_color, normal=1
  endif
endif
; Pass edges to output
limit = [ xpos[0], ypos[0], xpos[1], ypos[1] ]

; Title
if keyword_set( title ) then begin
  xyouts, (xpos[1]+xpos[0])/2., ypos[1]+spacing/2., title, /normal, $
          charsize=charsize_title, alignment=0.5, font=font, color=border_color
endif

;***********************************************************************
; The End

return
END
