;+
; NAME:
;    FILTER
;
; PURPOSE:
;    This function returns a smoothed version of the input vector.
;
; CATEGORY:
;    Time Series Analysis
;
; CALLING SEQUENCE:
;    result = filter( vector, [width], [window] )
;
; INPUTS:
;    VECTOR:  An vector of type floating point and length N containing the 
;        input data.
;    WIDTH:  The optional width, of type integer, of the smoothing window.  
;        Required if not input via a keyword input.
;    WINDOW:  An optional string containing the name of the smoothing window to 
;        use.  Options are 'boxcar', 'gaussian', 'hanning', 'triangle'.  The 
;        default is a boxcar window.
;
; KEYWORD PARAMETERS:
;    BOXCAR:  Sets the smoothing window to a boxcar filter.  This is the 
;        default.  If set to a value, it replaces WIDTH.
;    EDGE_TRUNCATE:  Set this keyword to apply the smoothing to all points.  If 
;        the neighbourhood around a point includes a point outside the array, 
;        the nearest edge point is used to compute the smoothed result.  If 
;        EDGE_TRUNCATE is not set, the points near the end are replaced with 
;        NaNs.
;    FILTER:  A vector containing the filter window to use.  This overrides the 
;        window requested in the WINDOW input.  This also returns the filter 
;        after use.
;    FRAC_WINDOW_MIN:  An optional floating point fraction (between 0 and 1) 
;        specifiying the minimum weighted fraction of the filter window 
;        required to have data in order to calculate a value.  Only applies 
;        when NAN is set.
;    NAN:  Set this keyword to ignore NaN values in the input array, provided 
;        there is at least one defined value nearby.  The default is to return 
;        NaNs wherever they occur.
;    NO_NAN:  Obsolete version of NAN keyword retained for compatibility but no 
;        longer used.
;    START_INDEX:  The location of the centre of the window for the first 
;        averaged output value, in units of Vector indices.  Values must be 
;        greater than 0.  The default is 0.
;    STEP:  An integer defining the step size for window translation, in units 
;        of VECTOR indices.  The default is 1.
;    TRIANGLE:  Sets the smoothing window to a triangle filter.  The default
;		is a boxcar filter.  If set to a value, it replaces WIDTH.
;    WRAP_EDGES:  If set, the vector is treated as being cyclic and the ends 
;        are joined together when smoothing.
;
; OUTPUTS:
;    RESULT:  Returns the smoothed version of Vector.
;
; USES:
;    dimension.pro
;    filter_window.pro
;    plus.pro
;
; PROCEDURE:
;    This function manually convolves the input vector with the filter.
;
; EXAMPLE:
;    Create a vector of daily data and a sinusoid for a year.
;      x = randomn( seed, 365 ) + sin( 6.28 * findgen( 365 ) / 365. )
;    Smooth x with a boxcar filter of 7 days, wrapping the edges together.
;      result = filter( x, 7, 'boxcar', /wrap_edges )
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone, 2000-06-28.
;    Modified:  DAS, 2000-07-06 (removed LENGTH.pro, added DIMENSION.pro).
;    Modified:  DAS, 2001-01-24 (added reform(vector) line).
;    Modified:  DAS, 2003-11-19 (added Window input and FILTER keyword, edited 
;        style).
;    Modified:  DAS, 2004-07-14 (allowed input of complex vectors)
;    Modified:  DAS, 2005-08-05 (replaced sum.pro use with IDL's total; added 
;        NAN keyword for consistency with other routines; removed functionality 
;        of NO_NAN keyword; removed constants.pro use; streamlined coding)
;    Modified:  DAS, 2005-10-21 (removed requirement of odd WIDTH;  allowed 
;        operation with WIDTH=1)
;    Modified:  DAS, 2009-01-05 (added and implemented START_INDEX and STEP 
;        keywords)
;    Modified:  DAS, 2019-09-13 (Added FRAC_WINDOW_MIN keyword input;  Updated 
;        documentation format)
;    Modified:  DAS, 2020-05-20 (Switched to faster usage of IDL convol.pro;  
;        Added OLD keyword option to use old convol.pro-less method)
;-

;***********************************************************************

FUNCTION FILTER, $
    VECTOR, $
    WIDTH, $
    WINDOW, $
    BOXCAR=boxcar, TRIANGLE=triangle, $
    FILTER=filt, $
    FRAC_WINDOW_MIN=frac_window_min, $
    START_INDEX=start_index, STEP=step, $
    WRAP_EDGES=wrapedgesopt, EDGE_TRUNCATE=edgetruncateopt, $
    NAN=nanopt, NO_NAN=nonanopt, OLD=old_opt

;***********************************************************************
; Constants, Variables, and Options

; Load absolute constants
nan = !values.f_nan

; Load smoothing window
if not( keyword_set( filt ) ) then begin
  filt = filter_window( width, window, boxcar=boxcar, triangle=triangle )
endif

; Edge-handling options
if keyword_set( wrapedgesopt ) then begin
  wrapedgesopt = 1
endif else begin
  wrapedgesopt = 0
endelse
if keyword_set( edgetruncateopt ) then begin
  edgetruncateopt = 1
endif else begin
  edgetruncateopt = 0
endelse

; NaN handling option
if keyword_set( nanopt ) then begin
  nanopt = 1
endif else begin
  nanopt = 0
endelse
; The default minimum weighted data requirement
if not( keyword_set( frac_window_min ) ) then begin
  if keyword_set( opt_opt ) then begin
    frac_window_min = 0.
  endif else begin
    ; There seem to be numerical issues with convol.pro, so this is numerically 
    ; equivalent to zero
    frac_window_min = 0.0001
  endelse
endif

; Half-length of filter
hwidth0 = ( width - 1 ) / 2
hwidth1 = width / 2

; Length of vector
vector = reform( vector )
n = n_elements( vector )

; The default position of the first application of the smoothing window
if n_elements( start_index ) eq 0 then start_index = 0
if start_index lt 0 then stop
; The default window stepping
if not( keyword_set( step ) ) then step = 1

; Output vector
n_out = n - start_index
if step ne 1 then begin
  n_out = ( n_out + start_index - start_index / width * width ) / step
endif
outvec = nan * float( vector[0:n_out-1] )

;***********************************************************************
; Smooth Vector

; The old code
if keyword_set( old_opt ) then begin

  ; Expanded version of Vector.
  ; This needs to be done to handle the edges.
  newvec = vector
  ; If EDGE_TRUNCATE is set
  if edgetruncateopt eq 1 then begin
    ; Pad ends with edge values
    if hwidth0 gt 0 then newvec = [ 0*fltarr(hwidth0)+vector[0], newvec ]
    if hwidth1 gt 0 then newvec = [ newvec, 0*fltarr(hwidth1)+vector[n-1] ]
  ; If WRAP_EDGES is set
  endif else if wrapedgesopt eq 1 then begin
    ; Pad ends with values from opposite end
    if hwidth0 gt 0 then newvec = [ vector[n-hwidth0:n-1], newvec ]
    if hwidth1 gt 0 then newvec = [ newvec, vector[0:hwidth1-1] ]
  ; Default method for handling edges
  endif else begin
    ; Pad ends with NaNs
    if hwidth0 gt 0 then newvec = [ 0*fltarr(hwidth0)+nan, vector ]
    if hwidth1 gt 0 then newvec = [ newvec, 0*fltarr(hwidth1)+nan ]
  endelse

  ; The minimum non-zero filter loading
  id = where( filt gt 0., n_id )
  if n_id eq 0 then stop
  filt_min = min( filt[id] )
  ; Iterate through all points
  for i_out = 0, n_out - 1 do begin
    ; Extract windowed segment
    temp = newvec[start_index+i_out*step:start_index+i_out*step+width-1]
    ; Determine weighted fraction of window with data
    temp_frac = total( filt * finite( temp ) )
    if temp_frac gt 1 - filt_min / 2. then begin
      ; Take weighted average
      outvec[i_out] = total( filt * temp )
    endif else if ( nanopt eq 1 ) and ( temp_frac ne 0. ) then begin
      if temp_frac ge frac_window_min then begin
        ; Take weighted average
        outvec[i_out] = total( filt * temp, nan=nanopt ) / temp_frac
      endif
    endif
  endfor

; The new convol.pro-dependent code
endif else begin

  ; Apply filter
  if hwidth1 ne hwidth0 then begin
    outvec = convol( [vector,nan], filt, edge_truncate=edgetruncateopt, $
        edge_wrap=wrapedgesopt, normalize=1, center=1 )
    outvec = outvec[1:n]
  endif else begin
    outvec = convol( vector, filt, edge_truncate=edgetruncateopt, $
        edge_wrap=wrapedgesopt, normalize=1, center=1 )
  endelse
  ; Deal with missing data
  if nanopt eq 0 then begin
    if hwidth1 ne hwidth0 then begin
      outvec_finite = convol( [ float( finite( vector ) ), 0. ], filt )
      outvec_finite = outvec_finite[1:n]
    endif else begin
      outvec_finite = convol( float( finite( vector ) ), filt )
    endelse
    id = where( outvec_finite + frac_window_min lt 1., n_id )
    if n_id gt 0 then outvec[id] = nan
  endif

endelse

;***********************************************************************
; The End

return, outvec
END
