;+
; NAME:
;    theilsen
;
; PURPOSE:
;    This function calculates the Theil-Sen slope estimator.
;
; CATEGORY:
;    Time Series Analysis
;
; CALLING SEQUENCE:
;    result = theilsen( [DATA_X,] DATA_Y )
;
; INPUTS:
;    DATA_Y:  The input vector of type integer or floating point and length 
;        N_DATA.
;
; OPTIONAL INPUTS:
;    DATA_X:  An optional ordinate vector of type integer or floating point.  
;        It must be of length N_DATA.  If not set, DATA_Y's elements are 
;        positioned at [0,1,2,...,N-1].
;
; KEYWORD PARAMETERS:
;    NAN:  If set, the function ignores NaNs (Not-a-Number) as missing values 
;        if any appear in DATA_X or DATA_Y.  The default is to return NaN.
;
; OUTPUTS:
;    Result:  Returns the Theil-Sen slope of DATA_Y(DATA_X) as a floating-point 
;        number.
;
; USES:
;    -
;
; PROCEDURE:
;    This function calculates the Theil-Sen slope directly.
;
; EXAMPLE:
;    Create a vector of 20 random Gaussian-distributed data elements.
;      data_y = randomn( seed, 20 )
;    Calculate the trend in data_y.
;      result = trend( data_y )
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2024-11-08
;-

;***********************************************************************

FUNCTION THEILSEN, $
    DATA_X, DATA_Y, $
    NAN=nan_opt

;***********************************************************************
; Variables, Constants, and Options

; Load constants
nan = !values.f_nan

; Input vectors
if n_params() eq 1 then begin
  data_y_use = data_x
endif else begin
  data_y_use = data_y
endelse
n_data = n_elements( data_y_use )

; Ordinate vector
if n_params() eq 1 then begin
  data_x_use = findgen( n_data )
endif else begin
  data_x_use = data_x
endelse
if n_elements( data_x_use ) ne n_data then begin
  print, 'Error in theilsen.pro:  Inconsistent input vector lengths'
  stop
endif

; Not-a-Number option
if keyword_set( nan_opt ) then begin
  nan_opt = 1
endif else begin
  nan_opt = 0
endelse

;***********************************************************************
; Setup

; Remove NaNs if NAN_OPT is set
if nan_opt eq 1 then begin
  ; Search for good values
  id = where( ( finite( data_y_use ) eq 1 ) and ( finite( data_x_use ) eq 1 ), $
      n_id )
  ; If we have some good values then remove the bad ones
  if n_id gt 1 then begin
    data_x_use = data_x_use[id]
    data_y_use = data_y_use[id]
  ; If we have no good values then return
  endif else begin
    return, nan
  endelse
endif

;***********************************************************************
; Calculate slope

; Subtract mean (gives better accuracy)
mean_y = mean( data_y_use )
data_y_use = data_y_use - mean_y
mean_x = mean( data_x_use )
data_x_use = data_x_use - mean_x

; Initialise vector containing slopes of different combinations of points
slope_val = fltarr( n_data * n_data )
; Initialise counter of slopes calculated
ctr_slope = 0
; Iterate through points (except last one)
for i_1 = 0, n_data - 2 do begin
  ; Iterate through other points (only do subsequent points)
  ; (Taking prior points would just repeat the exercise)
  for i_2 = i_1 + 1, n_data - 1 do begin
    ; If these points do not have the same ordinate value
    if data_x_use[i_1] ne data_x_use[i_2] then begin
      ; Calculate the slope and increment the counter
      slope_val[ctr_slope] = ( data_y_use[i_2] - data_y_use[i_1] ) $
          / ( data_x_use[i_2] - data_x_use[i_1] )
      ctr_slope = ctr_slope + 1
    endif
  endfor
endfor

; If no slopes were calculated, return NaN
if ctr_slope eq 0 then return, nan

; Take the median of all slopes
slope_val = median( slope_val[0:ctr_slope-1] )

;***********************************************************************
; The End

return, slope_val
END
