;+
; NAME:
;    tek_color_spectrum
;
; PURPOSE:
;    This function returns colour indices that make a spectrum out of the 
;    TEK_COLOR colours.
;
; CATEGORY:
;    Graphics
;
; CALLING SEQUENCE:
;    Result = tek_color_spectrum( [N_COLOR] )
;
; INPUTS:
;    -
;
; OPTIONAL INPUTS:
;    N_COLOR:  The number of indices to return.  Index values are interpolated 
;        between the default 12 values (so some values are repeated if 
;        N_COLOR>12).
;
; KEYWORD PARAMETERS:
;    COLORBREWER:  If set, then the base colours are taken from the 
;        colorbrewer2.org "Paired" table, rather than for the tek_color table.  
;        The indices of the strong colours in the "Paired" table are returned 
;        in RESULT, while the indices of the weak colours are returned in 
;        SHADE.  The default number of colors is 5, but this can be 
;        interpolated using N_COLOR too.
;    SHADE:  Returns the indices of the weak colours from the colorbrewer2.org 
;        "Paired" table.
;    START_INDEX:  A scalar integer specifying The color index at which to 
;        start this color table.  The default is 2 (after the foreground and 
;        background colors).
;
; OUTPUT:
;    RESULT:  The vector of colour index values.
;    SHADE
;
; USES:
;    -
;
; PROCEDURE:
;    This function returns a set vector of index values.
;
; EXAMPLE:
;    n_color = 12
;    result = tek_color_spectrum( n_color )
;    plot, findgen( n_color + 2 ), nodata=1
;    plots, findgen( n_color ) + 1, findgen( n_color ) + 1, psym=4, color=result
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone, 2002-09-11.
;    Modified:  DAS, 2004-07-21 (Added library documentation, NCOLORS input).
;    Modified:  DAS, 2023-11-03 (Added COLORBREWER keyword option and SHADE 
;        keyword output;  Incorporated loading of tek_color table into the 
;        function, rather than leaving it outside)
;    Modified:  DAS, 2023-11-10 (Added START_INDEX keyword input)
;-

FUNCTION TEK_COLOR_SPECTRUM, $
    N_COLOR, $
    START_INDEX=start_index, $
    SHADE=shade, $
    COLORBREWER=colorbrewer_opt

;***********************************************************************
; Create Colour Spectrum Index Vector

; For the colorbrewer2 "Paired" table
if keyword_set( colorbrewer_opt ) then begin
  color_rgb = [ [ 227, 26, 28 ], [ 255, 127, 0], [ 51, 160, 44], $
      [ 31, 120, 180 ], [ 106, 61, 154 ], [ 251, 154, 153 ], $
      [ 253, 191, 111 ], [ 178, 223, 138 ], [ 166, 206, 227 ], $
      [ 202, 178, 214 ] ]
  color_rgb = transpose( color_rgb )
  n_color_0 = n_elements( color_rgb[*,0] ) / 2
  color = indgen( n_color_0 )
; For the tek_color table
endif else begin
  color = [ 6, 13, 2, 8, 7, 9, 3, 10, 5, 11, 4, 12 ]
  n_color_0 = n_elements( color )
endelse

; The default start index
if n_elements( start_index ) eq 0 then start_index = 2

;***********************************************************************
; Option to Return Longer or Shorter Vector of Colour Indices

; If the number of colors to return has been specified
if keyword_set( n_color ) then begin
  ; For the colorbrewer2 "Paired" table
  if keyword_set( colorbrewer_opt ) then begin
    new_color = findgen( n_color ) / ( n_color - 1. ) * ( n_color_0 - 1. )
    new_rgb = fltarr( n_color * 2, 3 )
    for i_rgb = 0, 2 do begin
      new_rgb[0:n_color-1,i_rgb] = spline( color, $
          color_rgb[0:n_color_0-1,i_rgb], new_color )
      new_rgb[n_color:2*n_color-1,i_rgb] = spline( n_color_0 + color, $
          color_rgb[n_color_0:2*n_color_0-1,i_rgb], n_color_0 + new_color )
    endfor
    color_rgb = temporary( fix( new_rgb ) )
    color = indgen( n_color )
    shade = color + n_color
    id = where( color_rgb lt 0, n_id )
    if n_id gt 0 then color_rgb[id] = 0
    id = where( color_rgb gt 255, n_id )
    if n_id gt 0 then color_rgb[id] = 255
  ; For the tek_color table
  endif else begin
    id = round( findgen( n_color ) / ( n_color - 1. ) * ( n_color_0 - 1 ) )
    color = color[id]
  endelse
endif

;***********************************************************************
; Launch colour table

; For the colorbrewer2 "Paired" table
if keyword_set( colorbrewer_opt ) then begin
  ; Add a grey, because sometimes it is useful
  color_rgb = [ color_rgb, reform( [ 127, 127, 127 ], 1, 3 ) ]
  ; Set new colour table
  tvlct, color_rgb[*,0], color_rgb[*,1], color_rgb[*,2], start_index
  color = color + start_index
  shade = shade + start_index
; For the tek_color table
endif else begin
  tek_color
endelse

;***********************************************************************
; The End

return, color
END
