;+
; NAME:
;    regiondna_conf_sign.pro
;
; PURPOSE:
;    This function determines a confidence factor for the confidence metric 
;    used in the IPCC AR6 WGII detection and attribution assessments, based on 
;    comparisons of observed and required trend signs.
;
; CATEGORY:
;    REGIONAL D-AND-A CONFIDENCE
;
; CALLING SEQUENCE:
;    conf_factor = regiondna_conf_sign( obs_data, impact_sign=impact_sign )
;
; INPUT:
;    OBS_DATA:  A required float array, of size N_TIME,N_SOURCE, containing the 
;        observed data for determining trends.  N_TIME is the number of time 
;        steps, and N_SOURCE is the number of observational products.
;    FRAC_TIME_THRESH, IMPACT_SIGN
;
; KEYWORD PARAMETERS:
;    N_TIME_THRESH_GOOD:  An optional integer scalar specifying the minimum 
;        number of time steps in OBS_DATA for a given observational product 
;        in order for that time series to be considered for trend calculation.  
;        The default is 0.75*N_TIME.
;    IMPACT_SIGN:  A required integer scalar, either -1 or +1, specifying the 
;        required trend sign in the observed climate variable for a given 
;        impact.
;
; OUTPUT:
;    CONF_FACTOR:  A float scalar in the range [0,1] specifying the confidence 
;        factor by which to multiply the confidence metric in ipccar6_conf.pro, 
;        according to the trend sign comparison.
;
; USES:
;    plus.pro
;    sign.pro
;    trend.pro
;
; PROCEDURE:
;    This function calculates linear least-squares trends, determines their 
;    sign, and compares against the expected sign.  See Stone and Hansen (2016) 
;    for more details.
;
; EXAMPLE:
;    See regiondna_conf.pro.
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2015-03-23 (As 
;        part of hanseng_conf_assess_up.pro)
;    Modified:  DAS, 2020-09-09 (Adapted from hanseng_conf_assess_up.pro)
;-

;***********************************************************************

FUNCTION REGIONDNA_CONF_SIGN, $
    OBS_DATA, $
    IMPACT_SIGN=impact_sign, $
    N_TIME_THRESH_GOOD=n_time_thresh_good

;***********************************************************************
; Constants and options

; Determine the number of observed data sets
n_source = n_elements( obs_data[0,*] )
; Determine the number of time steps
n_time = n_elements( obs_data[*,0] )

; The default required fraction of time steps with valid data
if not( keyword_set( n_time_thresh_good ) ) then begin
  n_time_thresh_good = floor( 0.75 * n_time )
endif

;***********************************************************************
; Determine whether signs match

; Initialise vector of trend signs in observed data
obs_sign = intarr( n_source )

; Iterate through observed data sets
for i_source = 0, n_source - 1 do begin
  ; Extract this data set
  temp_obs_data = obs_data[*,i_source]
  id_good = where( finite( temp_obs_data ) eq 1, n_good )
  ; If there is insufficient data
  if n_good lt n_time_thresh_good then begin
    ; Mark as no-sign
    obs_sign[i_source] = 0
  ; If there is sufficient data
  endif else begin
    ; Determine the sign of the trend
    obs_sign[i_source] = sign( trend( temp_obs_data[id_good] ) )
  endelse
endfor

; Count for which impacts the required trend signs match the observed
id_match = where( obs_sign eq impact_sign, n_id_match )
; Calculate the confidence factor based on lack of matches in the trend sign
id_signed = where( obs_sign ne 0, n_id_signed )
conf_factor = ( float( n_id_match ) / n_id_signed - 0.5 ) * 2
conf_factor = plus( conf_factor ) * abs( conf_factor )

;***********************************************************************
; The end

return, conf_factor
END
