;+
; NAME:
;    ipccar6dna_conf_size.pro
;
; PURPOSE:
;    This function determines a degradation factor for the confidence metric 
;    used in the IPCC AR6 WGII detection and attribution assessments, based on 
;    region size.
;
; CATEGORY:
;    IPCC AR6 WGII DA
;
; CALLING SEQUENCE:
;    result = ipccar6dna_conf_size( region_area )
;
; INPUT:
;    REGION_AREA:  A float scalar specifying the area of the region in km^2.
;
; KEYWORD PARAMETERS:
;    CELL_REF:  An optional scalar float specifying the area of the average 
;        cell in the numerical grid of a climate model.  This function assumes 
;        a slightly diffusive numerical scheme, leading to an effective grid 
;        resolution of DIFFUSIVE_FACTOR^2 times CELL_REF.  If REGION_AREA 
;        equals this value, the function will return 0.75; and theoutput 
;        approaches 0.5 or 1 as REGION_AREA becomes much smaller or larger, 
;        respectively.  This is a new feature for the IPCC AR6 exercise.
;    DIFFUSIVE_FACTOR:  An optional scalar float specifying the diffusive 
;        factor to assume when calculating the effective cell area based on 
;        CELL_REF.  The default is 5.
;
; OUTPUT:
;    RESULT:  A float scalar in the range [0,1] specifying the degradation 
;        factor by which to multiply the confidence metric in ipccar6_conf.pro, 
;        according to the region size.
;
; USES:
;    -
;
; PROCEDURE:
;    This function calculates the degradation factor according to the formula 
;    used in Stone and Hansen (2016).
;
; EXAMPLE:
;    See ipccar6dna_conf.pro.
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2014-12-23 (As 
;        part of hanseng_conf_assess.pro)
;    Modified:  DAS, 2020-10-30 (Adapted from hanseng_conf_assess.pro)
;    Modified:  DAS, 2021-01-13 (Added DIFFUSIVE_FACTOR keyword input)
;-

;***********************************************************************

FUNCTION IPCCAR6DNA_CONF_SIZE, $
    REGION_AREA, $
    CELL_AREA=cell_area, $
    DIFFUSIVE_FACTOR=diffusive_factor

;***********************************************************************
; Constants

; The default diffusive factor
if not( keyword_set( diffusive_factor ) ) then diffusive_factor = 5.

; Define the default reference area (km^2)
if not( keyword_set( cell_area ) ) then begin
  area_ref = 1.e6
; Otherwise adopt a reference area of diffusive_factor^2 times the input cell 
; area
endif else begin
  area_ref = ( diffusive_factor ^ 2. ) * cell_area
endelse

;***********************************************************************

; Degrade confidence for small regions
degrade_factor =  0.5 + atan( region_area / area_ref ) / !pi

;***********************************************************************
; The end

return, degrade_factor
END

