;+
; NAME:
;    ipccar6dna_conf.pro
;
; PURPOSE:
;    This procedure generates a confidence assessment in the detection and 
;    attribution of observed climate change to anthropogenic emissions for the 
;    IPCC AR6 WGII.
;
; CATEGORY:
;    IPCC AR6 WGII DA
;
; CALLING SEQUENCE:
;    ipccar6dna_conf, impact_period=impact_period, $
;        impact_region_def=impact_region_def, impact_season=impact_season, $
;        impact_var_label=impact_var_label, version=version
;
; INPUTS:
;    DENSE_N_LONLAT_MIN, DEP_OPTIONS, DEP_SCENARIO, DIAGNOSTIC, 
;      GUESS_FILE_RESTRICT, IMPACT_DOMAIN, IMPACT_PERIOD, IMPACT_REGION_DEF, 
;      IMPACT_REGION_LAT, IMPACT_REGION_LON, IMPACT_REGION_MASK, 
;      IMPACT_REGION_PREPROCESSED, IMPACT_SEASON, IMPACT_VAR_LABEL, 
;      IMPACT_VAR_SIGN, INDEP_OPTION, INDEP_IN_SCENARIO, INDEP_OUT_SCENARIO, 
;      NOISE_OPTIONS, NOISE_SCENARIO, OPTIONS, VERSION, VERBOSE
;
; KEYWORD PARAMETERS:
;    DENSE_N_LONLAT_MIN:  The optional N_LONLAT_MIN keyword input for 
;        ipccar6dna_conf_density.pro.
;    DEP_OPTIONS:  An optional string vector specifying various criteria for 
;        selection of observational data sources.  For example, 
;        ['var_label=tas','source_label=GISTEMPv4'] would limit analysis to 
;        temperature from the GISTEMPv4 observational product.  See 
;        ipccar6dna_conf_define_source.pro for more details.  Note that it is 
;        sometimes necessary to specify the variable label, for instance 
;        if the observed variable is skin temperature when IMPACT_VAR_LABEL is 
;        'tas' (2-m air temperature).
;    DEP_SCENARIO:  An optional string specifying the type of data products to 
;        load for the dependent (generally observed) scenario, as specified in 
;        the <*_file> entries in ipccar6dna_map_source.xml.  The default is 
;        'observed'.
;    DIAGNOSTIC:  An optional vector string listing various diagnostic outputs 
;        to produce.  Supported values are:
;        * 'betas':  Plot the regression coefficients to a postscript file.
;        * 'p_resid':  Plot the p-values of the regression residuals to a 
;          postscript file.
;        * 'print':  Print all of the confidence factors to standard output.
;    GUESS_FILE_RESTRICT:  The GUESS_FILE_RESTRICT option for 
;        ipccar6dna_conf_load_data.pro.
;    IMPACT_AREA:  Returns a float array containing the areas of the N_REGION 
;        regions that have been assessed.
;    IMPACT_CONF_ATTRIB_MAJOR:  Returns a N_REGION string vector containing the 
;        confidence levels assessed for the attribution of a major role of an 
;        anthropogenic influence on observed climate variations.
;    IMPACT_CONF_DETECT:  Returns an N_REGION string vector containing the 
;        confidence levels assessed for the detection of an anthropogenic 
;        influence on observed climate variations.
;    IMPACT_CONF_FACTOR:  Returns an N_FACTOR,N_REGION float array reporting 
;        the numerical values calculated for the various steps (factors) that 
;        contributed to the confidence assessment.
;    IMPACT_CONF_LABEL:  Returns an N_FACTOR string vector labelling the steps 
;        in the confidence assessment as reported in the first dimension of 
;        the IMPACT_CONF_FACTOR output.
;    IMPACT_DOMAIN:  An optional string specifying the land-ocean domain to 
;        analyse.  Possible values are:
;        * 'atmos':  Atmospheric data with no land-ocean mask.
;        * 'atmos-land':  Atmospheric data over land only.
;        * 'atmos-ocean':  Atmospheric data over ocean only.
;        * 'ocean':  Ocean data in the ocean only.
;        The default is 'atmos'
;    IMPACT_PERIOD:  A required 2-element integer vector specifying the start 
;        and end years of the time period to load.  For instance [1991,2010] 
;        loads the 20 years between and including 1991 and 2010.
;    IMPACT_REGION_DEF:  An N_REGION string vector defining the regions to 
;        analyse.  See REGION_DEF keyword input in 
;        ipccar6dna_region_mask.pro for details, here using the "&"-delimited 
;        format when multiple definitions are needed for each region.  This is 
;        required unless IMPACT_REGION_MASK, IMPACT_REGION_LON, and 
;        IMPACT_REGION_LAT are input.
;    IMPACT_REGION_LAT:  Returns a float vector specifying the latitude 
;        coordinates for the IMPACT_REGION_MASK output array.  If instead 
;        IMPACT_REGION_MASK is input, then this is the required as input too.
;    IMPACT_REGION_LON:  Returns a float vector specifying the longitude 
;        coordinates for the IMPACT_REGION_MASK output array.  If instead 
;        IMPACT_REGION_MASK is input, then this is the required as input too.
;    IMPACT_REGION_MASK:  Returns the N_LON,N_LAT,N_REGION float array of 
;        region masks (ranging from 0 to 1) for the N_REGION regions to be 
;        analysed.  This can also be used to instead input the region masks, 
;        and is required with IMPACT_REGION_LAT and IMPACT_REGION_LON if 
;        IMPACT_REGION_DEF is not input.
;    IMPACT_REGION_PREPROCESSED:  If set, then ipccar6dna_conf_load_data.pro 
;        loads already region-masked and -averaged data.  See the 
;        REGION_PREPROCESSED keyword option for ipccar6dna_conf_load_data.pro.
;    IMPACT_SEASON:  A required 2-element integer vector specifying the season 
;        to analyses in month_to_seasons.pro [SEASON_ID,SEASON_LEN] format.  
;        For example [5,12] analyses January-December annual means.
;    IMPACT_VAR_LABEL:  A required scalar string specifying the climate 
;        variable for the requested data.  Supported values are:  'pr', 'tas', 
;        'tos'.
;    IMPACT_VAR_SIGN:  An optional N_REGION integer vector of -1 or +1 values, 
;        for input as IMPACT_SIGN to ipccar6dna_conf_sign.pro if the direction 
;        of observed matters for certain documented impacts.
;    INDEP_OPTIONS:  An optional string vector specifying various criteria for 
;        selection of independent data sources.  For example, 
;        ['MIP_label=CMIP5','source_label=CanESM2'] would limit analysis to 
;        data from the CanESM2 from the CMIP5 archive, following the 
;        INDEP_IN_SCENARIO scenarios.  See ipccar6dna_conf_define_source.pro 
;        for more details.
;    INDEP_IN_SCENARIO:  An optional string vector listing the scenarios to 
;        load for the independent data sets (signals) in the regression.  The 
;        default is ['hist-all','hist-nat'], i.e. "all forcing" historical 
;        simulations and "natural forcing" historical simulations.
;    INDEP_OUT_SCENARIO:  An option string vector listing the scenarios to 
;        output from the regression.  These can be different than 
;        INDEP_IN_SCENARIO, but need to be expressable as a linear combination 
;        of the INDEP_IN_SCENARIO scenarios.  The default is 
;        ['hist-ant','hist-nat'], i.e. anthropogenic-only and natural-only 
;        forcing historical data.
;    NOISE_OPTIONS:  An optional string vector specifying various criteria for 
;        selection of noise data sources.  For example, 
;        ['MIP_label=CMIP5','source_label=CanESM2,GISS-E2-H-p1,GISS-E2-R-p1'] 
;        would limit noise estimationt to data from the CanESM2, GISS-E2-H-p1, 
;        and GISS-E2-R-p1 climate models from the CMIP5 archive, following the 
;        NOISE_SCENARIO scenario.  See ipccar6dna_conf_define_source.pro for 
;        more details.
;    NOISE_SCENARIO:  An optional string specifying the type of data products 
;        to load for the noise scenario for the regression, as specified in the 
;        <*_file> entries in ipccar6dna_map_source.xml.  The default is 'noise'.
;    OPTIONS:  An optional string vector specifying a list of options for 
;        various aspects of the analysis.  Only one of OPTIONS and VERSION can 
;        be input.  Supported values are:
;        * 'density from <source>':  Evaluation of station density in the 
;          observational data products should come only from source <source>.  
;          For example 'density from CRU-TS3-22' means that only CRU-TS3-22 
;          will be used for evaluating station density.
;        * 'no weight density':  Do not weight observational products according 
;          do their station density and coverage within the region.
;        * 'no weight size':  Do not weight data sets according to region size.
;        * 'regcoef=binomial', 'regcoef=distribution', 'regcoef=k-s', 
;          'regcoef=weighted binomial':  Specifies the method for analysis of 
;          the regression coefficients from the multiple linear regression.  
;          See ipccar6dna_conf_regcoef.pro for details.
;        * 'resid=binomial', 'resid=k-s', 'resid=weighted binomial':  Specifies 
;          the method for analysis of the residuals from the multiple linear 
;          regression.  See ipccar6dna_conf_resid.pro for details.
;        * 'step size depends on indep':  Consider climate model resolution in 
;          relation to the region size when considering the effective number 
;          if climate models, but do not do any weighting for other steps.
;        * 'weight dep density':  Weight observational data sets according to 
;          their station density and coverage within the region.  Can be in 
;          combination with 'weight dep resolution'.
;        * 'weight dep resolution':  Weight observational data sets according 
;          to their grid resolution in relation to the region size.  Can be in 
;          combination with 'weight dep density'.
;        * 'weight indep resolution':  Weight climate models according to their 
;          grid resolution in relation to the region size.
;    VERBOSE:  If set then real-time status and timing notifications are 
;        printed to standard output.
;    VERSION:  An option string scalar generating a standard OPTIONS list.  
;        Supported values are:
;        * 'Stone and Hansen (2016)':  The options used in the original Stone 
;          and Hansen (2016) implementation.
;        * 'IPCC AR6 WGII':  The options used for a first draft of assessments 
;          for the IPCC AR6 WGII report.
;        * 'IPCC AR6 WGII':  The options used for the second (final) version 
;          of assessments for the IPCC AR6 WGII report.
;        Only one of OPTIONS and VERSION can be input.
;
; OUTPUT:
;    IMPACT_AREA, IMPACT_CONF_ATTRIB_MAJOR, IMPACT_CONF_DETECT, 
;      IMPACT_CONF_FACTOR, IMPACT_CONF_LABEL, IMPACT_REGION_LAT, 
;      IMPACT_REGION_LON, IMPACT_REGION_MASK
;
; USES:
;    $IPCCAR6DNA_DATA
;    $IPCCAR6DNA_IDL
;    gendetec.pro
;    ipccar6dna_conf_define_source.pro
;    ipccar6dna_conf_density.pro
;    ipccar6dna_conf_load_data.pro
;    ipccar6dna_conf_physics.pro
;    ipccar6dna_conf_regcoef.pro
;    ipccar6dna_conf_resid.pro
;    ipccar6dna_conf_sign.pro
;    ipccar6dna_conf_size.pro
;    ipccar6dna_region_mask.pro
;    isin.pro
;    netcdf_read.pro
;    ps_close.pro
;    ps_open.pro
;    shuffle.pro
;    str.pro
;
; PROCEDURE:
;    This procedure performs the detection and attribution assessments of the 
;    role of anthropogenic influence in observed climate variations following 
;    the steps described in Stone and Hansen (2016, 10.1007/s00382-015-2909-2) 
;    and Hansen and Stone (2016, 10.1038/nclimate2896).
;
; EXAMPLE:
;    ; Before starting IDL, define the directory containing the 
;    ; "IPCC AR6 WGII DA" code and the directory containing the data.
;    export IPCCAR6DNA_IDL="${HOME}/idl/papers/IPCC-AR6/"
;    export IPCCAR6DNA_DATA="${HOME}/data/"
;    ; In IDL, specify that we want to assess the confidence for annual mean 
;    ; temperature over the WRAF5-v4.1 "southern Canada" region, using the 
;    ; specified data sets and the 'IPCC AR6 WGII v2' approach.
;    dep_options = [ 'var_label=tas', 'source_label=GISTEMPv4' ]
;    dep_scenario = 'observed'
;    indep_options = [ 'MIP_label=CMIP5', 'source_label=CanESM2' ]
;    indep_in_scenario = [ 'hist-all', 'hist-nat' ]
;    indep_out_scenario = [ 'hist-ant', 'hist-nat' ]
;    noise_options = [ 'MIP_label=CMIP5', $
;        'source_label=CanESM2,GISS-E2-H-p1,GISS-E2-R-p1']
;    noise_scenario = [ 'noise' ]
;    ipccar6dna_conf, dep_options=dep_options, dep_scenario=dep_scenario, $
;        indep_options=indep_options, indep_in_scenario=indep_in_scenario, $
;        indep_out_scenario=indep_out_scenario, noise_options=noise_options, $
;        noise_scenario=noise_scenario, impact_domain='atmos-land', $
;        impact_period=[1951,2010], impact_region_def='WRAF5-v4-1=1.2', $
;        impact_season=[5,12], impact_var_label='tas', diagnostic='print', $
;        impact_area=impact_area, impact_conf_detect=impact_conf_detect_label, $
;        impact_conf_attrib_major=impact_conf_attrib_major_label, $
;        version='IPCC AR6 WGII v2'
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2015-03-23 (As 
;        hanseng_conf_assess_up.pro)
;    Modified:  DAS, 2020-10-29 (Adapted from hanseng_conf_assess_up.pro)
;    Modified:  DAS, 2020-11-20 (Added OPTIONS keyword input;  Added 
;        'size step depends on indep' and 'weight indep resolution' options to 
;        OPTIONS)
;    Modified:  DAS, 2020-12-07 (Added capability to handle pre-processed 
;        IPCCAR6-Andes data)
;    Modified:  DAS, 2021-01-18 (Added 'weight dep density',  
;        'no weight density', 'weight dep resolution' options to OPTIONS;  
;        Added capability to weight according to station density;  Switched to 
;        rounding up number of time segments within IMPACT_PERIOD;  Added 
;        VERBOSE keyword option;  Added DENSE_N_LONLAT_MIN keyword input; 
;        Added possibility of different single dependent source options for 
;        station density calculation;  Split record of NSOURCE into NOBS and 
;        NMODEL;  Fixed area calculation when IMPACT_REGION_PREPROCESSED=1)
;    Modified:  DAS, 2021-01-20 (Fixed situation where confidence is greater 
;        than very high)
;    Modified:  DAS, 2021-02-02 (Added GUESS_FILE_RESTRICT keyword option)
;    Modified:  DAS, 2021-02-10 (Added the VERSION='IPCC AR6 WGII v2' setting)
;    Modified:  DAS, 2021-03-11 (Fixed value of max_degrade=0.4 to 
;        max_degrade=0.1 in call to ipccar6dna_conf_regcoef.pro for 
;        impact_conf_nat)
;    Modified:  DAS, 2021-04-06 (Switched from setting independent scenario 
;        values to NaN if any simulation has NaN to only doing so if less than 
;        two simulations are non-NaN)
;    Modified:  DAS, 2021-12-20 (Completed documentation;  Changed DIAG_PLOT 
;        keyword input to DIAGNOSTIC keyword input)
;-

;***********************************************************************

PRO IPCCAR6DNA_CONF, $
    DEP_OPTIONS=dep_options, DEP_SCENARIO=dep_scenario, $
    DENSE_N_LONLAT_MIN=dense_n_lonlat_min, $
    DIAGNOSTIC=diagnostic, $
    IMPACT_DOMAIN=impact_domain, IMPACT_PERIOD=impact_period, $
      IMPACT_REGION_DEF=impact_region_def, IMPACT_SEASON=impact_season, $
      IMPACT_VAR_LABEL=impact_var_label, IMPACT_VAR_SIGN=impact_var_sign, $
    IMPACT_REGION_MASK=impact_region_mask, $
      IMPACT_REGION_LON=impact_region_lon, $
      IMPACT_REGION_LAT=impact_region_lat, $
    IMPACT_REGION_PREPROCESSED=impact_region_preprocessed_opt, $
    INDEP_OPTIONS=indep_options, INDEP_IN_SCENARIO=indep_in_scenario, $
      INDEP_OUT_SCENARIO=indep_out_scenario, $
    NOISE_OPTIONS=noise_options, NOISE_SCENARIO=noise_scenario, $
    OPTIONS=options, VERSION=version, $
    GUESS_FILE_RESTRICT=guess_file_restrict_opt, $
    VERBOSE=verbose_opt, $
    IMPACT_AREA=impact_area, $
    IMPACT_CONF_DETECT=impact_conf_detect_label, $
      IMPACT_CONF_ATTRIB_MAJOR=impact_conf_attrib_major_label, $
    IMPACT_CONF_FACTOR=impact_conf_factor, $
      IMPACT_CONF_LABEL=impact_conf_label

;***********************************************************************
; Options and constants

; Option to print status updates
verbose_opt = keyword_set( verbose_opt )
if verbose_opt eq 1 then begin
  verbose_time_0 = systime( seconds=1 )
  print, 'ipccar6dna_conf:  Starting options and constants '
  verbose_time_old = verbose_time_0
endif

; The random number seed
seed = 1

; Set standard p-value
p_limit = 0.1

; Earth's radius and area (in km and km2)
r_earth = 6370.95
a_earth = 4. * !pi * r_earth ^ 2.

; The default analysis option list (none)
if keyword_set( version ) then begin
  if n_elements( options ) ne 0 then stop
  ; For the Stone and Hansen (2016) approach
  if version eq 'Stone and Hansen (2016)' then begin
    options = [ 'no weight size', 'resid=binomial', 'regcoef=binomial', $
        'no weight density', 'density from CRU-TS3-22' ]
  ; The IPCC AR6 WGII approach
  endif else if version eq 'IPCC AR6 WGII' then begin
    options = [ 'weight indep resolution', 'resid=k-s', 'regcoef=k-s', $
        'weight dep density', 'weight dep resolution' ]
  ; The IPCC AR6 WGII v2 approach
  endif else if version eq 'IPCC AR6 WGII v2' then begin
    options = [ 'weight indep resolution', 'resid=weighted binomial', $
        'regcoef=weighted binomial', 'weight dep density', $
        'weight dep resolution' ]
  ; Otherwise the approach is not supported
  endif else begin
    stop
  endelse
endif
if not( keyword_set( options ) ) then options = ''

; Determine the number of regions requested
n_region = n_elements( impact_region_def )
if n_region eq 0 then begin
  if not( keyword_set( impact_region_mask ) ) then stop
  if not( keyword_set( impact_region_lon ) ) then stop
  if not( keyword_set( impact_region_lat ) ) then stop
  n_region = n_elements( impact_region_mask[0,0,*] )
endif else begin
  if keyword_set( impact_region_mask ) then begin
    if n_elements( impact_region_mask[0,0,*] ) ne n_region then stop
    impact_region_mask_opt = 1
  endif
endelse
; Ensure only a single impact is requested
; (perhaps for multiple regions, but all regions need to be of the same type, 
; e.g. all land-based regions)
if n_elements( impact_period ) ne 2 then stop
if n_elements( impact_season ) ne 2 then stop
if n_elements( impact_var_label ) ne 1 then stop
if max( n_elements( impact_var_sign ) eq [ 0, n_region ] ) eq 0 then stop
if not( keyword_set( impact_domain ) ) then impact_domain = 'atmos'

; Set mask options
if impact_domain eq 'atmos' then begin
  mask_options = 'no mask'
endif else if impact_domain eq 'atmos-land' then begin
  mask_options = 'land'
endif else if max( impact_domain eq [ 'ocean', 'atmos-ocean' ] ) eq 1 then begin
  mask_options = 'ocean'
endif else begin
  stop
endelse

; The default dependent scenario for input
if not( keyword_set( dep_scenario ) ) then dep_scenario = [ 'observed' ]
n_dep_scenario = n_elements( dep_scenario )

; The default independent scenarios for input
if not( keyword_set( indep_in_scenario ) ) then begin
  indep_in_scenario = [ 'hist-all', 'hist-nat' ]
  ;indep_in_scenario = [ 'hist-all', 'hist-ghg', 'hist-nat' ]
endif
n_indep_in_scenario = n_elements( indep_in_scenario )

; The default independent scenarios for output
if not( keyword_set( indep_out_scenario ) ) then begin
  indep_out_scenario = [ 'hist-ant', 'hist-nat' ]
  ;indep_out_scenario = [ 'hist-ghg', 'hist-antnoghg', 'hist-nat' ]
endif
n_indep_out_scenario = n_elements( indep_out_scenario )
if n_indep_out_scenario gt n_indep_in_scenario then stop
; Determine arithmetic for converting input to output independent scenarios
temp_in = strjoin( indep_in_scenario, ';' )
temp_out = strjoin( indep_out_scenario, ';' )
if ( temp_in eq 'hist-all;hist-nat' ) and ( temp_out eq 'hist-ant;hist-nat' ) $
    then begin
  indep_out_algebra = [ '0', '0+1' ]
  indep_out_onedim_map = [ 1, -1 ]
endif else if ( temp_in eq 'hist-all;hist-ghg;hist-nat' ) $
    and ( temp_out eq 'hist-ghg;hist-antnoghg;hist-nat' ) then begin
  indep_out_algebra = [ '0+1', '0', '0+2' ]
  indep_out_onedim_map = [ 0, 1, 0 ]
endif else begin
  ; Not yet supported
  stop
endelse
; Determine the names of the output independent scenarios
indep_out_scenario_name = strarr( n_indep_out_scenario )
id = where( indep_out_scenario eq 'hist-ant', n_id )
if n_id gt 0 then indep_out_scenario_name[id] = 'Anthropogenic'
id = where( indep_out_scenario eq 'hist-antnoghg', n_id )
if n_id gt 0 then indep_out_scenario_name[id] = 'Anthropogenic except GHG'
id = where( indep_out_scenario eq 'hist-ghg', n_id )
if n_id gt 0 then indep_out_scenario_name[id] = 'Greenhouse gas'
id = where( indep_out_scenario eq 'hist-nat', n_id )
if n_id gt 0 then indep_out_scenario_name[id] = 'Natural'
if max( indep_out_scenario_name eq '' ) eq 1 then stop

; The noise scenarios
if not( keyword_set( noise_scenario ) ) then noise_scenario = [ 'noise' ]
n_noise_scenario = n_elements( noise_scenario )

; The default regression type
regress_type = 'OLS'
; Default no optimisation setting for gendetec.pro
no_optimise_opt = 1
; Set the quantiles to sample for the attributable response distributions
n_regress_dist_prob = 100
n_regress_point = 2500l
; Set the threshold for fraction of trend to qualify for major role
frac_thresh_major = 1. / 3.

; Get list of dependent variable (observed) sources
ipccar6dna_conf_define_source, select_values=dep_options, $
    scenario=dep_scenario, var_label=impact_var_label, $
    mask_file=dep_mask_file, mask_var_label=dep_mask_var_label, $
    resolution=dep_resolution, source_label=dep_source_label, $
    var_file=dep_var_file
n_dep_source = n_elements( dep_var_file )

; Get list of independent variable (climate model) sources
ipccar6dna_conf_define_source, select_values=indep_options, $
    scenario=indep_in_scenario, var_label=impact_var_label, $
    mask_file=indep_mask_file, mask_var_label=indep_mask_var_label, $
    resolution=indep_resolution, var_file=indep_var_file
n_indep_source = n_elements( indep_var_file[0,*] )

; Get list of noise variable (climate model) sources
ipccar6dna_conf_define_source, select_values=noise_options, $
    scenario=noise_scenario, var_label=impact_var_label, $
    mask_file=noise_mask_file, mask_var_label=noise_mask_var_label, $
    resolution=noise_resolution, var_file=noise_var_file
n_noise_source = n_elements( noise_var_file[0,*] )

; Define the anomaly method
if max( impact_var_label eq [ 'tas', 'tos', 'ts' ] ) eq 1 then begin
  impact_anomaly_method = 'absolute'
endif else if impact_var_label eq 'pr' then begin
  impact_anomaly_method = 'fractional'
endif else begin
  stop
endelse

; The number of time values
n_time_year = impact_period[1] - impact_period[0] + 1
; The size of the time segments in years
time_seg_len = 5
; The number of time segments (rounding up)
n_time_seg = ceil( float( n_time_year ) / float( time_seg_len ) )
; The fraction of segments requiring data to be considered a usable data set
n_time_thresh_good = ceil( 0.75 * n_time_seg )
;n_time_thresh_good = n_time_seg
; Redefine impact_period to match the segments
;impact_period_use = [ impact_period[0] + ( n_time_year mod time_seg_len ), $
;    impact_period[1] ]
impact_period_use = [ impact_period[1] - n_time_seg * time_seg_len + 1, $
    impact_period[1] ]

; Define the confidence labels
conf_label = [ 'none', 'very low', 'low', 'medium', 'high', 'very high' ]

; Default no diagnostic plots
if n_elements( diagnostic ) eq 0 then diagnostic = ''

; The option to load already region-masked and -averaged data
impact_region_preprocessed_opt = keyword_set( impact_region_preprocessed_opt )
; Remove mask details if unneeded
if impact_region_preprocessed_opt eq 1 then begin
  dep_mask_file = 0
  dep_mask_var_label = 0
  indep_mask_file = 0
  indep_mask_var_label = 0
  noise_mask_file = 0
  noise_mask_var_label = 0
endif

; The default option on guess files based on time period information in file 
; names
if n_elements( guess_file_restrict_opt ) gt 0 then begin
  guess_file_restrict_opt = keyword_set( guess_file_restrict_opt )
endif else begin
  guess_file_restrict_opt = 1
endelse

;***********************************************************************
; Load data

; Determine the region area.
if verbose_opt eq 1 then begin
  verbose_time_new = systime( seconds=1 )
  print, 'ipccar6dna_conf:  Determining region area ', $
      verbose_time_new - verbose_time_old, verbose_time_new - verbose_time_0
  verbose_time_old = verbose_time_new
endif
if keyword_set( impact_region_mask ) then begin
  impact_area = impact_region_mask
  temp_lat = impact_region_lat
  for i_region = 0, n_region - 1 do begin
    temp_weight = impact_area
    for i_lat = 0, n_elements( temp_lat ) - 1 do begin
      temp_weight[*,i_lat] = cos( !pi * temp_lat[i_lat] / 180. )
    endfor
    temp_weight = temp_weight / total( temp_weight ) * 4. * !pi $
        * ( 6370.95 ^ 2. )
    impact_area[0,0,i_region] = total( impact_area[*,*,i_region] * temp_weight )
    temp_weight = 0
  endfor
  impact_area = reform( impact_area[0,0,*] )
endif else if keyword_set( impact_region_preprocessed_opt ) eq 1 then begin
  impact_area = fltarr( n_region )
  for i_region = 0, n_region - 1 do begin
    temp_region = strsplit( impact_region_def[i_region], '=', extract=1, $
        count=n_temp )
    if n_temp ne 2 then stop
    if strpos( temp_region[0], 'WRAF' ) eq 0 then begin
      spawn, 'echo ${IPCCAR6DNA_DATA}', temp_file
      temp_file = temp_file[0] + 'WRAF/v4-1/fx/atmos-' + temp_region[0] $
          + '/region/4-1-0/region_fx-' + temp_region[0] $
          + '_WRAF_All-Hist_est1_v4-1_4-1-0_000000-000000.nc'
    endif else if strpos( temp_region[0], 'IPCCAR6-Andes' ) eq 0 then begin
      spawn, 'echo ${IPCCAR6DNA_DATA}', temp_file
      temp_file = temp_file[0] + 'IPCC-AR6-WGII/Andes/fx/atmos/region/' $
          + impact_var_label + '/region-masks_IPCC-AR6-Andes_' $
          + impact_var_label + '.nc'
    endif else begin
      stop
    endelse
    temp_regionid = string( netcdf_read( temp_file, 'regionid' ) )
    temp_impact_area = string( netcdf_read( temp_file, 'area' ) )
    id = where( temp_regionid eq temp_region[1], n_id )
    if n_id ne 1 then stop
    impact_area[i_region] = temp_impact_area[id[0]]
  endfor
endif else begin
  impact_area = fltarr( n_region )
  for i_region = 0, n_region - 1 do begin
    impact_area[i_region] = ipccar6dna_region_mask( $
        region=impact_region_def[i_region] )
  endfor
endelse

; Initialise dependent data array
if verbose_opt eq 1 then begin
  verbose_time_new = systime( seconds=1 )
  print, 'ipccar6dna_conf:  Loading dependent data', $
      verbose_time_new - verbose_time_old, verbose_time_new - verbose_time_0
  verbose_time_old = verbose_time_new
endif
dep_var_data = fltarr( n_time_seg, n_dep_source, n_region )
; Iterate through dependent data sources
for i_source = 0, n_dep_source - 1 do begin
  ; Determine file list
  temp_file = strtrim( strsplit( dep_var_file[i_source], '&', extract=1, $
      count=n_temp_file ), 2 )
  ;if verbose_opt eq 1 then print, '...', i_source, temp_file[0]
  ; Load data for this source
  if keyword_set( dep_mask_var_label ) then begin
    temp_mask_var_label = dep_mask_var_label[i_source]
    temp_mask_file = dep_mask_file[i_source]
  endif else begin
    temp_mask_var_label = ''
    temp_mask_file = ''
  endelse
  temp_data = ipccar6dna_conf_load_data( temp_file, $
      anomaly=impact_anomaly_method, mask_file=temp_mask_file, $
      mask_options=mask_options, mask_var_label=temp_mask_var_label, $
      period=impact_period_use, time_seg_len=time_seg_len, $
      region_def=impact_region_def, season_id=impact_season[0], $
      season_len=impact_season[1], var_label=impact_var_label, $
      region_mask=impact_region_mask, region_lon=impact_region_lon, $
      region_lat=impact_region_lat, $
      region_preprocessed=impact_region_preprocessed_opt )
  temp_data = reform( temp_data )
  ; Take ensemble average
  if n_temp_file gt 1 then temp_data = mean( temp_data, dimension=2 )
  ; Record data
  dep_var_data[*,i_source,*] = temp_data
endfor

; Initialise independent data array
if verbose_opt eq 1 then begin
  verbose_time_new = systime( seconds=1 )
  print, 'ipccar6dna_conf:  Loading independent data', $
      verbose_time_new - verbose_time_old, verbose_time_new - verbose_time_0
  verbose_time_old = verbose_time_new
endif
indep_in_var_data = fltarr( n_time_seg, n_indep_in_scenario, n_indep_source, $
    n_region )
; Iterate through independent data sources
for i_source = 0, n_indep_source - 1 do begin
  ; Iterate through scenarios
  for i_scenario = 0, n_indep_in_scenario - 1 do begin
    ; Determine file list
    temp_file = strtrim( strsplit( indep_var_file[i_scenario,i_source], '&', $
        extract=1, count=n_temp_file ), 2 )
    ;if verbose_opt eq 1 then print, '...', i_source, i_scenario, temp_file[0]
    ; Load data for this source and scenario
    if keyword_set( indep_mask_var_label ) then begin
      temp_mask_var_label = indep_mask_var_label[i_source]
      temp_mask_file = indep_mask_file[i_source]
    endif else begin
      temp_mask_var_label = ''
      tmep_mask_file = ''
    endelse
    temp_data = ipccar6dna_conf_load_data( temp_file, $
        anomaly=impact_anomaly_method, mask_file=temp_mask_file, $
        mask_options=mask_options, mask_var_label=temp_mask_var_label, $
        period=impact_period_use, time_seg_len=time_seg_len, $
        region_def=impact_region_def, season_id=impact_season[0], $
        season_len=impact_season[1], var_label=impact_var_label, $
        region_mask=impact_region_mask, region_lon=impact_region_lon, $
        region_lat=impact_region_lat, $
        region_preprocessed=impact_region_preprocessed_opt, $
        guess_file_restrict=guess_file_restrict_opt )
    temp_data = reform( temp_data )
    ; Take ensemble average (require at least two active simulations)
    temp_finite = finite( temp_data )
    if n_temp_file gt 1 then temp_finite = total( temp_finite, 2, integer=1 )
    id = where( temp_finite lt 2, n_id )
    if n_temp_file gt 1 then temp_data = mean( temp_data, dimension=2, nan=1 )
    if n_id gt 0 then temp_finite[id] = !values.f_nan
    ; Record data
    indep_in_var_data[*,i_scenario,i_source,*] = reform( temp_data, $
        n_time_seg, 1, 1, n_region )
  endfor
endfor

; Initialise noise data array
if verbose_opt eq 1 then begin
  verbose_time_new = systime( seconds=1 )
  print, 'ipccar6dna_conf:  Loading noise data', $
      verbose_time_new - verbose_time_old, verbose_time_new - verbose_time_0
  verbose_time_old = verbose_time_new
endif
n_noise_set = 0
; Iterate through noise data sources
for i_source = 0, n_noise_source - 1 do begin
  ; Iterate through scenarios
  for i_scenario = 0, n_noise_scenario - 1 do begin
    ; Determine file list
    temp_file = strtrim( strsplit( noise_var_file[i_scenario,i_source], '&', $
        extract=1, count=n_temp_file ), 2 )
    ;if verbose_opt eq 1 then print, '...', i_source, i_scenario, temp_file[0]
    ; Iterate through sets
    for i_set = 0, n_temp_file - 1 do begin
      ; Load data for this source and scenario
      if keyword_set( noise_mask_var_label ) then begin
        temp_mask_var_label = noise_mask_var_label[i_source]
        temp_mask_file = noise_mask_file[i_source]
      endif else begin
        temp_mask_var_label = ''
        temp_mask_file = ''
      endelse
      temp_data = ipccar6dna_conf_load_data( temp_file[i_set], $
          anomaly=impact_anomaly_method, mask_file=temp_mask_file, $
          mask_options=mask_options, mask_var_label=temp_mask_var_label, $
          time_seg_len=time_seg_len, region_def=impact_region_def, $
          season_id=impact_season[0], season_len=impact_season[1], $
          var_label=impact_var_label, region_mask=impact_region_mask, $
          region_lon=impact_region_lon, region_lat=impact_region_lat, $
          pretend_period=impact_period_use, $
          region_preprocessed=impact_region_preprocessed_opt, $
          guess_file_restrict=guess_file_restrict_opt )
      temp_data = reform( temp_data )
      n_temp_set = n_elements( temp_data[0,*] )
      ; Record data
      if n_noise_set eq 0 then begin
        noise_var_data = temp_data
      endif else begin
        noise_var_data = [ [ noise_var_data ], [ temp_data ] ]
      endelse
      n_noise_set = n_noise_set + n_temp_set
    endfor
  endfor
endfor

;***********************************************************************
; Perform optimal detection analysis

if verbose_opt eq 1 then begin
  verbose_time_new = systime( seconds=1 )
  print, 'ipccar6dna_conf:  Performing regression', $
      verbose_time_new - verbose_time_old, verbose_time_new - verbose_time_0
  verbose_time_old = verbose_time_new
endif

; Initialise output arrays
regress_p_resid = !values.f_nan $
    * fltarr( n_dep_source, n_indep_source, n_region )
regress_beta_est = !values.f_nan $
    * fltarr( 3, n_indep_out_scenario, n_dep_source, n_indep_source, n_region )
regress_frac_dist_trend_major = !values.f_nan $
    * fltarr( n_dep_source, n_indep_source, n_region )
regress_frac_dist_beta_positive = !values.f_nan $
    * fltarr( n_indep_out_scenario, n_dep_source, n_indep_source, n_region )
regress_frac_dist_beta_one = !values.f_nan $
    * fltarr( n_indep_out_scenario, n_dep_source, n_indep_source, n_region )

; Iterate through regions
for i_region = 0, n_region - 1 do begin
  ; Iterate through dependent data sources
  for i_dep = 0, n_dep_source - 1 do begin
    ; Extract data for dependent variable (including consistency check)
    temp_dep_var_data = dep_var_data[*,i_dep,i_region]
    ; Proceed only if there is enough data
    if total( finite( temp_dep_var_data ), integer=1 ) gt n_time_thresh_good $
        then begin
      ; Determine mask for dependent data
      temp_dep_mask = finite( temp_dep_var_data )
      ; Iterate through independent data sources
      for i_indep = 0, n_indep_source - 1 do begin
        ; Initialise independent variable array
        temp_indep_var_data = indep_in_var_data[*,*,i_indep,i_region]
        ; Only proceed if we have scenario data
        temp = total( finite( temp_indep_var_data ), 1, integer=1 )
        if min( temp ) gt 1 then begin
          ; Determine mask for independent data
          temp_indep_mask = min( finite( temp_indep_var_data ), dimension=2 )
          ; Extract noise data
          temp_noise_var_data = noise_var_data[*,*,i_region]
          ; Apply masks
          id_seg = where( temp_dep_mask + temp_indep_mask eq 2, n_id_seg )
          temp_dep_var_data_use = temp_dep_var_data[id_seg]
          temp_indep_var_data = temp_indep_var_data[id_seg,*]
          temp_noise_var_data = temp_noise_var_data[id_seg,*]
          ; Stop if we have insufficient data
          if n_id_seg lt 2 * n_indep_out_scenario then stop
          ; Shuffle noise data (okay because we are not doing overlapping noise 
          ; segments)
          if no_optimise_opt eq 1 then begin
            temp_noise_var_data_1 = temp_noise_var_data
            n_id_noise_1 = n_noise_set
            temp_noise_var_data_2 = 0
            n_id_noise_2 = 0
          endif else begin
            id_noise = shuffle( indgen( n_noise_set ), seed=seed )
            n_id_noise_1 = n_noise_set / 2
            id_noise_1 = id_noise[0:n_id_noise_1-1]
            temp_noise_var_data_1 = temp_noise_var_data[*,id_noise_1]
            n_id_noise_2 = n_noise_set - n_id_noise_1
            id_noise_2 = id_noise[n_id_noise_1:n_noise_set-1]
            temp_noise_var_data_2 = temp_noise_var_data[*,id_noise_2]
          endelse
          ; Define truncation
          trunc = min( [ n_id_seg - 1, n_id_noise_1 ] ) - 1
          ; Request attributable response trend output
          temp_dist_prob = n_regress_dist_prob
          temp_beta_dist = 1
          ; Perform regression analysis
          gendetec, temp_dep_var_data_use, temp_indep_var_data, $
              temp_noise_var_data_1, temp_beta_est, p_limit=p_limit, $
              data_noise_2=temp_var_noise_data_2, type=regress_type, $
              n_point=n_regress_point, p_resid=temp_p_resid, $
              no_optimise=no_optimise_opt, double=1, $
              transform=indep_out_algebra, trunc=trunc, $
              beta_dist=temp_beta_dist, dist_prob=temp_dist_prob, $
              dist_weight=temp_dist_weight
          ; Record regression coefficient distribution and residual p-value
          regress_beta_est[*,*,i_dep,i_indep,i_region] = transpose( $
              temp_beta_est )
          regress_p_resid[i_dep,i_indep,i_region] = temp_p_resid
          ; Record fraction of beta distribution greater than zero and greater 
          ; than one
          temp_beta_dist = reform( temp_beta_dist, n_indep_out_scenario, $
              1l * n_regress_dist_prob * n_regress_point )
          temp_dist_weight = reform( temp_dist_weight, $
                  1l * n_regress_dist_prob * n_regress_point )
          for i_indep_out = 0, n_indep_out_scenario - 1 do begin
            ; Calculate fraction greater than zero
            id = where( temp_beta_dist[i_indep_out,*] gt 0, n_id )
            if n_id gt 0 then begin
              regress_frac_dist_beta_positive[i_indep_out,i_dep,i_indep,i_region] $
                  = total( temp_dist_weight[id] )
            endif else begin
              regress_frac_dist_beta_positive[i_indep_out,i_dep,i_indep,i_region] = 0.
            endelse
            ; Calculate fraction greater than one
            id = where( temp_beta_dist[i_indep_out,*] gt 1, n_id )
            if n_id gt 0 then begin
              regress_frac_dist_beta_one[i_indep_out,i_dep,i_indep,i_region] $
                  = total( temp_dist_weight[id] )
            endif else begin
              regress_frac_dist_beta_one[i_indep_out,i_dep,i_indep,i_region] = 0.
            endelse
          endfor
          temp_beta_dist = reform( temp_beta_dist, n_indep_out_scenario, $
              n_regress_dist_prob, n_regress_point )
          temp_dist_weight = reform( temp_dist_weight, n_regress_dist_prob, $
              n_regress_point )
          ; Record fraction of attributable response distribution that is 
          ; beyond frac_thresh_major times the observed trend/noise ratio
          id_indep_out = where( indep_out_scenario eq 'hist-ant', $
              n_id_indep_out )
          if n_id_indep_out eq 0 then begin
            id_indep_out = where( indep_out_scenario eq 'hist-ghg', $
                n_id_indep_out )
          endif
          if n_id_indep_out ne 1 then stop
          temp_beta_dist = reform( temp_beta_dist[id_indep_out[0],*,*] )
          temp_var_ratio = 0 * temp_beta_dist
          temp_dep = temp_dep_var_data_use - mean( temp_dep_var_data_use )
          temp_indep_out_var_data = indep_out_onedim_map ## temp_indep_var_data
          temp_indep_out_var_data = temp_indep_out_var_data $
              - mean( temp_indep_out_var_data )
          for i_dist = 0l, 1l * n_regress_dist_prob * n_regress_point - 1l $
              do begin
            temp_indep = temp_beta_dist[i_dist] * temp_indep_out_var_data
            temp_var_ratio[i_dist] = 1. $
                - total( ( temp_dep - temp_indep ) ^ 2. ) $
                / total( temp_dep ^ 2 )
          endfor
          id_dist = where( temp_var_ratio gt frac_thresh_major, n_id_dist )
          if n_id_dist gt 0 then begin
            regress_frac_dist_trend_major[i_dep,i_indep,i_region] $
                = total( temp_dist_weight[id_dist] )
          endif else begin
            regress_frac_dist_trend_major[i_dep,i_indep,i_region] = 0.
          endelse
        endif
      endfor
    endif
  endfor
endfor

;***********************************************************************
; Perform confidence assessment

if verbose_opt eq 1 then begin
  verbose_time_new = systime( seconds=1 )
  print, 'ipccar6dna_conf:  Starting confidence assessment', $
      verbose_time_new - verbose_time_old, verbose_time_new - verbose_time_0
  verbose_time_old = verbose_time_new
endif

; Determine for which dependent-independent source combinations we have 
; sufficient data for calculations
id_source = where( finite( regress_p_resid ) eq 1, n_id_source )

; Modify confidence according to region size
if max( options eq 'weight indep resolution' ) eq 1 then begin
  indep_weight = fltarr( n_indep_source, n_region )
  temp_cell_area = a_earth $
        / reform( indep_resolution[0,*] * indep_resolution[1,*] )
endif else if max( options eq 'step size depends on indep' ) eq 1 then begin
  temp_cell_area = a_earth $
        / reform( indep_resolution[0,*] * indep_resolution[1,*] )
  impact_conf_size = fltarr( n_region )
endif else if max( options eq 'no weight size' ) eq 1 then begin
  impact_conf_size = fltarr( n_region )
endif else begin
  stop
endelse
for i_region = 0, n_region - 1 do begin
  temp_conf = ipccar6dna_conf_size( impact_area[i_region], $
      cell_area=temp_cell_area )
  if max( options eq 'weight indep resolution' ) eq 1 then begin
    indep_weight[*,i_region] = temp_conf
  endif else if max( options eq 'step size depends on indep' ) eq 1 then begin
    impact_conf_size[i_region] = mean( temp_conf )
  endif else if max( options eq 'no weight size' ) eq 1 then begin
    impact_conf_size[i_region] = temp_conf
  endif else begin
    stop
  endelse
endfor

; Modify confidence according to station coverage.
if verbose_opt eq 1 then begin
  verbose_time_new = systime( seconds=1 )
  print, 'ipccar6dna_conf:  Analysing station density', $
      verbose_time_new - verbose_time_old, verbose_time_new - verbose_time_0
  verbose_time_old = verbose_time_new
endif
temp_period = str( impact_period_use ) + [ '0101', '1231' ]
; Calculate separately for each source if appropriate
if max( options eq 'weight dep density' ) eq 1 then begin
  temp_dep_source_label = dep_source_label
; Calculate for one representative source only if appropriate
endif else if max( strpos( options, 'density from ' ) ) eq 0 then begin
  id = where( strpos( options, 'density from ' ) eq 0, n_id )
  if n_id ne 1 then stop
  pos = strlen( 'density from ' )
  temp_dep_source_label = strmid( options[id[0]], pos, $
      strlen( options[id[0]] ) - pos )
endif else begin
  stop
endelse
n_temp_dep_source = n_elements( temp_dep_source_label )
dep_conf_dense = fltarr( n_temp_dep_source, n_region )
for i_region = 0, n_region - 1 do begin
  for i_dep = 0, n_temp_dep_source - 1 do begin
    if keyword_set( impact_region_mask_opt ) then begin
      temp_region_mask = impact_region_mask[*,*,i_region]
    endif else begin
      temp_region_def = impact_region_def[i_region]
    endelse
    dep_conf_dense[i_dep,i_region] = ipccar6dna_conf_density( $
        domain_label=impact_domain, frac_active_thresh=0.9, $
        period=temp_period, source_label=temp_dep_source_label[i_dep], $
        region_def=temp_region_def, season_id=impact_season[0], $
        season_len=impact_season[1], seed=seed, var_label=impact_var_label, $
        region_mask=temp_region_mask, region_lon=impact_region_lon, $
        region_lat=impact_region_lat, n_lonlat_min=dense_n_lonlat_min )
  endfor
  ; Assume that undetermined values should be the average of determined values
  id_miss = where( finite( dep_conf_dense[*,i_region] ) eq 0, n_id_miss ) 
  if n_id_miss eq n_dep_source then stop
  if n_id_miss gt 0 then begin
    dep_conf_dense[id_miss,i_region] = mean( dep_conf_dense[*,i_region], nan=1 )
  endif
endfor
; Calculate separately for each source if appropriate
if max( options eq 'weight dep density' ) eq 1 then begin
  dep_weight = dep_conf_dense
; Calculate for one representative source only if appropriate
endif else if max( options eq 'density from CRU-TS3-22' ) eq 1 then begin
  impact_conf_dense = reform( dep_conf_dense[0,*] )
endif else begin
  stop
endelse
; Modify observation weighting according to grid resolution if appropriate
if max( options eq 'weight dep resolution' ) eq 1 then begin
  temp_cell_area = a_earth $
        / reform( dep_resolution[0,*] * dep_resolution[1,*] )
  for i_region = 0, n_region - 1 do begin
    temp_conf = ipccar6dna_conf_size( impact_area[i_region], $
        cell_area=temp_cell_area, diffusive_factor=1. )
    dep_weight[*,i_region] = dep_weight[*,i_region] * temp_conf
  endfor
endif

if verbose_opt eq 1 then begin
  verbose_time_new = systime( seconds=1 )
  print, 'ipccar6dna_conf:  Continuing confidence assessment', $
      verbose_time_new - verbose_time_old, verbose_time_new - verbose_time_0
  verbose_time_old = verbose_time_new
endif

; Create dependent-by-independent source weight array
if keyword_set( dep_weight ) and keyword_set( indep_weight ) then begin
  depindep_weight = add_dim( indep_weight, 0, n_dep_source ) $
      * add_dim( dep_weight, 1, n_indep_source )
endif else if keyword_set( indep_weight ) then begin
  depindep_weight = add_dim( indep_weight, 0, n_dep_source )
endif else if keyword_set( indep_weight ) then begin
  depindep_weight = add_dim( dep_weight, 1, n_indep_source )
endif

; Initialise on basis that doubling source numbers increases confidence by one.
; If we are weighting by the resolution of the dependent source
if max( options eq 'weight dep density' ) eq 1 then begin
  impact_conf_nsource_dep = sqrt( total( dep_weight ^ 2., 1 ) )
endif else if max( options eq 'no weight density' ) eq 1 then begin
  impact_conf_nsource_dep = sqrt( n_dep_source ) + fltarr( n_region )
endif else begin
  stop
endelse
; If we are weighting by the resolution of the independent source
if max( options eq 'weight indep resolution' ) eq 1 then begin
  impact_conf_nsource_indep = sqrt( total( indep_weight ^ 2., 1 ) )
endif else if max( options eq 'no weight size' ) eq 1 then begin
  impact_conf_nsource_indep = sqrt( n_indep_source ) + fltarr( n_region )
endif else begin
  stop
endelse

; Check whether observed trend matches requested sign.
if keyword_set( impact_var_sign ) then begin
  impact_conf_sign = fltarr( n_region )
  for i_region = 0, n_region - 1 do begin
    impact_conf_sign[i_region] = ipccar6dna_conf_sign( $
        dep_var_data[*,*,i_region], impact_sign=impact_var_sign[i_region], $
        n_time_thresh_good=n_time_thresh_good )
  endfor
endif

; Degrade confidence for physical understanding
impact_conf_physics = fltarr( n_region ) $
    + ipccar6dna_conf_physics( impact_var_label )

; Modify confidence according to the probability of the anthropogenic 
; regression coefficient being greater than zero
id_indep_out = where( indep_out_scenario eq 'hist-ant', n_id_indep_out )
if n_id_indep_out eq 0 then begin
  id_indep_out = where( indep_out_scenario eq 'hist-ghg', n_id_indep_out )
endif
if n_id_indep_out ne 1 then stop
impact_conf_signal = fltarr( n_region )
for i_region = 0, n_region - 1 do begin
  if n_elements( depindep_weight ) gt 0 then begin
    temp_weight = reform( depindep_weight[*,*,i_region], $
        n_dep_source * n_indep_source )
  endif else if ( max( options eq 'no weight size' ) eq 1 ) $
      and ( max( options eq 'no weight density' ) eq 1 ) then begin
    temp_weight = 0
  endif else begin
    stop
  endelse
  if ( n_elements( depindep_weight ) gt 0 ) and ( max( temp_weight ) eq 0 ) $
      then begin
    impact_conf_signal[i_region] = !values.f_nan
  endif else begin
    temp_frac = reform( $
        regress_frac_dist_beta_positive[id_indep_out,*,*,i_region], $
        n_id_indep_out, n_dep_source * n_indep_source )
    impact_conf_signal[i_region] = ipccar6dna_conf_regcoef( $
        'consistent with gt 0', frac_dist_beta_positive=temp_frac, $
        source_weight=temp_weight )
  endelse
endfor

; Modify confidence according to consistency of the regression coefficients for 
; the primary anthropogenic forcings with 1
id_indep_out = where( indep_out_scenario eq 'hist-ant', n_id_indep_out )
if n_id_indep_out eq 0 then begin
  id_indep_out = where( indep_out_scenario eq 'hist-ghg', n_id_indep_out )
endif
if n_id_indep_out ne 1 then stop
impact_conf_ant = fltarr( n_region )
for i_region = 0, n_region - 1 do begin
  temp_beta = reform( regress_beta_est[*,id_indep_out[0],*,*,i_region], 3, $
      n_id_indep_out, n_dep_source * n_indep_source )
  ; If the evaluation is performed via a K-S test
  if max( options eq 'regcoef=k-s' ) eq 1 then begin
    if n_elements( depindep_weight ) gt 0 then begin
      temp_weight = reform( depindep_weight[*,*,i_region], $
          n_dep_source * n_indep_source )
    endif else begin
      temp_weight = 0
    endelse
    if ( n_elements( depindep_weight ) gt 0 ) and ( max( temp_weight ) eq 0 ) $
        then begin
      impact_conf_ant[i_region] = !values.f_nan
    endif else begin
      impact_conf_ant[i_region] = ipccar6dna_conf_regcoef( $
          'consistent with 1', beta_est=temp_beta, max_degrade=0.4, $
          p_limit=p_limit, method='k-s', source_weight=temp_weight )
    endelse
  ; If the evaluation is performed by integration across distributions
  endif else if max( options eq 'regcoef=distribution' ) eq 1 then begin
    if n_elements( depindep_weight ) gt 0 then begin
      temp_weight = reform( depindep_weight[*,*,i_region], $
          n_dep_source * n_indep_source )
    endif else begin
      temp_weight = 0
    endelse
    if ( n_elements( depindep_weight ) gt 0 ) and ( max( temp_weight ) eq 0 ) $
        then begin
      impact_conf_ant[i_region] = !values.f_nan
    endif else begin
      temp_frac = reform( $
          regress_frac_dist_beta_one[id_indep_out,*,*,i_region], $
          n_id_indep_out, n_dep_source * n_indep_source )
      impact_conf_ant[i_region] = ipccar6dna_conf_regcoef( $
          'consistent with 1', frac_dist_beta_positive=temp_frac, $
          max_degrade=0.4, p_limit=p_limit, method='distribution', $
          source_weight=temp_weight )
    endelse
  ; If the evaluation is performed by a weighted extreme p-value count
  ; (IPCC AR6)
  endif else if max( options eq 'regcoef=weighted binomial' ) eq 1 then begin
    if n_elements( depindep_weight ) gt 0 then begin
      temp_weight = reform( depindep_weight[*,*,i_region], $
          n_dep_source * n_indep_source )
    endif else begin
      temp_weight = 0
    endelse
    if ( n_elements( depindep_weight ) gt 0 ) and ( max( temp_weight ) eq 0 ) $
        then begin
      impact_conf_ant[i_region] = !values.f_nan
    endif else begin
      impact_conf_ant[i_region] = ipccar6dna_conf_regcoef( $
          'consistent with 1', beta_est=temp_beta, max_degrade=0.4, $
          p_limit=p_limit, method='weighted binomial', $
          source_weight=temp_weight )
    endelse
  ; If the evaluation is performed via extreme p-value count 
  ; (Stone and Hansen 2016)
  endif else if max( options eq 'regcoef=binomial' ) eq 1 then begin
    if max( options eq 'weight indep resolution' ) eq 1 then stop
    if max( options eq 'weight dep density' ) eq 1 then stop
    impact_conf_ant[i_region] = ipccar6dna_conf_regcoef( 'consistent with 1', $
        beta_est=temp_beta, max_degrade=0.4, p_limit=p_limit, $
        method='binomial' )
  endif else begin
    stop
  endelse
endfor

; Modify confidence according to consistency of the regression coefficients for 
; the other forcings with 1
id_indep_out = where( indep_out_scenario ne 'hist-ant', n_id_indep_out )
if n_id_indep_out eq n_indep_out_scenario then begin
  id_indep_out = where( indep_out_scenario ne 'hist-ghg', n_id_indep_out )
endif
if n_id_indep_out ne n_indep_out_scenario - 1 then stop
impact_conf_nat = fltarr( n_region )
for i_region = 0, n_region - 1 do begin
  temp_beta = reform( regress_beta_est[*,id_indep_out,*,*,i_region], 3, $
      n_id_indep_out, n_dep_source * n_indep_source )
  ; If the evaluation is performed via a K-S test
  if max( options eq 'regcoef=k-s' ) eq 1 then begin
    if n_elements( depindep_weight ) gt 0 then begin
      temp_weight = reform( depindep_weight[*,*,i_region], $
          n_dep_source * n_indep_source )
    endif else begin
      temp_weight = 0
    endelse
    if ( n_elements( depindep_weight ) gt 0 ) and ( max( temp_weight ) eq 0 ) $
        then begin
      impact_conf_nat[i_region] = !values.f_nan
    endif else begin
      impact_conf_nat[i_region] = ipccar6dna_conf_regcoef( $
          'consistent with 1', beta_est=temp_beta, max_degrade=0.1, $
          p_limit=p_limit, method='k-s', source_weight=temp_weight )
    endelse
  ; If the evaluation is performed by integration across distributions
  endif else if max( options eq 'regcoef=distribution' ) eq 1 then begin
    if n_elements( depindep_weight ) gt 0 then begin
      temp_weight = reform( depindep_weight[*,*,i_region], $
          n_dep_source * n_indep_source )
    endif else begin
      temp_weight = 0
    endelse
    if ( n_elements( depindep_weight ) gt 0 ) and ( max( temp_weight ) eq 0 ) $
        then begin
      impact_conf_ant[i_region] = !values.f_nan
    endif else begin
      temp_frac = reform( $
          regress_frac_dist_beta_one[id_indep_out,*,*,i_region], $
          n_id_indep_out, n_dep_source * n_indep_source )
      impact_conf_nat[i_region] = ipccar6dna_conf_regcoef( $
          'consistent with 1', frac_dist_beta_positive=temp_frac, $
          max_degrade=0.1, p_limit=p_limit, method='distribution', $
          source_weight=temp_weight )
    endelse
  ; If the evaluation is performed by a weighted extreme p-value count
  ; (IPCC AR6)
  endif else if max( options eq 'regcoef=weighted binomial' ) eq 1 then begin
    if n_elements( depindep_weight ) gt 0 then begin
      temp_weight = reform( depindep_weight[*,*,i_region], $
          n_dep_source * n_indep_source )
    endif else begin
      temp_weight = 0
    endelse
    if ( n_elements( depindep_weight ) gt 0 ) and ( max( temp_weight ) eq 0 ) $
        then begin
      impact_conf_nat[i_region] = !values.f_nan
    endif else begin
      impact_conf_nat[i_region] = ipccar6dna_conf_regcoef( $
          'consistent with 1', beta_est=temp_beta, max_degrade=0.1, $
          p_limit=p_limit, method='weighted binomial', $
          source_weight=temp_weight )
    endelse
  ; If the evaluation is performed via extreme p-value count 
  ; (Stone and Hansen 2016)
  endif else if max( options eq 'regcoef=binomial' ) eq 1 then begin
    if max( options eq 'weight indep resolution' ) eq 1 then stop
    if max( options eq 'weight dep density' ) eq 1 then stop
    impact_conf_nat[i_region] = ipccar6dna_conf_regcoef( 'consistent with 1', $
        beta_est=temp_beta, max_degrade=0.1, p_limit=p_limit, $
        method='binomial' )
  endif else begin
    stop
  endelse
endfor

; Modify confidence according to consistency of residuals
impact_conf_resid = fltarr( n_region )
for i_region = 0, n_region - 1 do begin
  ; If the evaluation is performed via a K-S test
  if max( options eq 'resid=k-s' ) eq 1 then begin
    if n_elements( depindep_weight ) gt 0 then begin
      temp_weight = depindep_weight[*,*,i_region]
    endif else begin
      temp_weight = 0
    endelse
    if ( n_elements( depindep_weight ) gt 0 ) and ( max( temp_weight ) eq 0 ) $
        then begin
      impact_conf_resid[i_region] = !values.f_nan
    endif else begin
      impact_conf_resid[i_region] = ipccar6dna_conf_resid( $
          regress_p_resid[*,*,i_region], p_limit=p_limit, $
          source_weight=temp_weight, method='k-s' )
    endelse
  ; If the evaluation is performed via a weighted extreme p-value count 
  ; (IPCC AR6)
  endif else if max( options eq 'resid=weighted binomial' ) eq 1 then begin
    if n_elements( depindep_weight ) gt 0 then begin
      temp_weight = depindep_weight[*,*,i_region]
    endif else begin
      temp_weight = 0
    endelse
    if ( n_elements( depindep_weight ) gt 0 ) and ( max( temp_weight ) eq 0 ) $
        then begin
      impact_conf_resid[i_region] = !values.f_nan
    endif else begin
      impact_conf_resid[i_region] = ipccar6dna_conf_resid( $
          regress_p_resid[*,*,i_region], p_limit=p_limit, $
          source_weight=temp_weight, method='weighted binomial' )
    endelse
  ; If the evaluation is performed via extreme p-value count 
  ; (Stone and Hansen 2016)
  endif else if max( options eq 'resid=binomial' ) eq 1 then begin
    if max( options eq 'weight indep resolution' ) eq 1 then stop
    if max( options eq 'weight dep density' ) eq 1 then stop
    impact_conf_resid[i_region] = ipccar6dna_conf_resid( $
        regress_p_resid[*,*,i_region], p_limit=p_limit, method='binomial' )
  endif else begin
    stop
  endelse
endfor

; Modify major-attribution confidence according to quantile of major role 
; threshold for attributable trends.
impact_conf_major = fltarr( n_region )
for i_region = 0, n_region - 1 do begin
  if n_elements( depindep_weight ) gt 0 then begin
    temp_weight = reform( depindep_weight[*,*,i_region], $
        n_dep_source * n_indep_source )
  endif else if ( max( options eq 'no weight size' ) eq 1 ) $
      and ( max( options eq 'no weight density' ) eq 1 ) then begin
    temp_weight = 0
  endif else begin
    stop
  endelse
  if ( n_elements( depindep_weight ) gt 0 ) and ( max( temp_weight ) eq 0 ) $
      then begin
    impact_conf_major[i_region] = !values.f_nan
  endif else begin
    temp_frac = reform( regress_frac_dist_trend_major[*,*,i_region], $
        1, n_dep_source * n_indep_source )
    impact_conf_major[i_region] = ipccar6dna_conf_regcoef( $
      'consistent with gt 0', frac_dist_beta_positive=temp_frac, $
      source_weight=temp_weight )
  endelse
endfor

; Calculate the detection confidence metric
impact_conf_factor = [ [ impact_conf_nsource_dep ], $
    [ impact_conf_nsource_indep ] ]
impact_conf_label = [ 'NOBS', 'NMODEL' ]
if n_elements( impact_conf_sign ) gt 0 then begin
  impact_conf_factor = [ [ impact_conf_factor ], [ impact_conf_sign ] ]
  impact_conf_label = [ impact_conf_label, 'SIGN' ]
endif
if n_elements( impact_conf_dense ) gt 0 then begin
  impact_conf_factor = [ [ impact_conf_factor ], [ impact_conf_dense ] ]
  impact_conf_label = [ impact_conf_label, 'DENSE' ]
endif
if n_elements( impact_conf_size ) gt 0 then begin
  impact_conf_factor = [ [ impact_conf_factor ], [ impact_conf_size ] ]
  impact_conf_label = [ impact_conf_label, 'SIZE' ]
endif
impact_conf_factor = [ [ impact_conf_factor ], [ impact_conf_physics ], $
    [ impact_conf_signal ], [ impact_conf_ant ], [ impact_conf_nat ], $
    [ impact_conf_resid ] ]
impact_conf_label = [ impact_conf_label, 'PHYSICS', 'SIGNAL', 'ANT', 'NAT', $
    'RESID' ]
impact_conf_factor = transpose( impact_conf_factor )
temp_impact_conf_factor = impact_conf_factor
id = where( finite( temp_impact_conf_factor ) eq 0, n_id )
if n_id gt 0 then temp_impact_conf_factor[id] = 0.
impact_conf_detect = product( temp_impact_conf_factor, 1 )
; Calculate the attribution confidence metric
impact_conf_factor = [ impact_conf_factor, reform( impact_conf_major, 1, n_region ) ]
impact_conf_label = [ impact_conf_label, 'MAJOR' ]
temp_impact_conf_factor = impact_conf_factor
id = where( finite( temp_impact_conf_factor ) eq 0, n_id )
if n_id gt 0 then temp_impact_conf_factor[id] = 0.
impact_conf_attrib_major = product( temp_impact_conf_factor, 1 )
; Count the number of steps we implemented
n_step = n_elements( impact_conf_label )
if n_elements( impact_conf_factor[*,0] ) ne n_step then stop
if n_elements( impact_conf_factor[0,*] ) ne n_region then stop

; Interpret "zero" as no confidence at all
id = where( impact_conf_detect lt 0.01, n_id )
if n_id gt 0 then impact_conf_detect[id] = -1.
id = where( impact_conf_attrib_major lt 0.01, n_id )
if n_id gt 0 then impact_conf_attrib_major[id] = -1.

; Determine detection confidence labels
id = where( impact_conf_detect ge 0, n_id )
if n_id gt 0 then begin
  impact_conf_detect[id] = alog10( 1. + impact_conf_detect[id] * 3. ) * 2.5 $
      / alog10( 4. )
endif
id = floor( impact_conf_detect )
if id gt 4 then id = 4
impact_conf_detect_label = conf_label[id+1]
; Determine attribution-major-role confidence labels
id = where( impact_conf_attrib_major ge 0, n_id )
if n_id gt 0 then begin
  impact_conf_attrib_major[id] $
      = alog10( 1. + impact_conf_attrib_major[id] * 3. ) * 2.5 / alog10( 4. )
endif
id = floor( impact_conf_attrib_major )
if id gt 4 then id = 4
impact_conf_attrib_major_label = conf_label[id+1]

if verbose_opt eq 1 then begin
  verbose_time_new = systime( seconds=1 )
  print, 'ipccar6dna_conf:  Completed assessment', $
      verbose_time_new - verbose_time_old, verbose_time_new - verbose_time_0
  verbose_time_old = verbose_time_new
endif

;***********************************************************************
; Scripts for plotting diagnostics

; Print output
if max( diagnostic eq 'print' ) eq 1 then begin
  ; Iterate through regions
  for i_region = 0, n_region - 1 do begin
    print, impact_region_def[i_region]
    ; Print component factors
    for i_factor = 0, n_step - 1 do begin
      print, '  ' + str( impact_conf_factor[i_factor,i_region] ) + '  ' $
          + impact_conf_label[i_factor]
    endfor
    ; Print climate confidence
    print, '  Detection:  ' + impact_conf_detect_label[i_region]
    print, '  Attribution major role:  ' $
        + impact_conf_attrib_major_label[i_region]
  endfor
endif

; Plot regression coefficients
if max( diagnostic eq 'betas' ) eq 1 then begin
  if n_region ne 1 then stop
  ps_open, filename='betas.ps', color=1
  device, helvetica=1, bold=1
  tek_color
  charsize = 1.3
  thick = 3
  xticks = n_indep_trans * n_good_mod + 1
  xtickv = findgen( n_indep_trans * n_good_mod + 1 ) * n_source_obs - 0.5
  xtickname = ' ' + strarr( xticks + 1 )
  yrange = [ -1, 3 ]
  xrange = [ -2, n_indep_trans * n_good_mod * n_source_obs + 1 ]
  plot, beta_est[1,0,*,*], nodata=1, xrange=xrange, xstyle=1, $
      yrange=yrange, font=0, charsize=charsize, xthick=thick, ythick=thick, $
      xtickv=xtickv, xtickname=xtickname, ystyle=1, xticks=xticks
  oplot, xrange, [0,0], linestyle=0, color=14, thick=thick
  oplot, xrange, [1,1], linestyle=0, color=14, thick=thick
  for i_trans = 0, n_indep_trans - 1 do begin
    for i_mod = 0, n_good_mod - 1 do begin
      for i_obs = 0, n_source_obs - 1 do begin
        xpos = i_trans * n_source_obs * n_good_mod + i_mod * n_source_obs $
            + i_obs + [ 0, 0 ]
        oplot, xpos, beta_est[1:2,i_trans,i_obs,id_good_mod[i_mod]], $
            color=2+i_trans, linestyle=i_mod, thick=thick, psym=-1
      endfor
    endfor
  endfor
  for i_trans = 0, n_indep_trans - 1 do begin
    xpos = ( i_trans + 0.5 ) * n_source_obs * n_good_mod
    ypos = yrange[1] + ( yrange[1] - yrange[0] ) / 40.
    xyouts, xpos, ypos, scenario_indep_trans_name[i_trans], alignment=0.5, $
        font=0, charsize=charsize
    for i_mod = 0, n_good_mod - 1 do begin
      xpos = i_trans * n_source_obs * n_good_mod $
          + ( i_mod + 0.5 ) * n_source_obs
      ypos = yrange[0] - ( yrange[1] - yrange[0] ) / 20.
      xyouts, xpos, ypos, source_unique_mod[id_good_mod[i_mod]], $
          orientation=90, font=0, charsize=charsize, alignment=1
      if i_trans + i_mod ne 0 then begin
        xpos = i_trans * n_source_obs * n_good_mod + i_mod * n_source_obs $
            - 0.5
        oplot, [xpos,xpos], yrange, thick=thick, color=15, linestyle=1
      endif
    endfor
  endfor
  ps_close
endif

; Plot p-value of residuals
if max( diagnostic eq 'p_resid' ) eq 1 then begin
  if n_region ne 1 then stop
  ps_open, filename='p_resid.ps', color=1
  device, helvetica=1, bold=1
  tek_color
  charsize = 1.3
  thick = 3
  xticks = n_good_mod + 1
  xtickv = findgen( n_good_mod + 1 ) * n_source_obs - 0.5
  xtickname = ' ' + strarr( xticks + 1 )
  yrange = [ 0, 1 ]
  xrange = [ -2, n_good_mod * n_source_obs + 1 ]
  plot, p_resid[*,*], nodata=1, xrange=xrange, xstyle=1, $
      yrange=yrange, font=0, charsize=charsize, xthick=thick, ythick=thick, $
      xtickv=xtickv, xtickname=xtickname, ystyle=1, xticks=xticks
  oplot, xrange, [0.05,0.05], linestyle=1, color=14, thick=thick
  oplot, xrange, [0.95,0.95], linestyle=1, color=14, thick=thick
  for i_mod = 0, n_good_mod - 1 do begin
    for i_obs = 0, n_source_obs - 1 do begin
      xpos = i_mod * n_source_obs + i_obs + [ 0, 0 ]
      plots, xpos, p_resid[i_obs,id_good_mod[i_mod]], color=2, thick=thick, $
          psym=4
    endfor
  endfor
  for i_mod = 0, n_good_mod - 1 do begin
    xpos = ( i_mod + 0.5 ) * n_source_obs
    ypos = yrange[0] - ( yrange[1] - yrange[0] ) / 20.
    xyouts, xpos, ypos, source_unique_mod[id_good_mod[i_mod]], $
        orientation=90, font=0, charsize=charsize, alignment=1
    if i_mod ne 0 then begin
      xpos = i_mod * n_source_obs - 0.5
      oplot, [xpos,xpos], yrange, thick=thick, color=15, linestyle=1
    endif
  endfor
  ps_close
endif

;***********************************************************************
; The end

;stop
return
END
