;+
; NAME:
;    gpd_fit
;
; PURPOSE:
;    This function estimates the parameters of the Generalised Pareto 
;    Distribution for a set of values exceeding a threshold.
;
; CATEGORY:
;    Statistics
;
; CALLING SEQUENCE:
;    result = gpd_fit( exceed )
;
; INPUTS:
;    EXCEED:  A float vector of length N_EXCEED containing the values 
;        that exceed some threshold.  Note that these values should be the 
;        amount by which the threshold is exceeded, not the original data, 
;        so all values should be non-negative.  This also means we are defining 
;        the location parameter for the distribution of the input values to be 
;        zero, so it is ignored in this function.  Required, unless DATA_BLOCK
;        and DATA_THRESHOLD are input.
;    BOOTSTRAP_SEED, CHARSIZE, CI_COLOR, COLOR, COVARIATE_DATA, 
;      COVARIATE_PARAM_ID, FONT, N_BOOTSTRAP, N_DATA, P_VALUE, 
;      PLOT_PROBABILITY, PLOT_QUANTILE, THICK, TITLE, XTHICK, XTITLE, YTHICK, 
;      YTITLE
;
; KEYWORD PARAMETERS:
;    BOOTSTRAP_CDF:  If CDF is set and N_BOOTSTRAP is input, then this returns 
;        the cumulative density function values for the bootstrap GP 
;        distributions at the EXCEED values.
;    BOOTSTRAP_SEED:  An optional scalar integer containing the seed for the 
;        random number generator used for the bootstrap sampling.  This is 
;        useful for reproducibility.
;    CHARSIZE:  The optional CHARSIZE keyword parameter for the plot function.
;    CDF:  If set, then this returns a N_EXCEED float vector containing the 
;        cumulative density function values of the GP distribution at the 
;        EXCEED values.
;    CI_COLOR:  An optional scalar integer specifying the color index for 
;        plotting confidence intervals.
;    COLOR:  The optional COLOR keyword parameter for the plot function.
;    COVARIATE_DATA:  An optional float array containing values of covariate 
;        functions at the location of each of the N_EXCEED values.  Of size 
;        N_EXCEED,N_COVARIATE, where N_COVARIATE is the number of covariates.  
;        If input then COVARIATE_PARAM_ID must also be input.
;    COVARIATE_PARAM_ID:  An optional integer array if length N_COVARIATE 
;        specifying the GP parameter incorporating each of the N_COVARIATE 
;        covariates in COVARIATE_DATA.  0 specifies the location parameter, 1 
;        the scale parameter, and 2 the shape parameter.  Required if 
;        COVARIATE_DATA is input.
;    DATA_BLOCK:  If EXCEED is not input, then this is a required 
;        N_SAMPLE,N_BLOCK float array of data from which to extract values that 
;        exceed DATA_THRESHOLD.  The difference between the NSAMPLE and N_BLOCK 
;        dimensions is that the N_BLOCK dimension is resampled if bootstrap 
;        confidence intervals are requested.  Note that this input internalises 
;        the peaks-of-threshold extraction that would have occurred before 
;        calling the function if EXCEED were input.  The advantage is that the 
;        bootstrap resampling can consider the full data set
;    DATA_THRESHOLD:  The scalar float specifying the threshold for designation 
;        of exceedances.  Required if DATA_BLOCK is input (and EXCEED is not).
;    FONT:  The optional FONT keyword parameter for the plot function.
;    N_BOOTSTRAP:  An optional scalar integer defining the number of bootstrap 
;        samples to use in estimating confidence intervals on the parameters 
;        using a bootstrap approach.
;    N_DATA:  A scalar integer specifying the total number of data values from 
;        which the threshold exceedances were taken.  This is used if bootstrap 
;        sampling is used in order to determine the probability of having 
;        different numbers of values exceeding the threshold.  If not set, then 
;        it is assumed that N_EXCEED values are sampled for each bootstrap 
;        sample.
;    P_VALUE:  An optional scalar float containing the p-value for any 
;        confidence interval estimates.  The default is 0.10.
;    PLOT_PROBABILITY:  If set then the function plots a probability plot.
;    PLOT_QUANTILE:  If set then the function plots a quantile-quantile plot.  
;        This is not possible if there are covariates defined.
;    PARAMS_CI:  Returns a 2,(3+N_COVARIATE) float array containing the 
;        estimated 1-P_VALUE confidence intervals on the GP model parameters.  
;        The first dimension returns the lower and upper bounds respectively, 
;        while the order of parameters in the second dimension corresponds to 
;        the order in RESULT.  If N_BOOTSTRAP is input, these confidence 
;        intervals are estimated using a bootstrap approach;  no other method 
;        is implemented yet.
;    THICK:  The optional THICK keyword parameter for the plot function.
;    [X,Y]THICK:  The optional XTHICK and YTHICK keyword parameters for the 
;        plot function.
;    TITLE:  The optional TITLE keyword parameter for the plot function.
;    [X,Y]TITLE:  The optional XTITLE and YTITLE keyword parameters for the 
;        plot function.
;
; OUTPUTS:
;    RESULT:  A float vector containing the estimated values of the GP model 
;        parameters.  The first element contains the value for the scale 
;        parameter, the second element contains the value for the shape 
;        parameter, and any further elements contain the values for the 
;        regression parameters on the N_COVARIATE covariate functions.
;    BOOTSTRAP_CDF, CDF, PARAMS_CI
;
; USES:
;    dbinom.pro
;    gpd_cdf.pro
;    gpd_pdf.pro
;    quantile_threshold.pro
;    shuffle.pro
;    ;gpd_fit_eqn.pro (included in this file)
;
; PROCEDURE:
;    This function estimates the parameters of the GPD model using the maximum 
;    log likelihood method.  If covariates are included, the total log 
;    likelihood function is the sum of the likelihood function at each of the 
;    specific N_EXCEED locations.  The log likelihood function is contained 
;    within the gpd_fit_eqn.pro subfunction.
;
; EXAMPLE:
;    ; Generate 100 years of daily data following a Gaussian distribution
;    n_day = 365
;    n_year = 200
;    data_block = randomn( 1, n_day, n_year )
;    ; Define a threshold for exceedance (approximately 1-in-1-year, i.e. comparable to annual maxima)
;    data_threshold = gauss_cvf( 1. / n_day )
;    ; Calculate the parameters of the GPD model and show the probability plot
;    params = gpd_fit( data_block=data_block, data_threshold=data_threshold, n_bootstrap=1000, plot_probability=1 )
;    ; Repeat, but calculating the exceedance before input
;    id = where( data_block ge datathreshold, n_exceed )
;    exceed = data_block[id] - data_threshold
;    ; Calculate the parameters of the GPD model and show the probability plot
;    params = gpd_fit( exceed, n_bootstrap=1000, plot_probability=1 )
;
; LICENSE:
;    This code was written as part of the Extreme Weather Event Real-Time 
;    Attribution Machine (EWERAM) project supported by the New Zealand Ministry 
;    of Business, Innovation and Employment.  The code is free for use under 
;    the terms of the Creative Commons License v2.0 
;    (http://creativecommons.org/licenses/by-nc-sa/2.0/).
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2019-11-05
;    Modified:  DAS, 2020-06-19 (Added preventation of negative scale values)
;-

;***********************************************************************

FUNCTION GPD_FIT, $
    EXCEED, $
    BOOTSTRAP_SEED=bootstrap_seed, $
    CHARSIZE=charsize, FONT=font, $
    COLOR=color, CI_COLOR=ci_color, $
    COVARIATE_DATA=covariate_data_0, COVARIATE_PARAM_ID=covariate_param_id, $
    DATA_BLOCK=data_block, DATA_THRESHOLD=data_threshold, $
    N_BOOTSTRAP=n_bootstrap, $
    N_DATA=n_data, $
    P_VALUE=p_value, $
    THICK=thick, XTHICK=xthick, YTHICK=ythick, $
    TITLE=title, XTITLE=xtitle, YTITLE=ytitle, $
    PLOT_PROBABILITY=plot_probability_opt, PLOT_QUANTILE=plot_quantile_opt, $
    BOOTSTRAP_CDF=bootstrap_cdf, $
    CDF=cdf, $
    PARAMS_CI=params_ci

;***********************************************************************
; Constants and options

; Create common block for sharing with gpd_fit_eqn.pro
common common_gpd_fit_eqn, gpd_exceed, gpd_covariate_data, $
    gpd_covariate_param_id

; The tolerance value for the amoeba function
amoeba_ftol = 1.0e-5

; The default p-value
if not( keyword_set( p_value ) ) then p_value = 0.1

; The number of values that exceed the threshold
n_exceed = n_elements( exceed )
if ( n_exceed le 2 ) and ( n_exceed gt 0 ) then stop
; Ensure positive exceedances
if n_exceed gt 0 then begin
  if min( exceed ) lt 0 then stop
endif
; If exceedances are not input
if n_exceed eq 0 then begin
  ; Ensure that the full data set and threshold are input
  if not( keyword_set( data_block ) ) then stop
  if not( keyword_set( data_threshold ) ) then stop
  ; Determine the data size
  n_block = n_elements( data_block[0,*] )
  n_sample = n_elements( data_block[*,0] )
endif else begin
  if keyword_set( data_block ) then stop
  n_block = 0
  n_sample = 0
endelse

; Check covariate inputs
n_covariate = n_elements( covariate_param_id )
if n_covariate gt 0 then begin
  ; Confirm everything is there
  if not( keyword_set( covariate_data_0 ) ) then stop
  if n_exceed gt 0 then begin
    if n_elements( covariate_data_0[*,0] ) ne n_exceed then stop
    if n_elements( covariate_data_0[0,*] ) ne n_covariate then stop
  endif else begin
    if n_elements( covariate_data_0[*,0,0] ) ne n_sample then stop
    if n_elements( covariate_data_0[0,*,0] ) ne n_block then stop
    if n_elements( covariate_data_0[0,0,*] ) ne n_covariate then stop
  endelse
  ; Copy to array we will use (modified later if N_BLOCK>0)
  covariate_data = covariate_data_0
  ; Calculate covariate ranges (for use in setting up amoeba solver)
  if n_exceed gt 0 then begin
    covariate_range = max( covariate_data, dimension=1 ) $
        - min( covariate_data, dimension=1 )
  endif else begin
    covariate_range = max( reform( $
        covariate_data, n_sample * n_block, n_covariate ), dimension=1, $
        min=temp ) $
        - temp
  endelse
  ; Confirm covariates vary
  if min( covariate_range ) eq 0 then stop
endif

; Extract exceedances from full data set
if n_block gt 0 then begin
  id_exceed = where( data_block ge data_threshold, n_exceed )
  if n_exceed lt 10 then stop ;Arbitrary minimum for reasonable sampling
  exceed = data_block[id_exceed] - data_threshold
  if n_covariate gt 0 then begin
    covariate_data = reform( covariate_data, n_sample * n_block, n_covariate )
    covariate_data = covariate_data[id_exceed,*]
  endif
endif

; Copy data required by minimisation function to common block
; (Fitting is unreliable without double precision)
gpd_exceed = double( exceed )
if n_covariate gt 0 then begin
  gpd_covariate_data = double( covariate_data )
  gpd_covariate_param_id = covariate_param_id
; Otherwise clear unneeded variables
endif else begin
  if n_elements( gpd_covariate_data ) gt 0 then begin
    temp = temporary( gpd_covariate_data )
  endif
  if n_elements( gpd_covariate_param_id ) gt 0 then begin
    temp = temporary( gpd_covariate_param_id )
  endif
endelse

; Ensure the CDF is calculated if plotting is requested
if keyword_set( plot_probability_opt ) or keyword_set( plot_quantile_opt ) $
    then begin
  cdf = 1
endif

;***********************************************************************
; Estimate parameters

; Make a first guess of parameters (assuming near-exponential)
params_0 = dblarr( 2 )
params_0[1] = 1.e-8
params_0[0] = mean( gpd_exceed )
; Define scale for first iteration of solver's search
params_scale = [ 0.5 * params_0[0], 0.1 ]
; Add any covariates to the first guess
if n_covariate gt 0 then begin
  ; Assume zero scaling on covariates
  params_0 = [ params_0, fltarr( n_covariate ) ]
  ; Define scale for first iteration of solver's search
  params_scale = [ params_scale, fltarr( n_covariate ) ]
  for i_covariate = 0, n_covariate - 1 do begin
    params_scale[2+i_covariate] = params_0[covariate_param_id] * 0.5 $
        / covariate_range[i_covariate]
  endfor
endif

; Fit parameters
params = amoeba( amoeba_ftol, function_name='gpd_fit_eqn', p0=params_0, $
    scale=params_scale )

; Estimate confidence intervals using bootstrap
if keyword_set( n_bootstrap ) then begin
  ; Initialise array containing bootstrap parameter estimates
  boot_params = fltarr( 2 + n_covariate, n_bootstrap )
  ; Determine the probability of achieving various numbers of exceedances 
  ; given the data size and exceedance probability
  if keyword_set( n_data ) then begin
    if n_data lt n_exceed then stop
    boot_n_exceed = lindgen( n_data + 1 )
    boot_freq_exceed = fltarr( n_data + 1 )
    check_flag = 0
    index = n_exceed
    while check_flag eq 0 do begin
      if index lt 0 then begin
        check_flag = 1
      endif else begin
        temp = dbinom( boot_n_exceed[index], n_data, $
            double( n_exceed ) / n_data, binomial=0, double=1 )
        if temp eq 0 then begin
          check_flag = 1
        endif else begin
          boot_freq_exceed[index] = temp
          index = index - 1
        endelse
      endelse
    endwhile
    check_flag = 0
    index = n_exceed + 1
    while check_flag eq 0 do begin
      temp = dbinom( boot_n_exceed[index], n_data, $
          double( n_exceed ) / n_data, binomial=0, double=1 )
      if temp eq 0 then begin
        check_flag = 1
      endif else begin
        boot_freq_exceed[index] = temp
        index = index + 1
      endelse
    endwhile
    boot_freq_exceed_cumul = total( boot_freq_exceed, cumulative=1 ) $
        * n_bootstrap
  ; Otherwise clear variables
  endif else begin
    id_exceed = 0
  endelse
  ; Iterate through bootstrap samples
  for i_boot = 0, n_bootstrap - 1 do begin
    ; Extract exceedances from full data set if input
    if n_block gt 0 then begin
      index = shuffle( indgen( n_block ), seed=bootstrap_seed, replace=1 )
      temp_data_block = data_block[*,index]
      id_exceed = where( temp_data_block ge data_threshold, n_id_exceed )
      if n_id_exceed lt 10 then stop ;Arbitrary minimum for reasonable sampling
      gpd_exceed = temp_data_block[id_exceed] - data_threshold
      if n_covariate gt 0 then begin
        gpd_covariate_data = reform( covariate_data_0[*,index,*], $
            n_sample * n_block, n_covariate )
        gpd_covariate_data = gpd_covariate_data[id_exceed,*]
      endif
    ; If we just shuffle the input exceedances
    endif else begin
      ; Determine how many samples to take
      if keyword_set( n_data ) then begin
        id = max( where( boot_freq_exceed_cumul le i_boot + 0.5, n_id_exceed ) )
      endif
      ; Select random sample of exceedances
      gpd_exceed = shuffle( double( exceed ), seed=bootstrap_seed, $
          replace=1, index=boot_index, n_out=n_id_exceed )
      if n_covariate gt 0 then begin
        gpd_covariate_data = double( covariate_data[boot_index,*] )
      endif
    endelse
    ; Fit parameters
    boot_params[*,i_boot] = amoeba( 1.0e-5, function_name='gpd_fit_eqn', $
        p0=params_0, scale=params_scale )
  endfor
  ; Determine confidence interval
  params_ci = fltarr( 2, 2 + n_covariate )
  for i_param = 0, 2 + n_covariate - 1 do begin
    temp = quantile_threshold( reform( boot_params[i_param,*] ), $
        [ p_value / 2., 1. - p_value / 2. ] )
    params_ci[1,i_param] = params[i_param] + ( params[i_param] - temp[0] )
    params_ci[0,i_param] = params[i_param] - ( temp[1] - params[i_param] )
  endfor
endif

; Calculate the CDF for the PARAMS parameter values
if keyword_set( cdf ) then begin
  ; Assemble parameters
  if n_covariate eq 0 then begin
    sigma = params[0]
    xi = params[1]
  endif else begin
    sigma = params[0] + fltarr( n_exceed )
    id = where( covariate_param_id eq 0, n_id )
    if n_id gt 0 then sigma = sigma + params[2+id] ## covariate_data[*,id]
    xi = params[1] + fltarr( n_exceed )
    id = where( covariate_param_id eq 1, n_id )
    if n_id gt 0 then xi = xi + params[2+id] ## covariate_data[*,id]
  endelse
  ; Calculate the cdf
  cdf = gpd_cdf( exceed, scale=sigma, shape=xi )
  ; If we have bootstrap samples
  if keyword_set( n_bootstrap ) then begin
    ; Initialise bootstrap CDF array
    bootstrap_cdf = fltarr( n_exceed, n_bootstrap )
    ; Iterate through bootstrap samples
    for i_boot = 0, n_bootstrap - 1 do begin
      ; Assemble parameters
      if n_covariate eq 0 then begin
        sigma = boot_params[0,i_boot]
        xi = boot_params[1,i_boot]
      endif else begin
        sigma = boot_params[0,i_boot] + fltarr( n_exceed )
        id = where( covariate_param_id eq 0, n_id )
        if n_id gt 0 then begin
          sigma = sigma + boot_params[2+id,i_boot] ## covariate_data[*,id]
        endif
        xi = boot_params[1,i_boot] + fltarr( n_exceed )
        id = where( covariate_param_id eq 1, n_id )
        if n_id gt 0 then begin
          xi = xi + boot_params[2+id,i_boot] ## covariate_data[*,id]
        endif
      endelse
      ; Calculate the CDF
      bootstrap_cdf[*,i_boot] = gpd_cdf( exceed, scale=sigma, shape=xi ) 
    endfor
  endif
endif

;***********************************************************************
; Plot output

; Produce a probability plot
if keyword_set( plot_probability_opt ) then begin
  ; Default axis titles
  if not( keyword_set( xtitle ) ) then xtitle = 'Empirical'
  if not( keyword_set( ytitle ) ) then ytitle = 'Model'
  ; Determine the empirical quantiles
  quantiles = ( findgen( n_exceed ) + 0.5 ) / n_exceed
  ; Set up plotting window, including diagonal
  plot, [0,1], [0,1], isotropic=1, linestyle=1, xtitle=xtitle, ytitle=ytitle, $
      xthick=xthick, ythick=ythick, thick=thick, charsize=charsize, font=font, $
      title=title
  ; If we have bootstrap samples
  if keyword_set( n_bootstrap ) then begin
    ; Sort bootstrap samples
    for i_boot = 0, n_bootstrap - 1 do begin
      id = sort( bootstrap_cdf[*,i_boot] )
      bootstrap_cdf[*,i_boot] = bootstrap_cdf[id,i_boot]
    endfor
    ; Plot confidence range at each quantile
    for i_exceed = 0, n_exceed - 1 do begin
      temp = quantile_threshold( bootstrap_cdf[i_exceed,*], $
          [ p_value / 2., 1. - p_value / 2. ] )
      oplot, quantiles[i_exceed]+[0,0], temp, thick=thick, color=ci_color
    endfor
  endif
  ; Plot the points
  id = sort( cdf )
  oplot, quantiles, cdf[id], psym=4, thick=thick, color=color
endif

; Produce a quantile-quantile plot
if keyword_set( plot_quantile_opt ) then begin
  ; I do not think this is possible if there are covariates
  if n_covariate gt 0 then stop
  ; Default axis titles
  if not( keyword_set( xtitle ) ) then xtitle = 'Data'
  if not( keyword_set( ytitle ) ) then ytitle = 'Model'
  ; Determine the empirical locations for the quantiles
  id_sort = sort( exceed )
  temp_exceed = exceed[id_sort]
  ; Determine the GPD model locations for the empirical quantiles
  sigma = params[0]
  xi = params[1]
  quantiles = ( findgen( n_exceed ) + 0.5 ) / n_exceed
  if xi eq 0 then begin
    inv_quantiles = -sigma * alog( 1. - quantiles )
  endif else begin
    inv_quantiles = ( ( 1. - quantiles ) ^ ( -xi ) - 1. ) * sigma / xi
  endelse
  ; Determine plotting range
  xyrange = [ min( [ inv_quantiles, exceed ], max=temp ), temp ]
  xyrange = ( xyrange - mean( xyrange ) ) * 1.05 + mean( xyrange )
  ; Set up plotting window
  plot, xyrange, xyrange, xstyle=1, ystyle=1, isotropic=1, linestyle=1, $
      xtitle=xtitle, ytitle=ytitle, xthick=xthick, ythick=ythick, thick=thick, $
      charsize=charsize, font=font, title=title
  ; If we have bootstrap samples
  if keyword_set( n_bootstrap ) then begin
    ; Calculate the bootstrap GPD model locations
    boot_inv_quantiles = fltarr( n_exceed, n_bootstrap )
    for i_boot = 0, n_bootstrap - 1 do begin
      if boot_params[1,i_boot] eq 0 then begin
        boot_inv_quantiles[*,i_boot] = -boot_params[0,i_boot] $
            * alog( 1. - quantiles )
      endif else begin
        boot_inv_quantiles[*,i_boot] $
            = ( ( 1. - quantiles ) ^ ( -boot_params[1,i_boot] ) - 1. ) $
            * boot_params[0,i_boot] / boot_params[1,i_boot]
      endelse
    endfor
    ; Sort bootstrap samples
    for i_boot = 0, n_bootstrap - 1 do begin
      id = sort( boot_inv_quantiles[*,i_boot] )
      boot_inv_quantiles[*,i_boot] = boot_inv_quantiles[id,i_boot]
    endfor
    ; Plot confidence range at each quantile
    for i_exceed = 0, n_exceed - 1 do begin
      temp = quantile_threshold( boot_inv_quantiles[i_exceed,*], $
          [ p_value / 2., 1. - p_value / 2. ] )
      oplot, temp_exceed[[i_exceed,i_exceed]], temp, thick=thick, $
          color=ci_color
    endfor
  endif
  ; Plot quantile-quantile plot
  oplot, temp_exceed, inv_quantiles, psym=4, thick=thick, color=color
endif

;***********************************************************************
; The end

return, params
END


;***********************************************************************
; NAME:
;    gpd_fit
; PURPOSE:
;    This function contains the GP likelihood function to be minimised by 
;    gpd_fit.pro.
; INPUTS:
;    PARAMS:  A required float vector containing proposed values of the GP 
;        model parameters.  The length and order are the same as for RESULT 
;        from gpd_fit.pro.
; COMMON BLOCK common_gpd_fit_eqn inputs:
;    GPD_EXCEED:  Same as the EXCEED input for gpd_fit.pro.  Required.
;    GPD_COVARIATE_DATA:  Same as COVARIATE_DATA keyword input for 
;        gpd_fit.pro.  Required if covariates are to be included.
;    GPD_COVARIATE_PARAM_ID:  Same as COVARIATE_PARAM_ID keyword input for 
;        gpd_fit.pro.  Required if covariates are to be included.
; USES:
;    gpd_pdf.pro
; OUTPUTS:
;    RESULT:  The negative of the log likelihood function for the GP model 
;        given the input set of parameters.  The negative is returned because 
;        IDL's amoeba function performs a minimisation, while we want a 
;        maximumisation of the log likelihood.
;***********************************************************************

FUNCTION GPD_FIT_EQN, $
    PARAMS

;***********************************************************************
; Constants and inputs

; Get input values
common common_gpd_fit_eqn
; Determine the data length
n_exceed = n_elements( gpd_exceed )
; Determine the number of covariates
n_covariate = n_elements( gpd_covariate_param_id )
; Define parameters for use in calculations (including covariates)
sigma = params[0]
xi = params[1]
if n_covariate gt 0 then begin
  id = where( gpd_covariate_param_id eq 0, n_id )
  if n_id gt 0 then begin
    sigma = sigma + params[2+id] ## gpd_covariate_data[*,id]
  endif else begin
    sigma = sigma + fltarr( n_exceed )
  endelse
  id = where( gpd_covariate_param_id eq 1, n_id )
  if n_id gt 0 then begin
    xi = xi + params[2+id] ## gpd_covariate_data[*,id]
  endif else begin
    xi = xi + fltarr( n_exceed )
  endelse
endif

;***********************************************************************
; Calculate log likelihood

; Abort if illegal parameters
id_sigma = where( sigma lt 0, n_id_sigma )
if n_id_sigma gt 0 then begin
  return, double( n_exceed ) * 100
endif
id_xi = where( xi lt 0, n_id_xi )
if n_id_xi gt 0 then begin
  if n_covariate eq 0 then begin
    temp = -sigma / xi - gpd_exceed
  endif else begin
    temp = -sigma[id_xi] / xi[id_xi] - gpd_exceed[id_xi]
  endelse
  if min( temp ) lt 0 then return, double( n_exceed ) * 100
endif

; Calculate the log likelihood
result = gpd_pdf( gpd_exceed, scale=sigma, shape=xi, log=1 )
result = total( result )

; Take negative (so amoeba is effectively maximising rather than minimising)
result = -result

;***********************************************************************
; The end

return, result
END
