;+
; NAME:
;    gev_cdf
;
; PURPOSE:
;    This function calculates the cumulative distribution function of the 
;    Generalised Extreme Value distribution for a given set of parameters.
;
; CATEGORY:
;    Statistics
;
; CALLING SEQUENCE:
;    gev_cdf, values, location=location, scale=scale, shape=shape
;
; INPUTS:
;    VALUES:  A required floating vector of length N_VALUES specifying the 
;        values at which to calculate the cumulative distribution function.
;    LOCATION, SCALE, SHAPE
;
; KEYWORD PARAMETERS:
;    INVERT:  If set, then the input VALUES and output RESULT are swapped.  In 
;        other words, VALUES now should be the quantiles of the cumulative 
;        distribution function and RESULT will return the values at those 
;        quantiles.
;    LOCATION:  A required float scalar or N_VALUES vector specifying the 
;        location parameter of the GEV distribution.  If an N_VALUES vector, 
;        then each of the N_VALUES elements is matched with the corresponding 
;        element in the VALUES input.
;    SCALE:  A required float scalar or N_VALUES vector specifying the 
;        scale parameter of the GEV distribution.  If an N_VALUES vector, 
;        then each of the N_VALUES elements is matched with the corresponding 
;        element in the VALUES input.
;    SHAPE:  A required float scalar or N_VALUES vector specifying the 
;        shape parameter of the GEV distribution.  If an N_VALUES vector, 
;        then each of the N_VALUES elements is matched with the corresponding 
;        element in the VALUES input.
;
; OUTPUTS:
;    RESULT:  A floating vector of length N_VALUES containing the cumulative 
;        distribution function for each values specified in VALUES.
;
; USES:
;    -
;
; PROCEDURE:
;    This function calculates the cumulative distribution function of the 
;    Generalised Extreme Value distribution analytically.
;
; EXAMPLE:
;    ; Plot the cumulative distribution function for a given set of parameters
;    values = findgen( 501.) / 100.
;    cdf = gev_cdf( values, location=2.7, scale=0.30, shape=-0.11 )
;    plot, values, cdf
;
; LICENSE:
;    This code was written as part of the Extreme Weather Event Real-Time 
;    Attribution Machine (EWERAM) project supported by the New Zealand Ministry 
;    of Business, Innovation and Employment.  The code is free for use under 
;    the terms of the Creative Commons License v2.0 
;    (http://creativecommons.org/licenses/by-nc-sa/2.0/).
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2019-11-06
;    Modified:  DAS, 2022-11-08 (Added INVERT keyword option and capability;  
;        Corrected element-matching between parameters and input values when 
;        shape parameter is zero)
;-

;***********************************************************************

FUNCTION GEV_CDF, $
    VALUES, $
    LOCATION=location, SCALE=scale, SHAPE=shape, $
    INVERT=invert_opt

;***********************************************************************
; Constants and checks

; The number of values for which to calculate the CDF
n_values = n_elements( values )
if n_values eq 0 then stop

; Check the GEV parameters
n_location = n_elements( location )
if max( n_location eq [ 1, n_values ] ) ne 1 then stop
n_scale = n_elements( scale )
if max( n_scale eq [ 1, n_values ] ) ne 1 then stop
if min( scale ) lt 0. then stop
n_shape = n_elements( shape )
if max( n_shape eq [ 1, n_values ] ) ne 1 then stop
; The maximum number of parameters
n_param_max = max( [ n_location, n_scale, n_shape ] )

; The option to calculate the values at the requested quantiles
invert_opt = keyword_set( invert_opt )
if invert_opt eq 1 then begin
  if min( values ) lt 0. then stop
  if max( values ) gt 1. then stop
endif

;***********************************************************************
; Calculate the CDF

; Copy parameters to temporary variables, making all consistent in length
if n_param_max eq 1 then begin
  temp_location = location[0]
  temp_scale = scale[0]
  temp_shape = shape[0]
endif else begin
  if n_location eq 1 then begin
    temp_location = location + fltarr( n_param_max )
  endif else begin
    temp_location = location
  endelse
  if n_scale eq 1 then begin
    temp_scale = scale + fltarr( n_param_max )
  endif else begin
    temp_scale = scale
  endelse
  if n_shape eq 1 then begin
    temp_shape = shape + fltarr( n_param_max )
  endif else begin
    temp_shape = shape
  endelse
endelse

; Initialise the output scalar/vector
result = 0. * values
if n_param_max eq 1 then begin
  id = where( values gt temp_location, n_id )
  if n_id gt 0 then result[id] = 1.
endif else begin
  for i_values = 0, n_values - 1 do begin
    if values[i_values] gt temp_location[i_values] then result[i_values] = 1.
  endfor
endelse

; The case when the shape parameter is zero
id = where( temp_shape eq 0., n_id )
if n_id gt 0 then begin
  ; Calculate the values at the quantiles
  if invert_opt eq 1 then begin
    if n_param_max eq 1 then begin
      result = -alog( -alog( values ) ) * temp_scale + temp_location
    endif else begin
      result[id] = -alog( -alog( values[id] ) ) * temp_scale[id] $
          + temp_location[id]
    endelse
  ; Calculate the CDF
  endif else begin
    if n_param_max eq 1 then begin
      result = exp( -exp( -( values - temp_location ) / temp_scale ) )
    endif else begin
      result[id] = exp( -exp( -( values[id] - temp_location[id] ) $
          / temp_scale[id] ) )
    endelse
  endelse
endif

; The case when the shape parameter is not zero
id = where( temp_shape ne 0., n_id )
if n_id gt 0 then begin
  ; Calculate the values at the quantiles
  if invert_opt eq 1 then begin
    ; Calculate the values at the quantiles
    if n_param_max eq 1 then begin
      temp = ( -alog( values ) ) ^ ( -temp_shape )
      result = ( temp - 1. ) * temp_scale / temp_shape + temp_location
    endif else begin
      temp = ( -alog( values[id] ) ) ^ ( -temp_shape[id] )
      result[id] = ( temp - 1. ) * temp_scale[id] / temp_shape[id] $
          + temp_location[id]
    endelse
  ; Calculuate the CDF
  endif else begin
    ; If we have legal parameter combinations
    if n_param_max eq 1 then begin
      temp = 1. + temp_shape * ( values - temp_location ) / temp_scale
    endif else begin
      temp = 1. $
          + temp_shape[id] * ( values[id] - temp_location[id] ) / temp_scale[id]
    endelse
    id_good = where( temp gt 0, n_id_good )
    if n_id_good gt 0 then begin
      ; Calculate the CDF
      if n_param_max eq 1 then begin
        result[id_good] = exp( -( temp[id_good] ) ^ ( -1. / temp_shape ) )
      endif else begin
        result[id[id_good]] $
            = exp( -( temp[id_good] ) ^ ( -1. / temp_shape[id[id_good]] ) )
      endelse
    endif
  endelse
endif

;***********************************************************************
; The end

return, result
END
