;+
; NAME:
;    geo_mid
;
; PURPOSE:
;    This function returns the coordinates of the geographical point midway 
;    between the two input points.
;
; CATEGORY:
;    Geographical
;
; CALLING SEQUENCE:
;    result = geo_mid( lon_1, lat_1, lon_2, lat_2 )
;
; INPUTS:
;    LON_1:  The scalar longitude, of type floating point, of the first point.  
;        East is positive.
;    LAT_1:  The scalar latitude, of type floating point, of the first point.  
;        North is positive.
;    LON_2:  The scalar longitude, of type floating point, of the second 
;        point.  East is positive.
;    LAT_2:  The scalar latitude, of type floating point, of the second point.  
;        North is positive.
;    FRACTION
;
; KEYWORD PARAMETERS:
;    FRACTION:  An optional vector float specifying how far along the transect 
;        from the first point to the second point to select the output points.  
;        In units of fractional distance from the first point ("0") to the 
;        second point ("1").  The default value is 0.5, i.e. the mid point.
;
; OUTPUTS:
;    RESULT:  Returns a vector of [ longitude, latitude ] coordinates of the 
;        point midway between Points 1 and 2.
;
; USES:
;    constants.pro
;
; PROCEDURE:
;    The function uses double precision floating point arithmetic to calculate 
;    values.
;
; EXAMPLE:
;    Calculate the midpoint between Ottawa, ON (45.25N, 75.43W) and Victoria, 
;    BC (48.25N, 123.22W).
;      result = geo_mid( -75.43, 45.25, -123.22, 48.25 )
;
; MODIFICATION HISTORY:
;    Written by:   Daithi A. Stone (stoned@uvic.ca), 2000-07-24.
;    Modified:  DAS, 2002-08-12 (complies with CONSTANTS.pro revision)
;    Modified:  DAS (dastone@runbox.com), 2024-08-01 (Updated formating; Added 
;        FRACTION keyword input)
;-

;***********************************************************************

FUNCTION GEO_MID, $
    LON_1, LAT_1, $
    LON_2, LAT_2, $
    FRACTION=fraction

;***********************************************************************
; Define Constants

; Degrees in a circle, degrees to radians conversion coefficient
constants, ndegree=n_degree, degrad=degrad

; The default value for the fraction of the distance along the transect
; (the mid point)
if n_elements( fraction ) eq 0 then fraction = 0.5
n_fraction = n_elements( fraction )

; Check for opposing points
oppose = 0
if lat_1 eq -lat_2 then begin
  if lon_1 eq lon_2 + n_degree / 2 then oppose = 1
  if lon_1 eq lon_2 - n_degree / 2 then oppose = 1
endif

;***********************************************************************
; Calculations

; Calculate for non-opposing points
if oppose eq 0 then begin
  ; Convert to Cartesian coordinates
  x_1 = cos( lon_1 * degrad ) * cos( lat_1 * degrad )
  y_1 = sin( lon_1 * degrad ) * cos( lat_1 * degrad )
  z_1 = sin( lat_1 * degrad )
  x_2 = cos( lon_2 * degrad ) * cos( lat_2 * degrad )
  y_2 = sin( lon_2 * degrad ) * cos( lat_2 * degrad )
  z_2 = sin( lat_2 * degrad )
  ; Calculate point along the transect
  x_mid = x_1 + ( x_2 - x_1 ) * fraction
  y_mid = y_1 + ( y_2 - y_1 ) * fraction
  z_mid = z_1 + ( z_2 - z_1 ) * fraction
  ; Convert to point on spherical coordinates
  lon_mid = atan( y_mid / x_mid ) / degrad
  id = where( x_mid lt 0, n_id )
  if n_id gt 0 then lon_mid[id] = lon_mid[id] - n_degree / 2.
  id = where( lon_mid lt 0, n_id )
  if n_id gt 0 then lon_mid[id] = lon_mid[id] + n_degree
  lat_mid = asin( z_mid / sqrt( x_mid^2 + y_mid^2 + z_mid^2 ) ) / degrad
endif

; Calculate for opposing points
if oppose eq 1 then begin
  ; Returns the Eastern Hemisphere equatorial intercept
  lat_mid = 0.
  lon_mid = lon_1 + ( lon_2 - lon_1 ) * fraction
  id = where( lon_mid lt 0, n_id )
  if n_id gt 0 then lon_mid[id] = lon_mid[id] + n_degree
  id = where( lon_mid gt n_degree / 2., n_id )
  if n_id gt 0 then lon_mid[id] = lon_mid[id] - n_degree / 2.
endif

; If inputs used negative longitude format for Western Hemisphere then adopt in 
; output
if ( lon_1 lt 0 ) or ( lon_2 lt 0 ) then begin
  id = where( lon_mid gt n_degree / 2., n_id )
  if n_id gt 0 then lon_mid[id] = lon_mid[id] - n_degree
endif
; Send to output
pos = transpose( [ [ lon_mid ], [ lat_mid ] ] )

;***********************************************************************
; The End

return, pos
END
