;+
; NAME:
;    gamma_pdf
;
; PURPOSE:
;    This function returns the probability density function of the gamma 
;    distribution for a given set of parameters.
;
; CATEGORY:
;    Statistics
;
; CALLING SEQUENCE:
;    results = gamma_pdf( constants, values )
;
; INPUTS:
;    CONSTANTS:  A required two-element float vector specifying the shape 
;        paramenter and the inverse scale parameter respectively.
;    VALUES:  A required float vector of length N_VALUES specifying the values 
;        at which to calculate the probability density function.
;
; KEYWORD PARAMETERS:
;    CDF:  If set, then the function instead returns the cumulative 
;        distribution function.  The default is the probability distribution 
;        function.
;    MEAN:  Returns the mean of the probability distribution.
;    MODE:  Returns the mode of the probability distribution.
;    SKEWNESS:  Returns the skewness of the probability distribution.
;    VARIANCE:  Returns the variance of the probability distribution.
;
; OUTPUTS:
;    RESULT:  A floating vector of length N_VALUES containing the probability 
;        density function for each values specified in VALUES.
;    MEAN, MODE, SKEWNESS, VARIANCE
;
; USES:
;    -
;
; PROCEDURE:
;    This function calculates the probability density function of the gamma 
;    distribution analytically, as described at 
;    https://en.wikipedia.org/wiki/Gamma_distribution.
;
; EXAMPLE:
;    ; Plot the probability distribution function for the [5,1] parameter set.
;    values = ( findgen( 201.) + 0.5 ) / 10.
;    plot, values, gamma_pdf( [5.0,1.0], values )
;
; LICENSE:
;    The code is free for use under the terms of the Creative Commons License 
;    v2.0 (http://creativecommons.org/licenses/by-nc-sa/2.0/).
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2021-06-16
;-

;***********************************************************************

FUNCTION GAMMA_PDF, $
    CONSTANTS, $
    VALUES, $
    MEAN=property_mean, MODE=property_mode, SKEWNESS=property_skewness, $
      VARIANCE=property_variance, $
    CDF=cdf_opt

;***********************************************************************
; Constants and checks

; Confirm input of constants
if n_elements( constants ) ne 2 then stop
if min( constants ) le 0 then stop
; Confirm input of locations
if n_elements( values ) eq 0 then stop
if min( values ) le 0 then stop

;***********************************************************************
; Calculate the distribution

; If we want the probability density function
if not( keyword_set( cdf_opt ) ) then begin
  result = ( constants[1] ^ constants[0] ) * ( values ^ ( constants[0] - 1 ) ) $
      * exp( -constants[1] * values ) / gamma( constants[0] )
; If we want the cumulative density function
endif else begin
  ; It seems that IDL's igamma() does not need normalisation by gamma()
  ;result = igamma( constants[0], constants[1] * values ) $
  ;    / gamma( constants[0] )
  result = igamma( constants[0], constants[1] * values )
endelse

;***********************************************************************
; Calculate properties

; The mean
property_mean = constants[0] / ( 1. * constants[1] )
; The mode
if constants[0] ge 1 then begin
  property_mode = ( constants[0] - 1. ) /constants[1]
endif else begin
  property_mode = !values.f_nan
endelse
; The variance
property_variance = constants[0] / ( constants[1] ^ 2. )
; The skewness
property_skewness = 2. / sqrt( constants[0] )

;***********************************************************************
; The end

return, result
END
