;+
; NAME:
;    first_diff
;
; PURPOSE:
;    This function returns the first difference vector of the input.
;
; CATEGORY:
;    Time Series Analysis
;
; CALLING SEQUENCE:
;    result = first_diff( DATA )
;
; INPUTS:
;    DATA:  A vector of length N_DATA of type integer or floating point.  It 
;        can also be an array of size N_DATA,N_VEC where the first difference 
;        calculated on the N_DATA values separately for each of the N_VEC 
;        vectors.
;
; KEYWORD PARAMETERS:
;    APPROX_END:  If set then the results at the ends that cannot be calculated 
;        with the specified method are approximated with another method.  For 
;        instance, if BACKWARD=1, then RESULT[0] is calculated using the 
;        forward difference (so in this example RESULT[0]=RESULT[1]).  The 
;        default is to return NaN.  WRAF=1 overrides this option. 
;    BACKWARD:  If set then it forces calculation of the backward difference.  
;        This is the default.
;    CENTRED:  If set then it forces calculation of the centred difference.  
;        The default is the backward difference.
;    FORWARD:  If set then it forces calculation of the forward difference.  
;        The default is the backward difference.
;    WRAP:  If set then the results at the end(s) of the vector are calculated 
;        by wrapping around the input from the other end.  The default is to 
;        return NaN for end values that cannot be calculated.
;
; OUTPUTS:
;    RESULT:  Returns the first difference of the input vector/array.  If the 
;        input is of type integer, result will be of type float.
;
; PROCEDURE:
;    The function calculates the first difference of the values in the input 
;    vector.
;
; EXAMPLE:
;    Define a vector.
;      x = [ 1, 2, 4, 8 ]
;    Calculate the forward first differenc of x.
;      result = first_diff( x, forward=1 )
;    This should return:  [ 1., 2., 4., NaN ]
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2000-06-27
;    Modified:  DAS, 2000-07-06 (Removed length.pro)
;    Modified:  DAS, 2000-07-10 (Removed for loops)
;    Modified:  DAS, 2024-01-03 (Updated documentation;  Updated documentation; 
;        Added APPROX_END, WRAP keyword options;  Switched default to WRAP=0 
;        from equivalent of WRAP=1)
;-

;***********************************************************************

FUNCTION FIRST_DIFF, $
    DATA, $
    APPROX_END=approx_end_opt, $
    BACKWARD=backward_opt, CENTRED=centred_opt, FORWARD=forward_opt, $
    WRAP=wrap_opt

;***********************************************************************
; Variables and options

; Input data dimension
if n_elements( data ) eq 0 then stop
n_data = ( size( data, dimension=1 ) )[0]

; Output
f_diff = !values.f_nan * ( 1. * data )

; Difference type
if keyword_set( backward_opt ) then begin
  forward_opt = 0
  centred_opt = 0
endif else if keyword_set( forward_opt ) then begin
  backward_opt = 0
  centred_opt = 0
endif else if keyword_set( centred_opt ) then begin
  backward_opt = 0
  forward_opt = 0
endif else begin
  backward_opt = 1
  forward_opt = 0
  centred_opt = 0
endelse

; Option to wrap vector during calculation
wrap_opt = keyword_set( wrap_opt )
if keyword_set( wrap_opt ) then begin
  approx_end_opt = 0
endif else begin
  approx_end_opt = keyword_set( approx_end_opt )
endelse

;***********************************************************************
; Calculate first difference

; Backward difference
if backward_opt then begin
  ; Calculate result for feasible inputs
  f_diff[1:n_data-1,*] = data[1:n_data-1,*] - data[0:n_data-2,*]
  ; Approximate option for the first element
  if approx_end_opt eq 1 then f_diff[0,*] = f_diff[1,*]
  ; Wrap option for the first element
  if wrap_opt eq 1 then f_diff[0,*] = data[0,*] - data[n_data-1,*]
endif

; Forward difference
if forward_opt then begin
  ; Calculate result for feasible inputs
  f_diff[0:n_data-2,*] = data[1:n_data-1,*] - data[0:n_data-2,*]
  ; Approximate option for the last element
  if approx_end_opt eq 1 then f_diff[n_data-1,*] = f_diff[n_data-2,*]
  ; Wrap option for the last element
  if wrap_opt eq 1 then f_diff[n_data-1,*] = data[0,*] - data[n_data-1,*]
endif

; Centred difference
if centred_opt then begin
  ; Calculate result for feasible inputs
  f_diff[1:n_data-2,*] = ( data[2:n_data-1,*] - data[0:n_data-3,*] ) / 2.
  ; Approximate option for the end elements
  if approx_end_opt eq 1 then begin
    f_diff[0,*] = data[1,*] - data[0,*]
    f_diff[n_data-1,*] = data[n_data-1,*] - data[n_data-2,*]
  endif
  ; Wrap option for the end elements
  if wrap_opt eq 1 then begin
    f_diff[0,*] = ( data[1,*] - data[n_data-1,*] ) / 2.
    f_diff[n_data-1,*] = ( data[0,*] - data[n_data-2,*] ) / 2.
  endif
endif

;***********************************************************************
;The End

return, f_diff
END
