;+
; NAME:
;    FILTER
;
; PURPOSE:
;    This function returns a smoothed version of the input vector.
;
; CATEGORY:
;    Time Series Analysis
;
; CALLING SEQUENCE:
;    result = filter( vector, [width], [window] )
;
; INPUTS:
;    VECTOR:  An vector of type floating point and length N_VECTOR containing 
;        the input data.
;    WIDTH:  The optional width, of type integer, of the smoothing window.  
;        Required if not input via a keyword input.
;    WINDOW:  An optional string containing the name of the smoothing window to 
;        use.  Options are:
;        * 'boxcar':  A boxcar filter.
;        * 'gaussian':  A Gaussian filter.
;        * 'hanning':  A Hanning filter.
;        * 'triangle':  A triangle filter.  
;        Additional non-smoothing options include:
;        * 'max':  Takes the maximum value within the window.
;        * 'max@<n>':  Takes the '<n>'th-largest value within the window.  
;          'max#1' is hence the same as 'max'.
;        * 'max#<n>':  Takes the mean of the '<n>'th-largest values within the 
;          window.  'max#1' is hence the same as 'max'.
;        * 'median':  Takes the median value within the window.
;        * 'min':  Takes the minimum value within the window.
;        * 'min@<n>':  Takes the '<n>'th-smallest value within the window.  
;          'min#1' is hence the same as 'min'.
;        * 'min#<n>':  Takes the mean of the '<n>'th-smallest values within the 
;          window.  'min#1' is hence the same as 'min'.
;        The default is 'boxcar'.
;
; KEYWORD PARAMETERS:
;    BOXCAR:  If set then the smoothing window is set to a boxcar filter.  
;        If WIDTH is not input, then the integer value input here sets the 
;        width of the boxcar filter.  Retained for back compatibility.
;    EDGE_TRUNCATE:  Set this keyword to apply the smoothing to all points.  If 
;        the neighbourhood around a point includes a point outside the array, 
;        the nearest edge point is used to compute the smoothed result.  If 
;        EDGE_TRUNCATE is not set, the points near the end are replaced with 
;        NaNs.
;    FILTER:  A vector containing the filter window to use.  This overrides the 
;        window requested in the WINDOW input.  This also returns the filter 
;        after use.
;    FRAC_WINDOW_MIN:  An optional floating point fraction (between 0 and 1) 
;        specifiying the minimum weighted fraction of the filter window 
;        required to have data in order to calculate a value.  Only applies 
;        when NAN is set.
;    LOCATION:  If set, then this returns a vector of length N_VECTOR recording 
;        the location within the window of the maximum or minimum for each of 
;        the N_VECTOR steps in VECTOR when WINDOW is set to the 'max' or 'min' 
;        values.
;    NAN:  Set this keyword to ignore NaN values in the input array, provided 
;        there is at least one defined value nearby.  The default is to return 
;        NaNs wherever they occur.
;    NO_NAN:  Obsolete version of NAN keyword retained for compatibility but no 
;        longer used.
;    START_INDEX:  The location of the centre of the window for the first 
;        averaged output value, in units of Vector indices.  Values must be 
;        greater than 0.  The default is 0.
;    STEP:  An integer defining the step size for window translation, in units 
;        of VECTOR indices.  The default is 1.
;    TRIANGLE:  If set then the smoothing window is set to a triangle filter.  
;        If WIDTH is not input, then the integer value input here sets the 
;        width of the triangle filter.  Retained for back compatibility.
;    WRAP_EDGES:  If set, the vector is treated as being cyclic and the ends 
;        are joined together when smoothing.
;
; OUTPUTS:
;    RESULT:  Returns the smoothed version of Vector.
;    LOCATION
;
; USES:
;    dimension.pro
;    filter_window.pro
;    plus.pro
;
; PROCEDURE:
;    This function manually convolves the input vector with the filter.
;
; EXAMPLE:
;    Create a vector of daily data and a sinusoid for a year.
;      x = randomn( seed, 365 ) + sin( 6.28 * findgen( 365 ) / 365. )
;    Smooth x with a boxcar filter of 7 days, wrapping the edges together.
;      result = filter( x, 7, 'boxcar', /wrap_edges )
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone, 2000-06-28.
;    Modified:  DAS, 2000-07-06 (removed LENGTH.pro, added DIMENSION.pro).
;    Modified:  DAS, 2001-01-24 (added reform(vector) line).
;    Modified:  DAS, 2003-11-19 (added Window input and FILTER keyword, edited 
;        style).
;    Modified:  DAS, 2004-07-14 (allowed input of complex vectors)
;    Modified:  DAS, 2005-08-05 (replaced sum.pro use with IDL's total; added 
;        NAN keyword for consistency with other routines; removed functionality 
;        of NO_NAN keyword; removed constants.pro use; streamlined coding)
;    Modified:  DAS, 2005-10-21 (removed requirement of odd WIDTH;  allowed 
;        operation with WIDTH=1)
;    Modified:  DAS, 2009-01-05 (added and implemented START_INDEX and STEP 
;        keywords)
;    Modified:  DAS, 2019-09-13 (Added FRAC_WINDOW_MIN keyword input;  Updated 
;        documentation format)
;    Modified:  DAS, 2020-05-20 (Switched to faster usage of IDL convol.pro;  
;        Added OLD keyword option to use old convol.pro-less method)
;    Modified:  DAS, 2023-10-02 (Added 'max', 'max#<n>', 'median', and 'min' 
;        pseudo-windows)
;    Modified:  DAS, 2025-10-01 (Improved documentation.  Updated formatting.  
;        Added WINDOW='max@<n>', 'mim@<n>', amd 'min#<n>' pseudo-windows.  
;        Fixed bug with OLD default.  Fixed bugs with NaN values with new use 
;        of convol function.  Added LOCATION keyword output.  Permitted 
;        vectors with long-integer lengths.)
;-

;***********************************************************************

FUNCTION FILTER, $
    VECTOR, $
    WIDTH, $
    WINDOW, $
    BOXCAR=boxcar, TRIANGLE=triangle, $
    FILTER=filt, $
    FRAC_WINDOW_MIN=frac_window_min, $
    START_INDEX=start_index, STEP=step, $
    WRAP_EDGES=wrap_edges_opt, EDGE_TRUNCATE=edge_truncate_opt, $
    LOCATION=location, $
    NAN=nan_opt, NO_NAN=no_nan_opt, OLD=old_opt

;***********************************************************************
; Constants, Variables, and Options

; Load absolute constants
nan = !values.f_nan

; Certain operations only work with the rote code
if n_elements( old_opt ) eq 0 then begin
  old_opt = max( window eq [ 'max', 'median', 'min' ] ) $
      or ( strpos( window, 'max@' ) eq 0 ) $
      or ( strpos( window, 'max#' ) eq 0 ) $
      or ( strpos( window, 'min@' ) eq 0 ) or ( strpos( window, 'min#' ) eq 0 )
endif

; Determine n-th highest to record ('max@' operation)
if strpos( window, 'max@' ) eq 0 then begin
  temp = strsplit( window, '@', extract=1 )
  window_use = 'max@'
  window_num = fix( temp[1] )
; Determine number of maximum values to average ('max#' operation)
endif else if strpos( window, 'max#' ) eq 0 then begin
  temp = strsplit( window, '#', extract=1 )
  window_use = 'max#'
  window_num = fix( temp[1] )
; Determine n-th lowest to record ('max@' operation)
endif else if strpos( window, 'min@' ) eq 0 then begin
  temp = strsplit( window, '@', extract=1 )
  window_use = 'min@'
  window_num = fix( temp[1] )
; Determine number of minimum values to average ('min#' operation)
endif else if strpos( window, 'min#' ) eq 0 then begin
  temp = strsplit( window, '#', extract=1 )
  window_use = 'min#'
  window_num = fix( temp[1] )
; If the boxcar keyword option is set
endif else if keyword_set( boxcar ) then begin
  ; Adopt the window set if input
  if keyword_set( window ) then begin
    window_use = window
  ; Otherwise adopt the number input in the boxcar keyword input
  endif else begin
    window_use = boxcar
  endelse
; If the triangle keyword option is set
endif else if keyword_set( triangle ) then begin
  ; Adopt the window set if input
  if keyword_set( window ) then begin
    window_use = window
  ; Otherwise adopt the number input in the triangle keyword input
  endif else begin
    window_use = triangle
  endelse
; Otherwise just copy the window label
endif else begin
  window_use = window
endelse

; Load smoothing window
temp = [ 'max', 'max@', 'max#', 'median', 'min', 'min@', 'min#' ]
if not( keyword_set( filt ) ) or ( max( window_use eq temp ) eq 0 ) then begin
  filt = filter_window( width, temp_window )
endif

; Edge-handling options
wrap_edges_opt = keyword_set( wrap_edges_opt ) 
edge_truncate_opt = keyword_set( edge_truncate_opt )

; NaN handling option
if keyword_set( nan_opt ) then begin
  nan_opt = 1
endif else begin
  nan_opt = 0
endelse
; The default minimum weighted data requirement
if not( keyword_set( frac_window_min ) ) then begin
  if keyword_set( old_opt ) then begin
    frac_window_min = 0.
  endif else begin
    ; There seem to be numerical issues with convol.pro, so this is numerically 
    ; equivalent to zero
    frac_window_min = 0.1 / width
  endelse
endif

; Half-length of filter
width_half_0 = ( width - 1 ) / 2
width_half_1 = width / 2

; Length of vector
vector = reform( vector )
n_vector = n_elements( vector )
if n_vector eq 0 then stop

; The default position of the first application of the smoothing window
if n_elements( start_index ) eq 0 then start_index = 0
if start_index lt 0 then stop
; The default window stepping
if not( keyword_set( step ) ) then step = 1

; Output vector
n_out = n_vector - start_index
if step ne 1 then begin
  n_out = ( n_out + start_index - start_index / width * width ) / step
endif
out_vec = nan * float( vector[0:n_out-1] )
; The location output vector
if keyword_set( location ) then location = -1 + intarr( n_out )

;***********************************************************************
; Smooth Vector

; The old code
if keyword_set( old_opt ) then begin

  ; Expanded version of Vector.
  ; This needs to be done to handle the edges.
  new_vec = vector
  ; If EDGE_TRUNCATE is set
  if edge_truncate_opt eq 1 then begin
    ; Pad ends with edge values
    if width_half_0 gt 0 then begin
      new_vec = [ 0*fltarr(width_half_0)+vector[0], new_vec ]
    endif
    if width_half_1 gt 0 then begin
      new_vec = [ new_vec, 0*fltarr(width_half_1)+vector[n_vector-1] ]
    endif
  ; If WRAP_EDGES is set
  endif else if wrap_edges_opt eq 1 then begin
    ; Pad ends with values from opposite end
    if width_half_0 gt 0 then begin
      new_vec = [ vector[n_vector-width_half_0:n_vector-1], new_vec ]
    endif
    if width_half_1 gt 0 then new_vec = [ new_vec, vector[0:width_half_1-1] ]
  ; Default method for handling edges
  endif else begin
    ; Pad ends with NaNs
    if width_half_0 gt 0 then new_vec = [ 0*fltarr(width_half_0)+nan, vector ]
    if width_half_1 gt 0 then new_vec = [ new_vec, 0*fltarr(width_half_1)+nan ]
  endelse

  ; Identify where the filter is non-zero
  if keyword_set( filt ) then begin
    id_filt = where( filt gt 0., n_id_filt )
    if n_id_filt eq 0 then stop
    filt_use = filt[id_filt]
    filt_use = filt_use / total( filt_use )
  endif else begin
    n_id_filt = width
  endelse
  ; Iterate through all points
  for i_out = 0l, n_out - 1l do begin
    ; Extract windowed segment
    ; (We only need data for non-zero window weightings)
    temp = new_vec[start_index+i_out*step:start_index+i_out*step+width-1]
    if keyword_set( filt ) and ( n_id_filt ne width ) then temp = temp[id_filt]
    ; Determine how much of the data in this segment we have available
    temp_frac = total( finite( temp ), integer=1 )
    ; Proceed if we have sufficient data
    if nan_opt eq 1 then begin
      flag_good = ( temp_frac ge n_id_filt * frac_window_min ) $
          and ( temp_frac gt 0 )
    endif else begin
      flag_good = temp_frac ge n_id_filt
    endelse
    if flag_good eq 1 then begin
      ; Take the maximum value
      if window_use eq 'max' then begin
        out_vec[i_out] = max( temp, nan=nan_opt, id )
        if keyword_set( location ) then location[i_out] = id
      ; Take the n-th higest value or mean of n-th highest values
      endif else if ( window_use eq 'max@' ) or ( window_use eq 'max#' ) $
          then begin
        ; Extract good values
        if ( nan_opt eq 1 ) and ( temp_frac ne n_id_filt ) then begin
          id = where( finite( temp ) eq 1, n_id )
          if n_id gt 0 then temp = temp[id]
        endif else begin
          n_id = n_id_filt
        endelse
        ; Sort values
        id = sort( temp )
        ; Take the n-th highest value
        if window_use eq 'max@' then begin
          out_vec[i_out] = temp[id[n_id-window_num]]
        ; Or take the mean of the n-th highest values
        endif else if window_use eq 'max#' then begin
          out_vec[i_out] = mean( temp[id[[n_id-window_num:n_id-1]]] )
        endif
      ; Take the median value
      endif else if window_use eq 'median' then begin
        if ( nan_opt eq 1 ) and ( temp_frac ne n_id_filt ) then begin
          id = where( finite( temp ) eq 1, n_id )
          if n_id gt 0 then out_vec[i_out] = median( temp[id] )
        endif else begin
          out_vec[i_out] = median( temp )
        endelse
      ; Take the minimum value
      endif else if window_use eq 'min' then begin
        out_vec[i_out] = min( temp, nan=nan_opt, id )
        if keyword_set( location ) then location[i_out] = id
      ; Take the n-th lowest value or mean of n-th lowest values
      endif else if ( window_use eq 'min@' ) or ( window_use eq 'min#' ) $
          then begin
        ; Extract good values
        if ( nan_opt eq 1 ) and ( temp_frac ne n_id_filt ) then begin
          id = where( finite( temp ) eq 1, n_id )
          if n_id gt 0 then temp = temp[id]
        endif else begin
          n_id = n_id_filt
        endelse
        ; Sort values
        id = sort( temp )
        ; Take the n-th lowest value
        if window_use eq 'min@' then begin
          out_vec[i_out] = temp[id[window_num-1]]
        ; Or take the mean of the n-th lowest values
        endif else if window_use eq 'min#' then begin
          out_vec[i_out] = mean( temp[id[[0:window_num-1]]] )
        endif
      ; Otherwise take the convolution with the filter
      endif else begin
        if nan_opt eq 1 then begin
          out_vec[i_out] = total( filt_use * temp, nan=1 ) $
              / total( filt_use * finite( temp ) )
        endif else begin
          out_vec[i_out] = total( filt_use * temp )
        endelse
      endelse
    endif
  endfor

; The new convol.pro-dependent code
endif else begin

  ; If we are counting NaNs and are not applying edge truncation
  if ( nan_opt eq 1 ) and ( edge_truncate_opt eq 0 ) then begin
    ; Pad both ends of the vector
    temp_vector = [ nan * fltarr( width_half_0 ), vector, $
        nan * fltarr( width_half_1 ) ]
  ; If we have an even-sized filter
  endif else if width_half_1 ne width_half_0 then begin
    ; Pad the end with a NaN
    temp_vector = [ vector, nan ]
  ; If we have an odd-sized filter
  endif else begin
    ; No padding is needed
    temp_vector = vector
  endelse
  ; Convolve the filter and vector
  out_vec = convol( temp_vector, filt, edge_truncate=edge_truncate_opt, $
      edge_wrap=wrap_edges_opt, normalize=1, center=1, nan=nan_opt )
  ; Remove padded values
  if ( nan_opt eq 1 ) and ( edge_truncate_opt eq 0 ) then begin
    out_vec = out_vec[width_half_0:width_half_0+n_vector-1]
  endif else if width_half_1 ne width_half_0 then begin
    out_vec = out_vec[1:n_vector]
  endif
  ; Deal with missing data
  if ( ( nan_opt eq 1 ) and ( frac_window_min ne 0. ) ) $
      or ( edge_truncate_opt eq 0 ) then begin
    ; Determine the number of values being used in each convolution
    out_vec_finite = convol( float( finite( temp_vector ) ), filt, center=1, $
          edge_wrap=wrap_edges_opt, edge_truncate=edge_truncate_opt )
    ; Remove padded values
    if ( nan_opt eq 1 ) and ( edge_truncate_opt eq 0 ) then begin
      out_vec_finite = out_vec_finite[width_half_0:width_half_0+n_vector-1]
    endif else if width_half_1 ne width_half_0 then begin
      out_vec_finite = out_vec_finite[1:n_vector]
    endif
    ; Set values with insufficient samples to NaN
    id = where( out_vec_finite lt frac_window_min, n_id )
    if n_id gt 0 then out_vec[id] = nan
  endif

endelse

;***********************************************************************
; The End

return, out_vec
END
