;+
; NAME:
;    evattrib_calc
;
; PURPOSE:
;    This function calculates various measures of changes in extreme values 
;    between two samples of data representing factual and counterfactual 
;    scenarios.
;
; CATEGORY:
;    Statistics
;
; CALLING SEQUENCE:
;    result = evattrib_calc( metric=metric, data_factual=data_factual, $
;        data_cfactual=data_cfactual )
;
; KEYWORD PARAMETERS:
;    DATA_CFACTUAL:  A required floating point array containing the data used 
;        to estimate the probability, likelihood, or return value for the 
;        counterfactual scenario.
;    DATA_FACTUAL:  A required floating point array containing the data used to 
;        estimate the probability, likelihood, or return value for the factual 
;        scenario.
;    EVENT_DELTA_QUANTILE:  A floating point scalar specifying the half-width 
;        of the band around the quantile corresponding to the event to be used 
;        when METRIC is 'likelihood risk ratio' and the 'empirical' method is 
;        used.  Required for that case.
;    EVENT_QUANTILE:  A floating point scalar specifying the quantile of the 
;        event for estimating shifts in event magnitude.  Required when METRIC 
;        is 'magnitude shift'.
;    EVENT_VALUE:  A floating point scalar specifying the magnitude of the 
;        event for estimating risk ratios.  Required when METRIC is 'exceedance 
;        risk ratio' or 'likelihood risk ratio'.
;    GPD_QUANTILE:  An optional floating point scalar specifying the quantile 
;        of the input data (of either DATA_FACTUAL or DATA_CFACTUAL) to use as 
;        as the threshold for inclusion in estimation of the Generalised Pareto 
;        Distribution model.  The default is 0.9.
;    KERNEL_BANDWIDTH:  If set then the BANDWIDTH keyword option is set when 
;        calling the pdf.pro kernel estimator.
;    METHOD_CFACTUAL:  Same as METHOD_FACTUAL except for the counterfactual 
;        scenario.  The default is to be the same as METHOD_FACTUAL.
;    METHOD_FACTUAL:  A string vector specifying the method to use for 
;        estimating the probability, likelihood, or return value for the 
;        factual scenario.  Possible values are:
;        * 'empirical':  Estimate directly by counting the exceedance rate, 
;          counting the likelihood within an EVENT_DELTA_QUANTILE range, or 
;          taking the value at the empirical quantile.  Can be used with all 
;          implemented METRIC requests.
;        * 'GPD':  Fit a Generalised Pareto Distribution model to the tail of 
;          the data, and calculate the result from that model.  The GPD model 
;          is fit to data exceeding the GPD_QUANTILE quantile of the input 
;          data;  if this sample is smaller than N_GPD_SAMPLE_MIN then the 
;          request for GPD modelling will be ignored.  Can be used with all 
;          implemented METRIC requests.
;        * 'kernel':  Estimate a semi-empirical probability distribution using 
;          a Gaussian kernel method, and take the probability density 
;          (likelihood) for the specific EVENT_VALUE position.  Can be used 
;          with all implemented METRIC requests.  See pdf.pro for details of 
;          the kernel-based estimator.
;        When multiple methods are specified, the application is in the order 
;        of priority of 'GPD', 'kernel', and 'empirical', depending on whether 
;        requirements are satisfied.  For instance, if both 'GPD' and another 
;        method are specified, then if the sample available for estimating the 
;        GPD model is at least N_GPD_SAMPLE_MIN in size then the GPD method is 
;        used, otherwise the other method will be used.  The default is 
;        ['GPD','empirical'].
;    METRIC:  A required scalar string specific the measure to calculate.  
;        Supported values are:
;        * 'exceedance risk ratio':  Calculates the risk ratio of the 
;          probabilities of exceeding EVENT_VALUE for the two scenarios.  This 
;          calculation is for positive exceedances;  for negative exceedances 
;          simply multiply DATA_FACTUAL, DATA_CFACTUAL, and EVENT_VALUE by -1 
;          before input.
;        * 'likelihood risk ratio':  Calculates the risk ratio of the 
;          likelihoods of EVENT_VALUE occurring for the two scenarios.
;        * 'magnitude shift':  Calculates the shift from counterfactual to 
;          factual return values for the specified EVENT_QUANTILE quantile.
;    N_GPD_SAMPLE_MAX:  An optional integer scalar specifying the minimum 
;        number of datavalues exceeding the GPD_QUANTILE of the input data 
;        (either DATA_FACTUAL or DATA_CFACTUAL) for use in estimating the 
;        Generalised Pareto Distribution.  The default is 30.
;    N_PDF_X:  An optional integer scalar specifying the number of quantiles to 
;        sample when calculating a probability distribution using a 
;        kernel-based method.  The default is 1000.
;    PDF_RANGE:  An optional 2-element vector specifying the range over which 
;        to sample the probability distribution N_PDF_X times using a 
;        kernel-based method.  The default is the range of the data.
;
; OUTPUT:
;    RESULT:  If METRIC equals 'exceedance risk ratio' or 'likelihood risk 
;        ratio' then this returns the natural logarithm of the risk ratio.  If 
;        METRIC equals 'magnitude shift' then this returns the shift in the 
;        return value of a specific return probability.
;
; USES:
;    gpd_cdf.pro
;    gpd_fit.pro
;    gpd_pdf.pro
;    pdf.pro
;    quantile_threshold.pro
;
; PROCEDURE:
;    This function uses various probability distribution estimation methods to 
;    calculate the risk ratio of two exceedance probabilities or of two 
;    likelihoods, or to calculate the shift in two return values.
;
; EXAMPLE:
;    ; Note:  These examples use the same estimation method for both scenarios, 
;    ; but the method does not need to be the same for both scenarios.
;    ; Compile gpd_fit.pro
;    .r gpd_fit
;    ; Construct factual data set based on unit Gaussian distribution and 
;    ; counterfactual data set based on shifted unit Gaussian distribution
;    n_data = 100000l
;    data_factual = randomn( seed, n_data )
;    shift_cfactual = 0.1
;    data_cfactual = randomn( seed, n_data ) - shift_cfactual
;    ; Define event magnitude
;    event_value = 1.7
;    ; Define sampling delta
;    delta = 0.01
;    ; Compare analytic and estimated exceedance risk ratio
;    metric = 'exceedance risk ratio'
;    p_factual_analytic = 1. - gauss_pdf( event_value )
;    p_cfactual_analytic = 1. - gauss_pdf( event_value + shift_cfactual )
;    print, 'Analytic factual probability: ', p_factual_analytic
;    print, 'Analytic counterfactual probability: ', p_cfactual_analytic
;    print, 'Analytic risk ratio: ', p_factual_analytic / p_cfactual_analytic
;    temp_rr = exp( evattrib_calc( event_value=event_value, metric=metric, $
;        data_factual=data_factual, data_cfactual=data_cfactual, $
;        method_factual='empirical', method_cfactual='empirical' ) )
;    print, 'Empirical risk ratio: ', temp_rr
;    temp_rr = exp( evattrib_calc( event_value=event_value, metric=metric, $
;        data_factual=data_factual, data_cfactual=data_cfactual, $
;        method_factual='GPD', method_cfactual='GPD' ) )
;    print, 'GPD model risk ratio: ', temp_rr
;    ; Compare analytic and estimated likelihood risk ratio
;    metric = 'likelihood risk ratio'
;    p_factual_analytic = gauss_pdf( event_value + delta ) $
;        - gauss_pdf( event_value - delta )
;    p_cfactual_analytic = gauss_pdf( event_value + shift_cfactual + delta ) $
;        - gauss_pdf( event_value + shift_cfactual - delta )
;    print, 'Analytic factual probability: ', p_factual_analytic
;    print, 'Analytic counterfactual probability: ', p_cfactual_analytic
;    print, 'Analytic risk ratio: ', p_factual_analytic / p_cfactual_analytic
;    temp_rr = exp( evattrib_calc( event_value=event_value, metric=metric, $
;        data_factual=data_factual, data_cfactual=data_cfactual, $
;        method_factual='empirical', method_cfactual='empirical', $
;        event_delta_quantile=delta ) )
;    print, 'Empirical risk ratio: ', temp_rr
;    temp_rr = exp( evattrib_calc( event_value=event_value, metric=metric, $
;        data_factual=data_factual, data_cfactual=data_cfactual, $
;        method_factual='kernel', method_cfactual='kernel' ) )
;    print, 'Semi-empirical kernel estimator risk ratio: ', temp_rr
;    temp_rr = exp( evattrib_calc( event_value=event_value, metric=metric, $
;        data_factual=data_factual, data_cfactual=data_cfactual, $
;        method_factual='GPD', method_cfactual='GPD' ) )
;    print, 'GPD model risk ratio: ', temp_rr
;    ; Compare analytic and estimated magnitude shift
;    metric = 'magnitude shift'
;    print, 'Defined shift: ', shift_cfactual
;    event_quantile = gauss_pdf( event_value )
;    temp_shift = evattrib_calc( event_quantile=event_quantile, metric=metric, $
;        data_factual=data_factual, data_cfactual=data_cfactual, $
;        method_factual='empirical', method_cfactual='empirical' )
;    print, 'Empirical shift: ', temp_shift
;    temp_shift = evattrib_calc( event_quantile=event_quantile, metric=metric, $
;        data_factual=data_factual, data_cfactual=data_cfactual, $
;        method_factual='GPD', method_cfactual='GPD' )
;    print, 'GPD model shift: ', temp_shift
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2020-05-14
;    Modified:  DAS, 2020-05-25 (Clarified some variable names)
;    Modified:  DAS, 2020-06-22 (Introduced assumption that likelihood=0 when 
;        outside of sampled distribution range;  Added PDF_RANGE keyword input)
;    Modified:  DAS, 2021-05-19 (Added capability to use 'kernel' method for 
;        the 'exceedance risk ratio' and 'magnitude shift' metrics;  Added 
;        KERNEL_BANDWIDTH keyword option)
;-

;***********************************************************************

FUNCTION EVATTRIB_CALC, $
    DATA_FACTUAL=data_factual, DATA_CFACTUAL=data_cfactual, $
    EVENT_VALUE=event_value, EVENT_QUANTILE=event_quantile, $
      EVENT_DELTA_QUANTILE=event_quantile_delta, $
    GPD_QUANTILE=gpd_quantile, N_GPD_SAMPLE_MIN=n_gpd_sample_min, $
    KERNEL_BANDWIDTH=kernel_bandwidth_opt, $
    METRIC=metric, $
    METHOD_FACTUAL=method_factual, METHOD_CFACTUAL=method_cfactual, $
    N_PDF_X=n_pdf_x, PDF_RANGE=pdf_range

;***********************************************************************
; Constants and checks

; Confirm a metric is requested
if n_elements( metric ) ne 1 then stop

; The minimum exceedance number for non-parametric sampling
if not( keyword_set( n_gpd_sample_min ) ) then n_gpd_sample_min = 30
; The quantile cutoff for the GPD
if not( keyword_set( gpd_quantile ) ) then gpd_quantile = 0.90
; The PDF sample resolution (number)
if not( keyword_set( n_pdf_x ) ) then n_pdf_x = 1000

; Default method specifications
if not( keyword_set( method_factual ) ) then begin
  method_factual = [ 'GPD', 'empirical' ]
endif
if not( keyword_set( method_cfactual ) ) then method_cfactual = method_factual

;***********************************************************************
; Calculate measure of attributable contribution

; Determine sample sizes
n_factual_sample = n_elements( data_factual )
n_cfactual_sample = n_elements( data_cfactual )

; For risk ratio calculations
if strpos( metric, 'risk ratio' ) ge 0 then begin

  ; Calculate the positive exceedance counts
  if metric eq 'exceedance risk ratio' then begin

    ; Determine which method to use for the factual scenario.
    method_factual_use = ''
    ; Should it be GPD fitting?
    if max( method_factual eq 'GPD' ) eq 1 then begin
      ; Identify values to use in estimating GPD fit
      gpd_thresh = ( quantile_threshold( data_factual, gpd_quantile ) )[0]
      id_gpd_thresh = where( data_factual ge gpd_thresh[0], n_id_gpd_thresh )
      ; If we have a sufficient sample and the event is within the GPD domain
      if ( n_id_gpd_thresh ge n_gpd_sample_min ) $
          and ( event_value gt gpd_thresh ) then begin
        ; Adopt GPD fitting
        method_factual_use = 'GPD'
      endif
    endif
    ; Should it be counting?
    if ( max( method_factual eq 'empirical' ) eq 1 ) $
        and ( method_factual_use eq '' ) then begin
      method_factual_use = 'empirical'
    endif
    ; Should it be a semi-empirical estimator (kernel-based)?
    if ( max( method_factual eq 'kernel' ) eq 1 ) $
        and ( method_factual_use eq '' ) then begin
      method_factual_use = 'kernel'
    endif
    ; Confirm we have a method selected
    if method_factual_use eq '' then stop
    ; If we are to use the GPD
    if method_factual_use eq 'GPD' then begin
      ; Estimate GPD fit
      temp_param = gpd_fit( data_factual[id_gpd_thresh] - gpd_thresh )
      ; Estimate probability of exceeding the event threshold
      p_factual = gpd_cdf( event_value-gpd_thresh, scale=temp_param[0], $
          shape=temp_param[1] )
      p_factual = alog( $
          1. - ( gpd_quantile + ( 1. - gpd_quantile ) * p_factual ) )
    endif
    ; If we are to use empirical counting
    if method_factual_use eq 'empirical' then begin
      ; Count exceedances
      id = where( data_factual ge event_value, n_factual_count )
      p_factual = alog( n_factual_count ) - alog( n_factual_sample )
    endif
    ; If we are to estimate semi-empirically (kernel-based estimator)
    if method_factual_use eq 'kernel' then begin
      ; Estimate the PDF, using a kernel-based estimator
      pdf, data_factual, noplot=1, xid=temp_x, pdf=temp_pdf, npdf=n_pdf_x, $
          xrange=pdf_range, bandwidth=kernel_bandwidth_opt
      temp_pdf = temp_pdf / total( temp_pdf )
      ; Determine probability of exceeding event value
      id = where( temp_x ge event_value, n_id )
      if n_id eq 0 then begin
        ; Assume this means that we are numerically outside of the range of
        ; the distribution
        p_factual = alog( 0.5 ) - alog( n_factual_sample )
      endif else begin
        p_factual = alog( total( temp_pdf[id] ) )
      endelse
    endif
    ; Confirm we have an estimate
    if n_elements( p_factual ) ne 1 then stop

    ; Determine which method to use for the counterfactual scenario.
    method_cfactual_use = ''
    ; Should it be GPD fitting?
    if max( method_cfactual eq 'GPD' ) eq 1 then begin
      ; Identify values to use in estimating GPD fit
      gpd_thresh = ( quantile_threshold( data_cfactual, gpd_quantile ) )[0]
      id_gpd_thresh = where( data_cfactual ge gpd_thresh[0], n_id_gpd_thresh )
      ; If we have a sufficient sample and the event is within the GPD domain
      if ( n_id_gpd_thresh ge n_gpd_sample_min ) $
          and ( event_value gt gpd_thresh ) then begin
        ; Adopt GPD fitting
        method_cfactual_use = 'GPD'
      endif
    endif
    ; Should it be counting?
    if ( max( method_cfactual eq 'empirical' ) eq 1 ) $
        and ( method_cfactual_use eq '' ) then begin
      method_cfactual_use = 'empirical'
    endif
    ; Should it be a semi-empirical estimator (kernel-based)?
    if ( max( method_cfactual eq 'kernel' ) eq 1 ) $
        and ( method_cfactual_use eq '' ) then begin
      method_cfactual_use = 'kernel'
    endif
    ; Confirm we have a method selected
    if method_cfactual_use eq '' then stop
    ; If we are to use the GPD
    if method_cfactual_use eq 'GPD' then begin
      ; Estimate GPD fit
      temp_param = gpd_fit( data_cfactual[id_gpd_thresh] - gpd_thresh )
      ; Estimate probability of exceeding the event threshold
      p_cfactual = gpd_cdf( event_value-gpd_thresh, scale=temp_param[0], $
          shape=temp_param[1] )
      p_cfactual = alog( $
          1. - ( gpd_quantile + ( 1. - gpd_quantile ) * p_cfactual ) )
    endif
    ; If we are to use empirical counting
    if method_cfactual_use eq 'empirical' then begin
      ; Count exceedances
      id = where( data_cfactual ge event_value, n_cfactual_count )
      p_cfactual = alog( n_cfactual_count ) - alog( n_cfactual_sample )
    endif
    ; If we are to estimate semi-empirically (kernel-based estimator)
    if method_cfactual_use eq 'kernel' then begin
      ; Estimate the PDF, using a kernel-based estimator
      pdf, data_cfactual, noplot=1, xid=temp_x, pdf=temp_pdf, npdf=n_pdf_x, $
          xrange=pdf_range, bandwidth=kernel_bandwidth_opt
      temp_pdf = temp_pdf / total( temp_pdf )
      ; Determine probability of exceeding event value
      id = where( temp_x ge event_value, n_id )
      if n_id eq 0 then begin
        ; Assume this means that we are numerically outside of the range of
        ; the distribution
        p_dfactual = alog( 0.5 ) - alog( n_cfactual_sample )
      endif else begin
        p_cfactual = alog( total( temp_pdf[id] ) )
      endelse
    endif
    ; Confirm we have an estimate
    if n_elements( p_cfactual ) ne 1 then stop

    ; Calculate the risk ratio
    result = p_factual - p_cfactual

  ; Calculate the likelihood counts
  endif else if metric eq 'likelihood risk ratio' then begin

    ; If we might be estimating something using empirical counts
    if ( max( method_factual eq 'empirical' ) eq 1 ) $
        or ( max( method_cfactual eq 'empirical' ) eq 1 ) then begin
      ; Determine the range of event values for use in calculating likelihoods
      if keyword_set( event_quantile_delta ) then begin
        id = where( data_factual ge event_value, n_id )
        if n_id eq 0 then stop
        temp_quantile = 1. - float( n_id ) / n_factual_sample
        temp_quantile = temp_quantile + [ -1., 1 ] * event_quantile_delta
        event_value_range = quantile_threshold( data_factual, $
            temp_quantile )
      endif else begin
        stop
      endelse
    endif

    ; Determine how we are going to make the factual likelihood estimate.
    method_factual_use = ''
    ; Should it be GPD?
    if max( method_factual eq 'GPD' ) eq 1 then begin
      ; Determine if we are in the tail
      gpd_thresh = ( quantile_threshold( data_factual, gpd_quantile ) )[0]
      if gpd_thresh lt event_value then begin
        ; Identify values to use in estimating GPD fit
        id_gpd = where( data_factual ge gpd_thresh, n_id_gpd )
        ; Do we have a sufficient sample
        if n_id_gpd ge n_gpd_sample_min then method_factual_use = 'GPD'
      endif
    endif
    ; Should it be a semi-empirical estimator (kernel-based)?
    if ( max( method_factual eq 'kernel' ) eq 1 ) $
        and ( method_factual_use eq '' ) then begin
      ; Just use it
      method_factual_use = 'kernel'
    endif
    ; Should it be empirical counting?
    if ( max( method_factual eq 'empirical' ) eq 1 ) $
        and ( method_factual_use eq '' ) then begin
      ; Just use it
      method_factual_use = 'empirical'
    endif
    ; Confirm we have a method selected
    if method_factual_use eq '' then stop
    ; If we are to estimate parametrically (GPD)
    if method_factual_use eq 'GPD' then begin
      ; Estimate GPD fit
      temp_param = gpd_fit( data_factual[id_gpd] - gpd_thresh )
      ; Determine likelihood at specified event value
      p_factual = gpd_pdf( event_value-gpd_thresh, scale=temp_param[0], $
          shape=temp_param[1] )
      p_factual = ( 1. - gpd_quantile ) * p_factual
    endif
    ; If we are to estimate semi-empirically (kernel-based estimator)
    if method_factual_use eq 'kernel' then begin
      ;; Define the range over which to estimate
      ;pdf_range = [ min( [ data_factual, data_cfactual ], max=temp ), temp ]
      ;pdf_range = 1.2 * ( pdf_range[1] - pdf_range[0] ) / 2. * [ -1., 1. ] $
      ;    + ( pdf_range[0] + pdf_range[1] ) / 2.
      ; Estimate the PDF, using a kernel-based estimator
      pdf, data_factual, noplot=1, xid=temp_x, pdf=temp_pdf, npdf=n_pdf_x, $
          xrange=pdf_range, bandwidth=kernel_bandwidth_opt
      temp_pdf = temp_pdf / total( temp_pdf * ( temp_x[1] - temp_x[0] ) )
      ; Determine likelihood at specified event value
      temp = abs( temp_x - event_value )
      id = where( temp eq min( temp ), n_id )
      if n_id gt 2 then stop
      if ( id[0] eq 0 ) or ( id[n_id-1] eq n_pdf_x - 1 ) then begin
        ;stop
        ; Assume this means that we are numerically outside of the range of
        ; the distribution
        p_factual = 0.
      endif else begin
        p_factual = mean( temp_pdf[id] )
      endelse
    endif
    ; If we are to estimate the factual likelihood using counts
    if method_factual_use eq 'empirical' eq 1 then begin
      ; Calculate likelihood counts
      id = where( ( data_factual ge event_value_range[0] ) $
          and ( data_factual le event_value_range[1] ), n_factual_count )
      p_factual = float( n_factual_count ) $
          / ( event_value_range[1] - event_value_range[0] ) / n_factual_sample
    endif
    ; Confirm we have a factual estimate
    if n_elements( p_factual ) ne 1 then stop

    ; Determine how we are going to make the counterfactual likelihood estimate.
    method_cfactual_use = ''
    ; Should it be GPD?
    if max( method_cfactual eq 'GPD' ) eq 1 then begin
      ; Determine if we are in the tail
      gpd_thresh = ( quantile_threshold( data_cfactual, gpd_quantile ) )[0]
      if gpd_thresh lt event_value then begin
        ; Identify values to use in estimating GPD fit
        id_gpd = where( data_cfactual ge gpd_thresh, n_id_gpd )
        ; Do we have a sufficient sample
        if n_id_gpd ge n_gpd_sample_min then method_cfactual_use = 'GPD'
      endif
    endif
    ; Should it be a semi-empirical estimator (kernel-based)?
    if ( max( method_cfactual eq 'kernel' ) eq 1 ) $
        and ( method_cfactual_use eq '' ) then begin
      ; Just use it
      method_cfactual_use = 'kernel'
    endif
    ; Should it be empirical counting?
    if ( max( method_cfactual eq 'empirical' ) eq 1 ) $
        and ( method_cfactual_use eq '' ) then begin
      ; Just use it
      method_cfactual_use = 'empirical'
    endif
    ; Confirm we have a method selected
    if method_cfactual_use eq '' then stop
    ; If we are to estimate parametrically (GPD)
    if method_cfactual_use eq 'GPD' then begin
      ; Estimate GPD fit
      temp_param = gpd_fit( data_cfactual[id_gpd] - gpd_thresh )
      ; Determine likelihood at specified event value
      p_cfactual = gpd_pdf( event_value-gpd_thresh, scale=temp_param[0], $
          shape=temp_param[1] )
      p_cfactual = ( 1. - gpd_quantile ) * p_cfactual
    endif
    ; If we are to estimate semi-empirically (kernel-based estimator)
    if method_cfactual_use eq 'kernel' then begin
      ;; Define the range over which to estimate
      ;pdf_range = [ min( [ data_factual, data_cfactual ], max=temp ), temp ]
      ;pdf_range = 1.2 * ( pdf_range[1] - pdf_range[0] ) / 2. * [ -1., 1. ] $
      ;    + ( pdf_range[0] + pdf_range[1] ) / 2.
      ; Estimate the PDF, using a kernel-based estimator
      pdf, data_cfactual, noplot=1, xid=temp_x, pdf=temp_pdf, npdf=n_pdf_x, $
          xrange=pdf_range, bandwidth=kernel_bandwidth_opt
      temp_pdf = temp_pdf / total( temp_pdf * ( temp_x[1] - temp_x[0] ) )
      ; Determine likelihood at specified event value
      temp = abs( temp_x - event_value )
      id = where( temp eq min( temp ), n_id )
      if n_id gt 2 then stop
      if ( id[0] eq 0 ) or ( id[n_id-1] eq n_pdf_x - 1 ) then begin
        ;stop
        ; Assume this means that we are numerically outside of the range of
        ; the distribution
        p_cfactual = 0.
      endif else begin
        p_cfactual = mean( temp_pdf[id] )
      endelse
    endif
    ; If we are to estimate the counterfactual likelihood using counts
    if method_cfactual_use eq 'empirical' eq 1 then begin
      ; Calculate likelihood counts
      id = where( ( data_cfactual ge event_value_range[0] ) $
          and ( data_cfactual le event_value_range[1] ), n_cfactual_count )
      p_cfactual = float( n_cfactual_count ) $
          / ( event_value_range[1] - event_value_range[0] ) / n_cfactual_sample
    endif
    ; Confirm we have a counterfactual estimate
    if n_elements( p_cfactual ) ne 1 then stop

    ; Calculate the risk ratio
    result = alog( float( p_factual ) ) - alog( float( p_cfactual ) )

  ; Something not yet supported
  endif else begin
    stop
  endelse

; For magnitude changes
endif else if metric eq 'magnitude shift' then begin

  ; Determine how we are going to make the factual estimate.
  method_factual_use = ''
  ; Should we use a GPD fit?
  if max( method_factual eq 'GPD' ) eq 1 then begin
    ; Determine if we are in the tail
    if event_quantile gt gpd_quantile then begin
      ; Identify values to use in estimating GPD fit
      gpd_thresh = ( quantile_threshold( data_factual, gpd_quantile ) )[0]
      id_gpd = where( data_factual ge gpd_thresh, n_id_gpd )
      ; Do we have a sufficient sample
      if n_id_gpd ge n_gpd_sample_min then method_factual_use = 'GPD'
    endif
  endif
  ; Should we use empirical counting?
  if ( max( method_factual eq 'empirical' ) eq 1 ) $
      and ( method_factual_use eq '' ) then begin
    ; Just use it
    method_factual_use = 'empirical'
  endif
  ; Should it be a semi-empirical estimator (kernel-based)?
  if ( max( method_factual eq 'kernel' ) eq 1 ) $
      and ( method_factual_use eq '' ) then begin
    method_factual_use = 'kernel'
  endif
  ; Confirm we have a method selected
  if method_factual_use eq '' then stop
  ; If we are using a GPD fit
  if method_factual_use eq 'GPD' then begin
    ; Estimate GPD fit
    temp_param = gpd_fit( data_factual[id_gpd] - gpd_thresh )
    ; Determine location of event quantile
    temp_quantile = ( event_quantile - gpd_quantile ) / ( 1. - gpd_quantile )
    event_value_factual = gpd_cdf( temp_quantile, scale=temp_param[0], $
          shape=temp_param[1], invert=1 ) + gpd_thresh
  endif
  ; If we are using an empirical count
  if method_factual_use eq 'empirical' then begin
    ; Estimate the value
    event_value_factual = quantile_threshold( data_factual, event_quantile )
    event_value_factual = event_value_factual[0]
  endif
  ; If we are to estimate semi-empirically (kernel-based estimator)
  if method_factual_use eq 'kernel' then begin
    ; Estimate the PDF, using a kernel-based estimator
    pdf, data_factual, noplot=1, xid=temp_x, pdf=temp_pdf, npdf=n_pdf_x, $
        xrange=pdf_range, bandwidth=kernel_bandwidth_opt
    temp_pdf = total( temp_pdf, cumulative=1 ) / total( temp_pdf )
    ; Determine probability of exceeding event value
    temp = abs( temp_pdf - event_quantile )
    id = where( temp eq min( temp ), n_id )
    if n_id eq 0 then stop
    id = id[0]
    if ( ( id eq 0 ) or ( temp_pdf[id] lt event_quantile ) ) $
        and ( id ne n_pdf_x - 1 ) then begin
      id = id - 1
    endif
    event_value_factual = ( temp_x[id+1] - temp_x[id] ) $
        / ( temp_pdf[id+1] - temp_pdf[id] ) $
        * ( event_quantile - temp_pdf[id] ) $
        + temp_x[id]
    ;event_value_factual = temp_x[id[0]]
  endif
  ; Confirm we have an estimate for the factual value
  if n_elements( event_value_factual ) ne 1 then stop

  ; Determine how we are going to make the counterfactual estimate.
  method_cfactual_use = ''
  ; Should we use a GPD fit?
  if max( method_cfactual eq 'GPD' ) eq 1 then begin
    ; Determine if we are in the tail
    if event_quantile gt gpd_quantile then begin
      ; Identify values to use in estimating GPD fit
      gpd_thresh = ( quantile_threshold( data_cfactual, gpd_quantile ) )[0]
      id_gpd = where( data_cfactual ge gpd_thresh, n_id_gpd )
      ; Do we have a sufficient sample
      if n_id_gpd ge n_gpd_sample_min then method_cfactual_use = 'GPD'
    endif
  endif
  ; Should we use empirical counting?
  if ( max( method_cfactual eq 'empirical' ) eq 1 ) $
      and ( method_cfactual_use eq '' ) then begin
    ; Just use it
    method_cfactual_use = 'empirical'
  endif
  ; Should it be a semi-empirical estimator (kernel-based)?
  if ( max( method_cfactual eq 'kernel' ) eq 1 ) $
      and ( method_cfactual_use eq '' ) then begin
    method_cfactual_use = 'kernel'
  endif
  ; Confirm we have a method selected
  if method_cfactual_use eq '' then stop
  ; If we are using a GPD fit
  if method_cfactual_use eq 'GPD' then begin
    ; Estimate GPD fit
    temp_param = gpd_fit( data_cfactual[id_gpd] - gpd_thresh )
    ; Determine location of event quantile
    temp_quantile = ( event_quantile - gpd_quantile ) / ( 1. - gpd_quantile )
    event_value_cfactual = gpd_cdf( temp_quantile, scale=temp_param[0], $
          shape=temp_param[1], invert=1 ) + gpd_thresh
  endif
  ; If we are using an empirical count
  if method_cfactual_use eq 'empirical' then begin
    ; Estimate the value
    event_value_cfactual = quantile_threshold( data_cfactual, $
        event_quantile )
    event_value_cfactual = event_value_cfactual[0]
  endif
  ; If we are to estimate semi-empirically (kernel-based estimator)
  if method_cfactual_use eq 'kernel' then begin
    ; Estimate the PDF, using a kernel-based estimator
    pdf, data_cfactual, noplot=1, xid=temp_x, pdf=temp_pdf, npdf=n_pdf_x, $
        xrange=pdf_range, bandwidth=kernel_bandwidth_opt
    temp_pdf = total( temp_pdf, cumulative=1 ) / total( temp_pdf )
    ; Determine probability of exceeding event value
    temp = abs( temp_pdf - event_quantile )
    id = where( temp eq min( temp ), n_id )
    if n_id eq 0 then stop
    id = id[0]
    if ( ( id eq 0 ) or ( temp_pdf[id] lt event_quantile ) ) $
        and ( id ne n_pdf_x - 1 ) then begin
      id = id - 1
    endif
    event_value_cfactual = ( temp_x[id+1] - temp_x[id] ) $
        / ( temp_pdf[id+1] - temp_pdf[id] ) $
        * ( event_quantile - temp_pdf[id] ) $
        + temp_x[id]
  endif
  ; Confirm we have an estimate for the counterfactual value
  if n_elements( event_value_cfactual ) ne 1 then stop

  ; Calculate the shift
  result = event_value_factual[0] - event_value_cfactual[0]

; Otherwise not yet supported
endif else begin
  stop
endelse

;***********************************************************************
; The end

;stop
return, result
END
