;+
; NAME:
;    ebm_calc
;
; PURPOSE:
;    This function runs various types of energy balance models (EBMs).
;
; CATEGORY:
;    Physics
;
; CALLING SEQUENCE:
;    RESULT = ebm_calc( FORCE, D_TIME=D_TIME, HEATCAP=HEATCAP, LAMBDA=LAMBDA )
;
; INPUTS:
;    FORCE:  A floating point vector time series of external forcing values, of 
;        length N_TIME.  This can also be an array of the form N_TIME*N_VALUE 
;        for running multiple models in parallel.
;
; KEYWORD PARAMETERS:
;    D_TIME:  The required time step of the data in FORCE and to be used for 
;        RESULT.  In units of years.
;    DOUBLE:  If set, calculations are done in double precision.  The default 
;        is single precision.
;    HEATCAP:  The required value of the heat capacity of the mixed layer of 
;        the ocean, in units of W*a*m^-2*K^-1.  This can be a vector of length 
;        N_VALUE for running multiple models in parallel.
;    INIT:  The initial temperature value.  This can be a vector of length 
;        N_VALUE for running multiple models in parallel.  If no value is 
;        input, the equilibrium value is assumed.
;    KV:  The optional value of the vertical diffusivity parameter, used in 
;        representing diffusion across the mixed layer into the deep ocean.  In 
;        units of W*K^-1.  This can be a vector of length N_VALUE for running 
;        multiple models in parallel.
;    LAMBDA:  The required value of the climate sensitivity parameter, in units 
;        of W*K^-1*m^-2.  This can be a vector of length N_VALUE for running 
;        multiple models in parallel.
;    MODEL_TYPE:  An optional scalar string specifying the type of energy 
;        balance model to use.  Supported values are:
;        * 'heatcap lambda':  The simplest EBM, with only the heatcap and 
;          lambda parameters.  HEATCAP*dT/dt=FORCE-LAMBDA*T.
;        * 'diffusive':  Same as the 'heatcap lambda' model but with the 
;          addition of a simple representative of ocean diffusivity.  The KV 
;          input is required.  HEATCAP*dT/dt=FORCE-LAMBDA*T.
;        The default is 'heatcap lambda'.
;    SOLVER_D_TIME_FACTOR:  The optional scalar integer specifying the number 
;        of time steps used by the solver within each time step of FORCE and 
;        RECORD.  In other words, the time step used by the solver is 
;        D_TIME/SOLVER_D_TIME_FACTOR.  The default is 1.
;    SOLVER_TYPE:  An optional scalar string specifying the numerical scheme to 
;        use to integrate the model.  Supported options are:
;        * 'first_difference':  The first difference scheme.
;        The default is 'first difference'.
;    TOTAL:  If set, then the RESULT time series is the sum of the N_VALUE time 
;        series.  The default is to return all time series.
;
; OUTPUTS:
;    RESULT:  The resultant time series of temperature calculated by the energy 
;        balance model.
;
; USES:
;    add_dim.pro
;
; PROCEDURE:
;    This function runs various types of energy balance models.
;
; REFERENCE:
;    Stone, D. A., M. R. Allen, F. Selten, M. Kliphuis, and P. A. Stott.  2007.
;        The detection and attribution of climate change using an ensemble of 
;        opportunity.  Journal of Climate, 20, 504-516.
;
; EXAMPLE:
;    ; The response to 
;    n_time = 100
;    force = [ [ findgen( n_time ) / n_time ], $
;        [ sin( findgen( n_time ) / n_time * 2. * !pi ) / 2. ] ]
;    result = ebm_calc( force, d_time=1., heatcap=[1.,10.], lambda=1., $
;        init=0., total=1 )
;
; COMMENTS:
;    ebm_diffuse.pro:
;      Written by:  Myles Allen (m.allen1@physics.ox.ac.uk), 2005-01-01 as 
;          scm2.pro 	
;      Written by:  Daithi Stone (dastone@runbox.com), 2005-07-15 (switched to 
;          use of common EBM_PARAMS;  added documentation)
;      Modified:  DAS, 2009-02-17 (switched use of common EBM_PARAMS to input 
;          X;  converted documentation to standard format) 
;      Modified:  DAS, 2009-04-09 (corrected bug in calculation of QZ when 
;          N_VALUE>1) 
;      Modified:  DAS, 2011-04-08 (corrected bug when single forcing signal 
;          given;  improved ability to handle double-precision input;  fixed 
;          bug in 2009-04-09 modification)
;    ebm_dydt.pro:
;      Written by:  Daithi A. Stone (dastone@runbox.com), 2005-07-15
;      Modified:  DAS, 2009-02-13 (switched use of common EBM_PARAMS to input 
;          X;  converted documentation to standard format) 
;
; MODIFICATION HISTORY:
;    Written by:  Daithi Stone (dastone@runbox.com), 2009-02-20, as ebm_run.pro
;    Modified:  DAS, 2011-05-23 (added SOLVER_TYPE keyword)
;    Modified:  DAS, 2025-04-09 (Incorporated ebm_dydt.pro and ebm_diffuse.pro; 
;        Renamed ebm_calc.pro;  Removed array05 and array1 variables from 
;        "diffusive" model)
;-

;***********************************************************************

FUNCTION EBM_CALC, $
    FORCE, $
    D_TIME=d_time, $
    HEATCAP=heatcap, KV=kv, LAMBDA=lambda, $
    INIT=INIT, $
    MODEL_TYPE=model_type, $
    SOLVER_D_TIME_FACTOR=solver_d_time_factor, SOLVER_TYPE=solver_type, $
    DOUBLE=double_opt, $
    TOTAL=total_opt

;***********************************************************************
; Constants and Options

; The default model type
if not( keyword_set( model_type ) ) then model_type = 'heatcap lambda'
temp = [ 'heatcap lambda', 'diffusive' ]
if max( model_type eq temp ) eq 0 then stop

; The option for double precision
if keyword_set( double_opt ) then begin
  one = 1.d
  pi = !dpi
endif else begin
  one = 1.
  pi = !pi
endelse

; Copy input variables to ensure unaltered input, convert to double precision 
; if requested
heatcap_use = one * heatcap
lambda_use = one * lambda
if keyword_Set( kv ) then kv_use = one * kv
force_use = one * force
d_time = one * d_time

; The number of time steps
n_time = n_elements( force[*,0] )

; The default solver type
if not( keyword_set( solver_type ) ) then solver_type = 'first difference'
temp = [ 'first difference' ]
if max( solver_type eq temp ) eq 0 then stop
; The default solver time step
if not( keyword_set( solver_d_time_factor ) ) then solver_d_time_factor = 1
solver_d_time = d_time / solver_d_time_factor

; Determine number of parallel simulations
n_heatcap = n_elements( heatcap_use )
n_lambda = n_elements( lambda_use )
n_kv = n_elements( kv_use )
n_force = n_elements( force_use[0,*] )
n_init = n_elements( init )
n_value = max( [ n_heatcap, n_lambda, n_kv, n_force, n_init ] )
; Ensure all variables satisfy number of parallel simulations
if n_heatcap ne n_value then begin
  heatcap_use = add_dim( [ heatcap_use ], 1, n_value )
endif
if n_lambda ne n_value then lambda_use = add_dim( [ lambda_use ], 1, n_value )
if keyword_set( kv ) then begin
  if n_kv ne n_value then kv_use = add_dim( [ kv_use ], 1, n_value )
endif
if n_force ne n_value then force_use = add_dim( [ force_use ], 1, n_value )
; Reform to handy dimensions
heatcap_use = reform( heatcap_use )
lambda_use = reform( lambda_use )
if keyword_set( kv ) then kv_use = reform( kv_use )
force_use = reform( force_use, n_time, n_value )

; Copy the input initial state if provided
if n_elements( init ) gt 0 then begin
  init_use = one * init
  if n_init ne n_value then init_use = add_dim( [ init_use ], 1, n_value )
  init_use = reform( init_use )
; Otherwise adopt the default temperature (equilibrium)
endif else begin
  init_use = reform( force_use[0,*] ) / lambda_use
  id = where( finite( init_use ) eq 0, n_id )
  if n_id gt 0 then init_use[id] = one * 0.
endelse

; Interpolate forcing data
n_solver_time = ( n_time - 1 ) * solver_d_time_factor + 1
if solver_d_time_factor ne 1 then begin
  temp_force = fltarr( n_solver_time, n_force )
  temp_force[0,*] = force_use[0,*]
  index = indgen( n_time - 1 )
  for i_time = 1, solver_d_time_factor - 1 do begin
    index = indgen( n_time - 1 )
    index_solver = inddex * solver_d_time_factor + i_time
    temp_force[index_solver,*] $
        = ( solver_d_time_factor - i_time ) / float( solver_d_time_factor ) $
        * force_use[index-1,*] $
        + i_time / float( solver_d_time_factor ) * force_use[index,*]
  endfor
  force_use = temporary( temp_force )
endif

;***********************************************************************
; Run Simplest EBM

; If this model is selected
if model_type eq 'heatcap lambda' then begin
  ; Initialise output array
  result = one * !values.f_nan * fltarr( n_time, n_value )
  ; Initialise calculation
  result[0,*] = init_use
  temp_state = init_use
  ; Iterate through time
  for i_solver_time = 1l, n_solver_time - 1l do begin
    ; As long as we still have a finite state
    if min( finite( result[0,*] ) ) eq 1 then begin
      ; If we are using a first difference solver
      if solver_type eq 'first difference' then begin
        ; Calculate change
        temp_state = temp_state $
            + 1. / heatcap_use $
            * ( reform( force_use[i_solver_time,*] ) - lambda * temp_state ) $
            * solver_d_time
      ; Otherwise solver not yet supported
      endif else begin
        stop
      endelse
      ; Record value if appropriate
      if ( i_solver_time mod solver_d_time_factor ) eq 0 then begin
        result[i_solver_time/solver_d_time_factor,*] = temp_state
      endif
    endif
  endfor
endif

;***********************************************************************
; Run EBM with Diffusive Ocean Mixed Layer

; Run if positive kv is input
if model_type eq 'diffusive' then begin
  ; Initialise output time series
  result = one * !values.f_nan * fltarr( n_solver_time, n_value )
  result[0,*] = init_use
  ; Convert inputted parameters to more useful forms
  b0 = reform( one / lambda_use, 1, n_value )
  b1 = heatcap / ( lambda_use ^ 2 ) / solver_d_time
  e1 = reform( exp( -lambda_use * solver_d_time / heatcap_use ), 1, n_value )
  ; Some stuff to do with the diffusion term
  n_sqrt = sqrt( n_solver_time )
  isqrd = indgen( n_sqrt + 1 ) ^ 2
  isqri = sqrt( kv_use * solver_d_time / pi ) $
      ## reform( 1. / ( findgen( fix( n_sqrt + 1 ) ) + 0.5 ), $
      fix( n_sqrt ) + 1, 1 )
  isqri = reform( isqri, fix( n_sqrt + 1 ), n_value )
  qz = reform( one * fltarr( 1, n_value ), 1, n_value )
  qc = reform( 3. * sqrt( kv_use * solver_d_time / pi ) / 4. * solver_d_time, $
      1, n_value )
  ; Iterate through time
  for i_time = 1l, n_solver_time - 1 do begin
    ; The first difference solver
    if solver_type eq 'first difference' then begin
      ; Calculate next value
      result[i_time,*] = b0 * force_use[i_time,*] $
          - b1 * ( force_use[i_time,*] - force_use[i_time-1,*] ) $
          + ( result[i_time-1,*] - b0 * force_use[i_time-1,*] $
          + b1 * ( force_use[i_time,*] - force_use[i_time-1,*] ) ) * e1
    endif
    ; Add effect of diffusion term if we are far enough along in time
    if i_time gt 3 then begin
      i_time_s = floor( sqrt( i_time ) )
      ; Smooth diffusive term to eliminate 2*dtime noise
      temp = reform( result[i_time-1-isqrd[0:i_time_s-1],*] $
          - result[i_time-1-isqrd[1:i_time_s],*], i_time_s, n_value )
      qz = one / 2. $
          * ( qz $
          + reform( total( temp * isqri[0:i_time_s-1,*], 1 ), 1, n_value ) )
      ; Include impact of diffusion over timestep i
      result[i_time,*] = reform( $
          ( result[i_time,*] - qz + qc * result[i_time-1,*] ) $
          / ( one + qc ), $
          1, n_value )
    endif
  endfor
  ; Extract values on d_time time steps
  if solver_d_time_factor ne 1 then begin
    index = lindgen( n_time ) * solver_d_time_factor
    result = result[indexd,*]
    index = 0
  endif
endif

;***********************************************************************
; Post-Processing

; Sum up all responses, if requested
if keyword_set( total_opt ) then begin
  if n_force gt 1 then begin
    result = total( result, 2 )
  endif else begin
    result = reform( result )
  endelse
endif

;***********************************************************************
; The End

return, result
END
