;+
; NAME:
;    dbinom
;
; PURPOSE:
;    This function computes the density of the binomial distribution.
;
; CATEGORY:
;    Statistics
;
; CALLING SEQUENCE:
;    dbinom
;
; INPUTS:
;    OBS:  A required integer vector specifying the number of observations.  
;        Of length N_OBS.
;    TRIAL:  A required scalar integer specifying the number of trials.
;    PROB:  A required scalar float specifying the probability of success on 
;        each trial.
;
; KEYWORD PARAMETERS:
;    BINOMIAL:  If set, then calculations are performed using the binomial.pro 
;        function.  The default is to perform calculations internally.  
;        Currently binomial.pro has better numerical accuracy for large TRIAL, 
;        through use of the Gaussian approximation.
;    DOUBLE:  If set, calculations are performed in double precision.
;
; OUTPUTS:
;    RESULT:  A floating point vector of length N_OBS of the densities for each 
;        observation number in OBS.
;
; USES:
;    var_type.pro
;
; PROCEDURE:
;    This function calculates the density function and the binomial 
;    distribution numerically.
;
; EXAMPLE:
;    ; The probability of rolling 5 ones out of 10 rolls of a die.
;    print, dbinom( 5, 10, 1./6.)
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2019-10-11
;-

;***********************************************************************

FUNCTION DBINOM, $
    OBS, TRIAL, PROB, $
    BINOMIAL=binomial_opt, $
    DOUBLE=double_opt

;***********************************************************************
; Constants and options

; Ensure legal inputs
n_obs = n_elements( obs )
if n_obs eq 0 then stop
if max( var_type( obs ) eq [ 2, 3 ] ) ne 1 then stop 
if n_elements( trial ) ne 1 then stop
if max( var_type( trial ) eq [ 2, 3 ] ) ne 1 then stop 
if n_elements( prob ) ne 1 then stop
if ( prob lt 0. ) or ( prob gt 1. ) then stop

; Option for double precision
double_opt = keyword_set( double_opt )
prob_use = prob
if double_opt then prob_use = double( prob_use )

;***********************************************************************
; Calculate densities

; Initialise output vector
if double_opt eq 1 then begin
  result = double( obs )
endif else begin
  result = float( obs )
endelse
result[*] = !values.f_nan

; Iterate through number of observations
for i_obs = 0, n_obs - 1 do begin
  ; Use the binomial.pro function for calculations
  if keyword_set( binomial_opt ) then begin
    if obs[i_obs] eq trial then begin
      result[i_obs] = binomial( trial, trial, prob_use )
    endif else begin
      result[i_obs] = binomial( obs[i_obs], trial, prob_use ) $
          - binomial( obs[i_obs] + 1, trial, prob_use )
    endelse
  ; Perform the calculations internally
  endif else begin
    ; Work with logarithms for better numerical accuracy
    temp = 1.
    if double_opt eq 1 then temp = double( temp )
    temp = alog10( temp )
    for i_k = 0, obs[i_obs] - 1 do temp = temp + alog10( trial - i_k )
    ;result[i_obs] = temp - alog10( factorial( obs[i_obs] ) ) $
    ;    + alog10( prob ) * obs[i_obs] $
    ;    + alog10( 1. - prob ) * ( trial - obs[i_obs] )
    result[i_obs] = temp $
        - total( alog10( 1. + findgen( max( [ 1, obs[i_obs] ] ) ) ) ) $
        + alog10( prob ) * obs[i_obs] $
        + alog10( 1. - prob ) * ( trial - obs[i_obs] )
    result[i_obs] = 10. ^ result[i_obs]
  endelse
endfor

;***********************************************************************
; The end

return, result
END
