;+
; NAME:
;    convert_rotated_grid
;
; PURPOSE:
;    The functions transforms data from a rotated polar grid to an unrotated 
;    polar grid, from unrotated to rotated, or from rotated to rotated.
;
; CATEGORY:
;    Geographical
;
; CALLING SEQUENCE:
;    convert_rotated_grid, in_lon=in_lon, in_lat=in_lat, out_lon=out_lon, $
;        out_lat=out_lat, $
;        {in_north_lon=in_north_lon, in_north_lat=in_north_lat,} $
;        {out_north_lon=out_north_lon, out_north_lat=out_north_lat}
;
; INPUTS:
;    IN_LAT, IN_LON, IN_NORTH_LAT, IN_NORTH_LON, OUT_NORTH_LAT, OUT_NORTH_LON
;
; KEYWORD PARAMETERS:
;    IN_LAT:  A required float scalar, vector, or array containing the latitude
;        values in the input grid.
;    IN_LON:  A required float scalar, vector, or array containing the 
;        longitude values in the input grid.
;    IN_NORTH_LAT:  A float scalar required if the input grid is a rotated 
;        grid.  It specifies the latitude of the north pole of the input 
;        rotated grid on an unrotated grid.
;    IN_NORTH_LON:  A float scalar required if the input grid is a rotated 
;        grid.  It specifies the longitude of the north pole of the input 
;        rotated grid on an unrotated grid.
;    OUT_NORTH_LAT:  A float scalar required if the output grid is a rotated 
;        grid.  It specifies the latitude of the north pole of the output 
;        rotated grid on an unrotated grid.
;    OUT_NORTH_LON:  A float scalar required if the output grid is a rotated 
;        grid.  It specifies the longitude of the north pole of the output 
;        rotated grid on an unrotated grid.
;
; OUTPUTS:
;    OUT_LAT:  A float scalar, vector, or array, of the same dimensions and 
;        size as IN_LAT, containing the latitude values in the output grid.
;    OUT_LON:  A float scalar, vector, or array, of the same dimensions and 
;        size as IN_LON, containing the longitude values in the output grid.
;
; USES:
;    var_type.pro
;
; PROCEDURE:
;    This procedure performs the matrix multiplication needed to convert 
;    between different grid types.
;
; EXAMPLE:
;    ; Convert 45N,75W from a rotated grid with the north pole at 42.5N,83.0E to an unrotated pole
;      convert_rotated_grid, in_lon=-75., in_lat=45., out_north_lon=83.0, out_north_lat=42.5, out_lon=lon_unrotated, out_lat=lat_unrotated
;    ; Convert back to the original rotated grid
;      convert_rotated_grid, in_lon=lon_unrotated, in_lat=lat_unrotated, in_north_lon=83.0, in_north_lat=42.5, out_lon=out_lon, out_lat=out_lat

;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2019-12-17
;-

;***********************************************************************

PRO CONVERT_ROTATED_GRID, $
    IN_LON=in_lon, IN_LAT=in_lat, $
    IN_NORTH_LON=in_north_lon, IN_NORTH_LAT=in_north_lat, $
    OUT_LON=out_lon, OUT_LAT=out_lat, $
    OUT_NORTH_LON=out_north_lon, OUT_NORTH_LAT=out_north_lat

;***********************************************************************
; Inputs and constants

; Determine if we are converting from rotated or from unrotated
in_grid_type = ''
if ( n_elements( in_north_lon ) eq 1 ) and ( n_elements( in_north_lat ) eq 1 ) $
    then begin
  in_grid_type = 'rotated'
endif else begin
  in_grid_type = 'unrotated'
endelse
; Determine if we are converting to rotated or to unrotated
if ( n_elements( out_north_lon ) eq 1 ) $
    and ( n_elements( out_north_lat ) eq 1 ) then begin
  out_grid_type = 'rotated'
endif else begin
  out_grid_type = 'unrotated'
endelse
; If we are converting from unrotated to unrotated, then just return the input
if ( in_grid_type eq 'unrotated' ) and ( out_grid_type eq 'unrotated' ) $
    then begin
  return
endif

; Factor for conversion from degrees to radian
if var_type( in_lon ) eq 5 then begin
  degrad = !dpi / 180.d
endif else begin
  degrad = !pi / 180.
endelse

;***********************************************************************
; Convert to Cartesian

; Convert from an unrotated grid
if in_grid_type eq 'unrotated' then begin
  ; Convert to Cartesian
  unrotated_x = cos( in_lon * degrad ) * cos( in_lat * degrad )
  unrotated_y = sin( in_lon * degrad ) * cos( in_lat * degrad )
  unrotated_z = sin( in_lat * degrad )
endif

; Convert from a rotated grid
if in_grid_type eq 'rotated' then begin
  ; Convert to rotated Cartesian
  in_x = cos( in_lon * degrad ) * cos( in_lat * degrad )
  in_y = sin( in_lon * degrad ) * cos( in_lat * degrad )
  in_z = sin( in_lat * degrad )
  ; Convert to unrotated Cartesian
  in_theta = ( -90 + in_north_lat ) * degrad
  in_phi = ( 180. - in_north_lon ) * degrad
  unrotated_x = cos( in_theta ) * cos( in_phi ) * in_x $
      + sin( in_phi ) * in_y + sin( in_theta ) * cos( in_phi ) * in_z
  unrotated_y = -cos( in_theta ) * sin( in_phi ) * in_x $
      + cos( in_phi ) * in_y - sin( in_theta ) * sin( in_phi ) * in_z
  unrotated_z = -sin( in_theta ) * in_x + cos( in_theta ) * in_z
endif

;***********************************************************************
; Convert from Cartesian

; Convert to an unrotated grid
if out_grid_type eq 'unrotated' then begin
  ; Convert from Cartesian
  out_lat = asin( unrotated_z )
  out_lon = atan( unrotated_y, unrotated_x )
endif

; Convert to a rotated grid
if out_grid_type eq 'rotated' then begin
  ; Convert to rotated Cartesian
  out_theta = ( -90 + out_north_lat ) * degrad
  out_phi = out_north_lon * degrad
  out_x = -cos( out_theta ) * cos( out_phi ) * unrotated_x $
      - cos( out_theta ) * sin( out_phi ) * unrotated_y $
      - sin( out_theta ) * unrotated_z
  out_y = sin( out_phi ) * unrotated_x - cos( out_phi ) * unrotated_y
  out_z = -sin( out_theta ) * cos( out_phi ) * unrotated_x $
      - sin( out_theta ) * sin( out_phi ) * unrotated_y $
      + cos( out_theta ) * unrotated_z
  ; Convert to polar coordindates
  out_lat = asin( out_z )
  out_lon = atan( out_y, out_x )
endif

; Convert to degrees
out_lon = out_lon / degrad
out_lat = out_lat / degrad

;***********************************************************************
; The end

return
END
