;+
; NAME:
;    contour_legend
;
; PURPOSE:
;    This procedure plots a legend for colour-contour plots.
;
; CATEGORY:
;    Graphics
;
; CALLING SEQUENCE:
;    contour_legend, xpos, ypos
;
; INPUTS:
;    XPOS:  The x-coordinates of the legend box, of integer or floating point 
;        type, in the form of a 2-element vector.  The vector is of the form 
;        [ leftmost, rightmost ].
;    YPOS:  The y-coordinates of the legend box, of integer or floating point 
;        type, in the form of a 2-element vector.  The vector is of the form 
;        [ bottom, top ].
;
; KEYWORD PARAMETERS:
;    ARROWEND:  If set, then pointed end levels are plotted, indicating that 
;        the end levels extend beyond the range of the plot.  The default is 
;        rectangular end levels, like all the other levels.  A 2 element vector 
;        can also be input, where the first element gives the setting (0 or 1) 
;        for the left/bottom end and the second for the right/top end.
;    C_COLORS:  An optional vector of the contour level colour indices, of type 
;        integer.  This vector should be of size NLEVELS-1.  The default is 
;        [1,2,3,...,NLEVELS].  If LEVELS and NLEVELS are not input, then this 
;        is required.
;    [X,Y]CHARSHIFT:  Optional scalar floats that specify the horizontal and 
;        vertical size of characters when scaled to when CHARSIZE=1.  Used for 
;        positioning of text, but can be calculated automatically.
;    CHARSIZE:  An optional scalar float specifying the size of the text 
;        characters.  The default is 1.
;    COLOR:  An option scalar integer specifying the colour index of the text 
;        and border.  The default is !p.color.
;    FONT:  An optional scalar integer specifying the standard IDL font keyword.
;    HORIZONTAL:  If set, the procedure plots a horizontal legend.  This is the 
;        default.
;    LEVELS:  An optional vector of contour level values, of type integer or 
;        float.  Of lenght NLEVELS.  If not input, then this is calculated 
;        according to a method depending on what else is input.
;    [MIN,MAX]LEVEL:  An optional 2-element integer or float vector specifying 
;        the [minimum, maximum] level value.  If used, both values must be 
;        set.  If LEVELS is set it overrides these values.
;    NLEVELS:  The number of contour level values, of type integer.  If LEVELS 
;        is set, its size overrides NLEVELS.  If set to a negative value, then 
;        the routine picks a comfortable number of levels with -NLEVELS being a 
;        maximum.  The default is to pick a comfortable limit according to 
;        choose_levels.pro.
;    NORMAL:  If set the procedure uses normalised coordinates.  The default is 
;        data coordinates.
;    NTICKS:  An optional scalar integer specifying the number of tick marks.  
;        If TICKNAME is set, its size overrides NTICKS.
;    SUBTITLE:  An optional scalar string specifying the subtitle of the 
;        legend, giving the data units.
;    THICK:  An optional scalar integer specifying the line thickness.
;    TICK_DECPLACE:  An optional scalar integer specifying the number of 
;        decimal places to print in the tick labels.  This is used only if 
;        TICKNAME is not defined.
;    TICKLEN:  An optional scalar float specifying the length of the tick marks 
;        in terms of fraction of the legend box size, of type floating point.  
;        If positive then the ticks (and TICKNAME labels) are plotted to the 
;        left (VERTICAL=1) or below (HORIZONTAL=1) the colour bar.  If negative 
;        then the ticks (and TICKNAME labels) are plotted to the right 
;        (VERTICAL=1) or above (HORIZONTAL=1) the colour bar.  If zero then no 
;        tick marks are plotted.  The default is 0.1.
;    TICKNAME:  An optional string vector containing the labels for each tick 
;        mark.
;    TICKV:  An optional integer or floating vector containing the positions of 
;        the tick marks.
;    TIT_ALIGNMENT:  An optional float specifying the alignment of the title.  
;        0 means left-aligned, 0.5 means centred, and 1 means right-aligned.  
;        The default is 0.5.
;    TIT_CHARSIZE:  An optional float specifying the size of the text 
;        characters in the legend title.  The default is CHARSIZE.
;    TITLE:  An optional scalar string specifying the title of the legend.  If 
;        set to a non-string type (i.e. "/TITLE") then "Legend" is used.
;    VERTICAL:  If set, the procedure plots a vertical legend.  The default is 
;        horizontal.
;
; USES:
;    choose_levels.pro
;    decimal_place.pro
;    str.pro
;    var_type.pro
;
; PROCEDURE:
;    This procedure draws a box and fills it with colour-contoured levels.  It 
;    then adds ticks and labels to the box.
;
; EXAMPLE:
;    Plot a horizontal red-green-blue legend at the bottom of the display.
;      plot, [0,1], nodata=1, xstyle=4, ystyle=4
;      tek_color
;      contour_legend, [0.2,0.8], [0.1,0.2], levels=[0,1,2,3], $
;          c_colors=[2,3,4], title='RGB', horizontal=1, normal=1, $
;          tickv=[0.5,1.5,2.5], tickname=['Red','Green','Blue']
;    Plot a vertical red-green-blue legend at the right of the display with 
;    labels to the right.
;      plot, [0,1], nodata=1, xstyle=4, ystyle=4
;      tek_color
;      contour_legend, [0.8,0.85], [0.3,0.7], levels=[0,1,2,3], $
;          c_colors=[2,3,4], title='RGB', vertical=1, normal=1, $
;          tickv=[0.5,1.5,2.5], tickname=['Red','Green','Blue'], ticklen=-0.1
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2000-07-11.
;    Modified:  DAS, 2000-07-12 (corrected HORIZONTAL and VERTICAL keywords, 
;        added MINLEVEL and MAXLEVEL keywords).
;    Modified:  DAS, 2000-07-19 (added tick-choosing algorithm).
;    Modified:  DAS, 2000-09-21 (cleaning, debugging, documentation).
;    Modified:  DAS, 2000-09-25 (modified tick-choosing).
;    Modified:  DAS, 2000-11-28 (modified tick-choosing).
;    Modified:  DAS, 2002-11-22 (added FONT keyword).
;    Modified:  DAS, 2007-02-06 (added ARROWEND option;  changed plotting 
;        method)
;    Modified:  DAS, 2007-09-18 (allowed a flexible number of levels to be 
;        chosen;  some standardisation of code)
;    Modified:  DAS, 2009-05-04 (added the THICK keyword;  added noclip 
;        property to borders and ticks;  fixed bug with TICKNAME implementation 
;        when ARROWEND is set)
;    Modified:  DAS, 2009-10-27 (added TICKV keyword)
;    Modified:  DAS, 2019-12-18 (Added [X,Y]CHARSHIFT keyword inputs, limited 
;        character width calculations to when necessary)
;    Modified:  DAS, 2024-05-27 (Added zero and negative TICKLEN capability;  
;        Added TIT_ALIGNMENT, TIT_CHARSIZE keyword inputs;  Updated 
;        documentation;  Fixed bug if LEVELS is of type integer)
;-

;***********************************************************************

PRO CONTOUR_LEGEND, $
	Xpos, Ypos, $
	LEVELS=levels, NLEVELS=n_levels, MINLEVEL=min_level, $
	  MAXLEVEL=max_level, $
	C_COLORS=c_colors, $
	TICKNAME=tickname, TICK_DECPLACE=tickdecplace, TICKV=tickv, $
	NTICKS=n_ticks, TICKLEN=ticklen, $
	COLOR=color, $
	CHARSIZE=charsize, XCHARSHIFT=XCHARSHIFT, YCHARSHIFT=ycharshift, $
	FONT=font, $
	THICK=thick, $
        TIT_ALIGNMENT=tit_alignment, TIT_CHARSIZE=tit_charsize, $
	TITLE=title, SUBTITLE=subtitle, $
	HORIZONTAL=horizontal_opt, VERTICAL=vertical_opt, $
	NORMAL=normal_opt, $
	ARROWEND=arrowend_opt

;***********************************************************************
; Constants and Options

; Legend box dimensions
xdim = xpos[1] - xpos[0]
ydim = ypos[1] - ypos[0]

; Plotting orientation
horizontal_opt = keyword_set( horizontal_opt ) $
    or not( keyword_set( vertical_opt ) )
vertical_opt = not( horizontal_opt )

; Normal coordinates option
normal_opt = keyword_set( normal_opt )

; Arrowed end option
if keyword_set( arrowend_opt ) then begin
  if n_elements( arrowend_opt ) eq 1 then begin
    arrowend_opt = [ 0, 0 ] + arrowend_opt
  endif
endif else begin
  arrowend_opt = [ 0, 0 ]
endelse

;***********************************************************************
; Default settings

; Contour levels
if not( keyword_set( n_levels ) ) then begin
  if keyword_set( levels ) then begin
    n_levels = n_elements( levels )
  endif else if keyword_set( c_colors ) then begin
    n_levels = n_elements( c_colors ) + 1
  endif else begin
    n_levels = 0
  endelse
endif
if not( keyword_set( levels ) ) then begin
  if ( n_elements( min_level ) ne 0 ) $
      and ( n_elements( max_level ) ne 0 ) then begin
    if n_levels gt 0 then begin
      levels = min_level + findgen( n_levels ) / ( n_levels - 1 ) $
          * ( max_level - min_level )
    endif else begin
      levels = choose_levels( [ min_level, max_level ], nlevels=-n_levels )
      levels = choose_levels( [ min( levels ), max( levels ) ], $
          nlevels=-n_levels )
    endelse
  endif else begin
    levels = indgen( abs( n_levels ) )
  endelse
endif
n_levels = n_elements( levels )

; Contour colours
if not( keyword_set( c_colors ) ) then c_colors = indgen( n_levels - 1 ) + 1

; Level ticks
; Number of ticks
if not( keyword_set( tickname ) ) then begin
  if keyword_set( tickv ) then begin
    tickname = str( tickv, decplace )
    n_ticks = n_elements( tickv )
  endif else if keyword_set( n_ticks ) then begin
    tickname = findgen( n_ticks ) / ( n_ticks - 1 ) $
        * ( levels[n_levels-1] - levels[0] ) + levels[0]
    id = where( abs( tickname ) gt 0 )
    decplace = decimal_place( min( abs( tickname[id] ) ) )
    tickname = str( tickname, decplace )
  endif else begin
    tickname = choose_levels( levels, range=[levels[0],levels[n_levels-1]] )
    tickname = str( tickname, max( decimal_place( tickname ) ) )
    n_ticks = n_elements( tickname )
  endelse
endif else begin
  if not( keyword_set( n_ticks ) ) then begin
    if keyword_set( tickv ) then begin
      n_ticks = n_elements( tickv )
    endif else begin
      n_ticks = n_elements( tickname )
    endelse
  endif
endelse
if not( keyword_set( tickv ) ) then tickv = tickname
; The default tick mark length
if n_elements( ticklen ) eq 0 then ticklen = 0.1

; Border and text colour
if not( keyword_set( color ) ) then color = !p.color

; Set up the plotting coordinates
if normal_opt then begin
  plot, [0,1], [0,1], nodata=1, xstyle=5, ystyle=5, noerase=1, xmargin=[0,0], $
      ymargin=[0,0]
endif

; Text character constants
; Character size
if not( keyword_set( charsize ) ) then charsize = 1.
; Character scale
if not( keyword_set( xcharshift ) ) or not( keyword_set( ycharshift ) ) $
    then begin
  if not( keyword_set( xcharshift ) ) then begin
    xcharshift = 1. * !d.x_ch_size / !d.x_size
  endif
  if not( keyword_set( ycharshift ) ) then begin
    ycharshift = 1. * !d.y_ch_size / !d.y_size
  endif
  origin = convert_coord( [0], [0], data=1, to_normal=1 )
  charshift = convert_coord( [xcharshift+origin[0]], [ycharshift+origin[1]], $
      normal=1, to_data=1 )
  xcharshift = charshift[0]
  ycharshift = charshift[1]
endif

; Legend title
if keyword_set( title ) then begin
  ; The default title
  if var_type( title ) ne 7 then title = 'Legend'
  ; The default alignment
  if n_elements( tit_alignment ) eq 0 then tit_alignment = 0.5
  ; The default character size
  if not( keyword_set( tit_charsize ) ) then tit_charsize = charsize
endif

;***********************************************************************
; Plot the coloured contour levels

; Create the contoured grid for vertical legend
if vertical_opt then begin
  yvec = ypos[0] $
      + float( levels - levels[0] ) / ( levels[n_levels-1] - levels[0] ) * ydim
  yvec = [ [ yvec ], [ yvec ] ]
  xvec = fltarr( n_levels, 2 )
  xvec[1:n_levels-2,0] = xpos[0]
  xvec[1:n_levels-2,1] = xpos[1]
  if arrowend_opt[0] eq 0 then begin
    xvec[0,*] = xpos
  endif else begin
    xvec[0,*] = mean( xpos )
  endelse
  if arrowend_opt[1] eq 0 then begin
    xvec[n_levels-1,*] = xpos
  endif else begin
    xvec[n_levels-1,*] = mean( xpos )
  endelse
endif

; Create the contoured grid for horizontal legend
if horizontal_opt then begin
  xvec = xpos[0] $
      + float( levels - levels[0] ) / ( levels[n_levels-1] - levels[0] ) * xdim
  xvec = [ [ xvec ], [ xvec ] ]
  yvec = fltarr( n_levels, 2 )
  yvec[1:n_levels-2,0] = ypos[0]
  yvec[1:n_levels-2,1] = ypos[1]
  if arrowend_opt[0] eq 0 then begin
    yvec[0,*] = ypos
  endif else begin
    yvec[0,*] = mean( ypos )
  endelse
  if arrowend_opt[1] eq 0 then begin
    yvec[n_levels-1,*] = ypos
  endif else begin
    yvec[n_levels-1,*] = mean( ypos )
  endelse
endif

; Draw the coloured contour levels
for i_levels = 0, n_levels - 2 do begin
  temp_x = [ xvec[i_levels,0], xvec[i_levels,1], xvec[i_levels+1,1], $
      xvec[i_levels+1,0] ]
  temp_y = [ yvec[i_levels,0], yvec[i_levels,1], yvec[i_levels+1,1], $
      yvec[i_levels+1,0] ]
  polyfill, temp_x, temp_y, color=c_colors[i_levels]
endfor

;***********************************************************************
; Draw the legend border

; Left or bottom border
oplot, xvec[0,*], yvec[0,*], color=color, thick=thick, noclip=1
; Right or top border
oplot, xvec[n_levels-1,*], yvec[n_levels-1,*], color=color, thick=thick, $
    noclip=1
; Top or left border
oplot, xvec[*,0], yvec[*,0], color=color, thick=thick, noclip=1
; Bottom or right border
oplot, xvec[*,1], yvec[*,1], color=color, thick=thick, noclip=1

;***********************************************************************
; Legend tick marks and names

; Vertical ticks and tick names
if vertical_opt then begin
  ; The width of longest tick name (required for positioning subtitle)
  xwidth = 0
  ; Iterate through ticks
  for i_ticks = 0, n_ticks - 1 do begin
    ; Only do this if not an arrowed end
    check = 1
    if ( arrowend_opt[0] eq 1 ) and ( tickv[i_ticks] lt levels[1] ) then begin
      check = 0
    endif
    if ( arrowend_opt[1] eq 1 ) and ( tickv[i_ticks] gt levels[n_levels-2] ) $
        then check = 0
    if check eq 1 then begin
      ytick = ( tickv[i_ticks] - min( levels ) ) $
          / ( max( levels ) - min( levels ) ) * ydim
      ; Plot tick marks to the left
      if ticklen gt 0 then begin
        oplot, [xpos[0]-xdim*ticklen,xpos[0]], ypos[0]+ytick*[1,1], $
            color=color, thick=thick, noclip=1
      ; Plot tick marks to the right
      endif else if ticklen lt 0 then begin
        oplot, [xpos[1],xpos[1]-xdim*ticklen], ypos[0]+ytick*[1,1], $
            color=color, thick=thick, noclip=1
      endif
      ; Print tick names to the left
      if ticklen ge 0 then begin
        xyouts, xpos[0]-1.0*ticklen*xdim-0.5*xcharshift*charsize, $
            ypos[0]+ytick-0.5*ycharshift*charsize, tickname[i_ticks], $
            charsize=charsize, color=color, alignment=1, width=width, font=font
      ; Print tick names to the right
      endif else begin
        xyouts, xpos[1]-1.0*ticklen*xdim+0.5*xcharshift*charsize, $
            ypos[0]+ytick-0.5*ycharshift*charsize, tickname[i_ticks], $
            charsize=charsize, color=color, alignment=0, width=width, font=font
      endelse
      ; Record tick label width if this is the widest label
      if width gt xwidth then xwidth = width
    endif
  endfor
  ; Copy legend width for subtitle positioning purposes
  if keyword_set( subtitle ) then begin
    xwidth = ( convert_coord( [width+origin[0]], [0], normal=1, to_data=1 ) )[0]
  endif
endif

; Horizontal ticks and tick names
if horizontal_opt then begin
  for i_ticks = 0, n_ticks - 1 do begin
    ; Only do this if not an arrowed end
    check = 1
    if ( arrowend_opt[0] eq 1 ) and ( tickv[i_ticks] lt levels[1] ) then begin
      check = 0
    endif
    if ( arrowend_opt[1] eq 1 ) and ( tickv[i_ticks] gt levels[n_levels-2] ) $
        then check = 0
    if check eq 1 then begin
      xtick = ( tickv[i_ticks] - min( levels ) ) $
          / ( max( levels ) - min( levels ) ) * xdim
      ; Plot tick marks below
      if ticklen gt 0 then begin
        oplot, xpos[0]+xtick*[1,1], [ypos[0]-ydim*ticklen,ypos[0]], $
            color=color, thick=thick, noclip=1
      ; Plot tick marks above
      endif else if ticklen lt 0 then begin
        oplot, xpos[0]+xtick*[1,1], [ypos[1],ypos[1]-ydim*ticklen], $
            color=color, thick=thick, noclip=1
      endif
      ; Print tick names below
      if ticklen ge 0 then begin
        xyouts, xpos[0]+xtick, ypos[0]-ticklen*ydim-1.5*ycharshift*charsize, $
            tickname[i_ticks], charsize=charsize, color=color, alignment=0.5, $
            font=font
      ; Print tick names above
      endif else begin
        xyouts, xpos[0]+xtick, ypos[1]-ticklen*ydim+0.5*ycharshift*charsize, $
            tickname[i_ticks], charsize=charsize, color=color, alignment=0.5, $
            font=font
      endelse
    endif
  endfor
endif

;***********************************************************************
; Titles

; Main title
if keyword_set( title ) then begin
  ; Determine the horizontal position
  if tit_alignment eq 0 then begin
    temp_xpos = xpos[0]
  endif else if tit_alignment eq 1 then begin
    temp_xpos = xpos[1]
  endif else begin
    temp_xpos = xpos[0] + xdim / 2.
  endelse
  ; Print the title
  xyouts, temp_xpos[0], ypos[1]+ycharshift*tit_charsize, title, $
      charsize=tit_charsize, alignment=tit_alignment, color=color, font=font
endif

; Units title
if keyword_set( subtitle ) then begin
  ; For a vertical legend
  if vertical_opt then begin
    ; For tick labels (and units title) on the left side
    if ticklen ge 0 then begin
      ; Print units title
      xyouts, xpos[0]-1.5*xdim*ticklen-xwidth-0.5*xcharshift*charsize, $
          ypos[0]+ydim/2., subtitle, charsize=charsize, alignment=0.5, $
          orientation=90, color=color, font=font
    ; For tick labels (and units title) on the right side
    endif else begin
      ; Print units title
      xyouts, xpos[1]-1.5*xdim*ticklen+xwidth+0.5*xcharshift*charsize, $
          ypos[0]+ydim/2., subtitle, charsize=charsize, alignment=0.5, $
          orientation=270, color=color, font=font
    endelse
  endif
  ; For a horizontal legend
  if horizontal_opt then begin
    ; For tick labels (and units title) below
    if ticklen ge 0 then begin
      ; Print units title
      xyouts, xpos[0]+xdim/2., ypos[0]-ticklen*ydim-3*charsize*ycharshift, $
          subtitle, charsize=charsize, alignment=0.5, color=color, font=font
    ; For tick labels (and units title) above
    endif else begin
      ; Print units title
      xyouts, xpos[0]+xdim/2., ypos[1]-ticklen*ydim+2*charsize*ycharshift, $
          subtitle, charsize=charsize, alignment=0.5, color=color, font=font
    endelse
  endif
endif

;***********************************************************************
; The End

return
END
