;+
; NAME:
;    breakline
;
; PURPOSE:
;    This function breaks a line of text into pieces less than or equal to a 
;    maximum length.
;
; CATEGORY:
;    Miscellaneous
;
; CALLING SEQUENCE:
;    result = breakLine( LINE, MAXLENGTH )
;
; INPUTS:
;    IN_LINE:  A string of text.
;    MAXLEN:  An integer specifying the maximum length of the substrings.
;    SEPARATOR
;
; KEYWORD PARAMETERS:
;    SEPARATOR:  An optional scalar string specifying the character to split 
;       lines at.  The default is the space character.
;
; OUTPUTS:
;    RESULT:  A string array containing the substrings.
;
; EXAMPLE:
;    result = rreakLine('The quick brown fox jumped over the lazy dog', 15)
;
; MODIFICATION HISTORY:
;    Written by:  Edward C. Wiebe, 2002-05-08.
;    Modified:  Daithi A. Stone (dastone@runbox.com), 2024-01-15 (
;        Re-wrote algorithm;  Added SEPARATOR keyword input;  Re-named LINE 
;        input to IN_LINE;  Updated format)
;-

;***********************************************************************

FUNCTION BREAKLINE, $
    IN_LINE, MAXLEN, $
    SEPARATOR=separator

;***********************************************************************
; Constants and options

; The length of the string
in_len = strlen( in_line )
if in_len eq 0 then return, ''

; The default character for line separation
if not( keyword_set( separator ) ) then separator = ' '

;***********************************************************************
; Split string into lines

; Split line at SEPARATOR characters
in_seg = strsplit( in_line, separator, count=n_in_seg, extract=1 )
in_seg_len = strlen( in_seg )

; Iterate until done
flag_done = 0
id_seg = 0
while flag_done eq 0 do begin
  ; Determine the maximum length for this line
  temp_len = total( in_seg_len[id_seg:n_in_seg-1], cumulative=1 )
  id_max = max( where( temp_len le maxlen, n_id_max ) )
  ; If this bit cannot be split that small, then just take the first segment
  if n_id_max eq 0 then id_max = 0
  ; Copy this line to output
  temp = strjoin( in_seg[id_seg:id_seg+id_max], separator )
  if keyword_set( result ) then begin
    result = [ result, temp ]
  endif else begin
    result = temp
  endelse
  ; Iterate to next part of input line
  id_seg = id_seg + id_max + 1
  if id_seg ge n_in_seg then flag_done = 1
endwhile

;***********************************************************************
; ???OLD STUFF

if 1 eq 2 then begin
; We cannot split segments longer than MAXLEN
if ( maxlen lt max( in_seg_len ) ) then begin
  temp_message = "The line can't be broken with maxlen=" $
      + strtrim( maxlen, 2 ) + '.  It requires at least maxlen=' $
      + strtrim( max( len ), 2 ) + '.'
  message, /INFO, temp_message
  return, strarr( num ) + '-1'
endif

; Initialise the output string
places = num + ( rem ne 0 )
result = StrArr( places )
endif

if 1 eq 2 then begin
; Split the input string into multiple lines
sz = Size( sline )
cnt = sz[1]
done = 0
n = -1
m = 0
check = 0
r = 0
while (not done) do begin
  nn = 0
  l  = ''
  check = check + 1
  while ( nn lt maxlen ) and ( n lt cnt - 1 ) do begin
    n = n + 1
    nn = nn + len[n] + ( n ne m )
  endwhile  
  if ( nn gt maxlen ) then begin
    n = n - 1
    nn = nn - len[n]
  endif
  for b = m, n do begin
    l = l + StrMid( line, sline[b], len[b] )
    if ( b ne n ) then  l = l + ' '
  endfor
  if ( r eq places ) then begin 
    places = places + 1
    tmp = result
    result = StrArr( places )
    result[0:r-1] = tmp
  endif
  result[r] = l
  r = r + 1
  m  = n + 1
  nn = 0
  if ( n eq cnt - 1 ) or ( check gt length ) then done = 1     
endwhile
endif

;***********************************************************************
; The End
  
Return, result
END

