;+
; NAME:
;    regiondna_conf_define_source.pro
;
; PURPOSE:
;    This procedure defines input data sets to be used by regiondna_conf.pro.
;
; CATEGORY:
;    REGIONAL D-AND-A CONFIDENCE
;
; CALLING SEQUENCE:
;    regiondna_conf_define_source, scenario=scenario, var_label=var_label
;
; INPUTS:
;    SCENARIO, SELECT_VALUES, VAR_LABEL
;
; KEYWORD PARAMETERS:
;    MASK_FILE:  Returns a string vector of length N_SOURCE listing the file 
;        name of the land-sea mask file for each data source.
;    MASK_VAR_LABEL:  Returns a string vector of length N_SOURCE listing the 
;        label of the variable in each MASK_FILE entry to be used as the mask.  
;        The default is 'sftlf' (land percentage).
;    RESOLUTION:  Returns a float array of size 2,N_SOURCE listing the 
;        resolution in longitude and latitude (respectively in the first 
;        dimension) for each data source., in units of number of grid cells per 
;        global grid.
;    SCENARIO:  A required string vector listing the various scenarios to 
;        load.  Of length N_SCENARIO.  These scenarios are also used as 
;        criteria for selecting sources (i.e. the sources must have output for 
;        all of these scenarios).
;    SELECT_VALUES:  An optional string vector specifying various criteria for 
;        selection of data sources.  Note the SCENARIO criteria are included 
;        automatically and so do not need to be included here.  See 
;        markup_read.pro for format details.
;    SOURCE_LABEL:  Returns a string vector of length N_SOURCE listing the 
;        source labels.
;    VAR_FILE:  Returns a string array listing the files containing the data 
;        from the various sources.  Of size N_SCENARIO,N_SOURCE.  Each entry is 
;        a comma-delimited list of files, which may include wildcards.
;    VAR_LABEL:  A required scalar string specifying the climate variable for 
;        the requested data.  Supported values are:  'pr', 'tas', 'tos'.
;
; OUTPUTS:
;    MASK_FILE, MASK_VAR_LABEL, RESOLUTION, VAR_FILE
;
; USES:
;    regiondna_define_source.pro
;
; PROCEDURE:
;    This procedure is a wrapper for converting table output from 
;    regiondna_define_source.pro into separate outputs for the various 
;    parameters needed by regiondna_conf.pro.
;
; EXAMPLE:
;    regiondna_conf_define_source, scenario=['hist-all','hist-nat'], $
;        var_label='tas'
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2014-12-23 (As 
;        hanseng_conf_assess.pro)
;    Modified:  DAS, 2020-09-10 (Extracted and adapted from 
;        hanseng_conf_assess.pro)
;    Modified:  DAS, 2021-01-07 (Added SOURCE_LABEL keyword output)
;-

;***********************************************************************

PRO REGIONDNA_CONF_DEFINE_SOURCE, $
    SELECT_VALUES=select_values, $
    SCENARIO=scenario, $
    VAR_LABEL=var_label, $
    MASK_FILE=mask_file, MASK_VAR_LABEL=mask_var_label, RESOLUTION=resolution, $
        SOURCE_LABEL=source_label, VAR_FILE=var_file

;***********************************************************************
; Constants and options

; The number of requested scenarios
n_scenario = n_elements( scenario )
if n_scenario eq 0 then stop

; The requested variable
n_var = n_elements( var_label )
if n_var ne 1 then stop

;***********************************************************************
; Determine source information

; Get source tables for selected criteria
temp_options = scenario + '_file\='
if keyword_set( select_values ) then begin
  temp_options = [ temp_options, select_values ]
endif
regiondna_define_source, select_values=temp_options, var_label=var_label, $
    source_headers=source_headers, source_settings=source_settings
n_source = n_elements( source_settings[0,*] )

; Extract mask (probably land-sea) file names for selected sources
id = where( source_headers eq 'file_sftlf', n_id )
if n_id eq 0 then begin
  mask_file = strarr( n_source )
endif else if n_id gt 1 then begin
  stop
endif else begin
  mask_file = reform( source_settings[id[0],*] )
endelse

; Extract mask variable labels for selected sources
if max( strlen( mask_file ) ) gt 0 then begin
  id = where( source_headers eq 'var_sftlf', n_id )
  if n_id eq 0 then begin
    mask_var_label = strarr( n_source )
  endif else if n_id gt 1 then begin
    stop
  endif else begin
    mask_var_label = reform( source_settings[id[0],*] )
  endelse
  ; Assume missing entries are 'sftlf'
  id = where( mask_var_label eq '', n_id )
  if n_id gt 0 then mask_var_label[id] = 'sftlf'
  ; Convert 'N/A' values to '' (none)
  id = where( mask_var_label eq 'N/A', n_id )
  if n_id gt 0 then mask_var_label[id] = ''
endif

; Initialise array of data file lists for selected sources and scenarios
var_file = strarr( n_scenario, n_source )
; Iterate through scenarios
for i_scenario = 0, n_scenario - 1 do begin
  ; Extract data file lists for this scenario
  id = where( source_headers eq scenario[i_scenario] + '_file', n_id )
  if n_id ne 1 then stop
  var_file[i_scenario,*] = source_settings[id[0],*]
endfor

; Extract spatial resolution information
if max( var_label eq [ 'pr', 'tas' ] ) eq 1 then begin
  id = where( source_headers eq 'resolution_atmos', n_id )
  if n_id eq 0 then stop
  resolution = reform( source_settings[id[0],*] )
endif else if var_label eq 'tos' then begin
  id = where( source_headers eq 'resolution_ocean', n_id )
  if n_id eq 0 then stop
  resolution = reform( source_settings[id[0],*] )
endif else begin
  ; Variable not yet supported
  stop
endelse
; Parse resolution
resolution = [ reform( resolution, 1, n_source ),strarr( 1, n_source ) ]
for i_source = 0, n_source - 1 do begin
  temp = strsplit( resolution[0,i_source], '*', extract=1, count=n_temp )
  if n_temp ne 2 then stop
  resolution[*,i_source] = strtrim( temp, 2 )
endfor
resolution = float( resolution )

; Extract source labels
id = where( source_headers eq 'source_label', n_id )
if n_id ne 1 then stop
source_label = reform( source_settings[id[0],*] )

;***********************************************************************
; The end

return
END
