;+
; NAME:
;    moisture_flux
;
; PURPOSE:
;    This function calculates the column-integrated moisture flux convergence 
;    in a longitude, latitude, pressure-level, time dimensioned data set.
;
; CATEGORY:
;    Graphics
;
; CALLING SEQUENCE:
;    QFLUX_DATA = moisture_flux( )
;
; INPUTS:
;    HUS_DATA, HUS_FILE, HUS_LAT, HUS_LON, HUS_PLEV, PLEV_MAX, PLEV_MIN, 
;    PS_DATA, PS_FILE, PS_LON, PS_LAT, QFLUX_FILE, UA_DATA, UA_FILE, UA_LAT, 
;    UA_LON, UVA_PLEV, VA_DATA, VA_FILE, VA_LAT, VA_LON
;
; KEYWORD PARAMETERS:
;    FLUX_ONLY:  If set, the function calculates the column integrates moisture 
;        flux Q*WIND instead of the default moisture flux convergence.
;    HUS_DATA:  A N_HUS_LON,N_HUS_LAT,N_HUS_PLEV,N_HUS_TIME float array 
;        containing the specific humidity data.  Either this or HUS_FILE is 
;        required.  HUS_DATA requires that HUS_LAT, HUS_LON, and HUS_PLEV are 
;        also input.  In units of "kg/kg" (kg water per kg air).
;    HUS_FILE:  A string vector listing the names of the files from which to 
;        load the specific humidity data.  Either this or HUS_DATA is required.
;    HUS_LAT:  A float vector of length N_HUS_LAT containing the latitude 
;        dimension for HUS_DATA.  Required if HUS_DATA is input.
;    HUS_LON:  A float vector of length N_HUS_LON containing the longitude 
;        dimension for HUS_DATA.  Required if HUS_DATA is input.
;    HUS_PLEV:  A float vector of length N_HUS_PLEV containing the pressure 
;        level dimension for HUS_DATA.  In units of "Pa".  It should be in 
;        increasing order i.e. from the top down.
;    PLEV_MAX:  An optional scalar float specifying the maximum boundary on a 
;        pressure band to which to restrict all calculations.  In units of 
;        "Pa".  This is ignored if PS_DATA or PS_FILE is input.
;    PLEV_MIN:  An optional scalar float specifying the minimum boundary on a 
;        pressure band to which to restrict all calculations.  In units of 
;        "Pa".
;    PS_DATA:  A N_PS_LON,N_PS_LAT,N_PS_TIME float array containing the 
;        surface pressure data.  Only one of PS_DATA and PS_FILE can be 
;        input.  PS_DATA requires that PS_LAT and PS_LON are also input.  
;        In units of "Pa".
;    PS_FILE:  A string vector listing the names of the files from which to 
;        load the surface pressure data.  Only one of PS_DATA and PS_FILE 
;        can be input.
;    PS_LAT:  A float vector of length N_PS_LAT containing the latitude 
;        dimension for PS_DATA.  Required if PS_DATA is input.  Currently 
;        PS_LAT must equal HUS_LAT, i.e. no spatial interpolation is 
;        implemented yet.
;    PS_LON:  A float vector of length N_PS_LON containing the longitude 
;        dimension for PS_DATA.  Required if PS_DATA is input.  Currently 
;        PS_LAT must equal HUS_LAT, i.e. no spatial interpolation is 
;        implemented yet.
;    QFLUX_FILE:  An scalar string specifying the NetCDF file into which to 
;        write the moisture flux output.
;    INSTANTANEOUS_TO_AVERAGE:  If set, then the code understands that the 
;        input data is instantaneous but it should be coverted to time-step 
;        averages.
;    UA_DATA:  A N_UA_LON,N_UA_LAT,N_UA_PLEV,N_UA_TIME float array containing 
;        the zonal wind data.  Either this or UA_FILE is required.  UA_DATA 
;        requires that UA_LAT, UA_LON, and UVA_PLEV are also input.  In units 
;        of "m/s".
;    UA_FILE:  A string vector listing the names of the files from which to 
;        load the zonal wind data.  Either this or UA_DATA is required.
;    UA_LAT:  A float vector of length N_UA_LAT containing the latitude 
;        dimension for UA_DATA.  Required if UA_DATA is input.
;    UA_LON:  A float vector of length N_UA_LON containing the longitude 
;        dimension for UA_DATA.  Required if UA_DATA is input.
;    UVA_PLEV:  A float vector of length N_UVA_PLEV containing the pressure 
;        level dimension for UVA_DATA.  Currently UVA_PLEV must equal HUS_PLEV, 
;        i.e. no vertical interpolation is incorporated yet.  In units of "Pa".
;    VA_DATA:  A N_VA_LON,N_VA_LAT,N_VA_PLEV,N_VA_TIME float array containing 
;        the meridional wind data.  Either this or VA_FILE is required.  
;        VA_DATA requires that VA_LAT, VA_LON, and UVA_PLEV are also input.  In 
;        units of "m/s".
;    VA_FILE:  A string vector listing the names of the files from which to 
;        load the meridional wind data.  Either this or VA_DATA is required.
;    VA_LAT:  A float vector of length N_VA_LAT containing the latitude 
;        dimension for VA_DATA.  Required if VA_DATA is input.
;    VA_LON:  A float vector of length N_VA_LON containing the longitude 
;        dimension for VA_DATA.  Required if VA_DATA is input.
;
; OUTPUTS:
;    QFLUX_DATA:  Returns a N_HUS_LON,N_HUS_LAT,N_HUS_TIME float array 
;        containing the column-integrated horizontal moisture flux into each 
;        grid cell.  In units of "kg/m^2/s".
;
; USES:
;    add_dim.pro
;    constants.pro
;    netcdf_read_geo_multitime.pro
;    netcdf_write.pro
;
; PROCEDURE:
;    This function calculates the column-integrated, per m^2 flux of water 
;    mass moving into/out of each grid cell in a longitude-latitude gridded 
;    data product.  See P. C. Banacos and D. M. Schultz (2005, "Moisture flux 
;    convergence: its history and application in convective initiation 
;    forecasting", https://doi.org/10.1175/WAF858.1).
;
; EXAMPLE:
;    Fully advect one parcel of pure water vapour one 1-degree grid cell to the 
;    west.
;      hus_data = [ [ 0., 0., 0., 0. ], [ 0., 0., 1., 0. ], [ 0., 0., 0., 0. ] ]
;      hus_lon = findgen( 4. )
;      hus_lat = findgen( 3. ) - 1.
;      hus_plev = [ 100000. ]
;      ua_data = 111000. + fltarr( 4, 3 )
;      ua_lon = hus_lon
;      ua_lat = hus_lat
;      va_data = 0. + fltarr( 4, 3 )
;      va_lon = ua_lon
;      va_lat = ua_lat
;      result = moisture_flux( hus_data=hus_data, hus_lon=hus_lon, $
;          hus_lat=hus_lat, hus_plev=hus_plev, ua_data=ua_data, ua_lon=ua_lon, $
;          ua_lat=ua_lat, va_data=va_data, va_lon=va_lon, va_lat=va_lat, $
;          uva_plev=hus_plev )
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2024-01-05
;-

;***********************************************************************

FUNCTION MOISTURE_FLUX, $
    HUS_DATA=hus_data, HUS_LAT=hus_lat, HUS_LON=hus_lon, HUS_PLEV=hus_plev, $
      HUS_TIME=hus_time, $
    HUS_FILE=hus_file, $
    PLEV_MAX=plev_max, plev_min=plev_min, $
    PS_DATA=ps_data, PS_LAT=ps_lat, PS_LON=ps_lon, $
    PS_FILE=ps_file, $
    QFLUX_FILE=qflux_file, $
    UA_DATA=ua_data, UA_LAT=ua_lat, UA_LON=ua_lon, $
    UA_FILE=ua_file, $
    UVA_PLEV=uva_plev, $
    VA_DATA=va_data, VA_LAT=va_lat, VA_LON=va_lon, $
    VA_FILE=va_file, $
    FLUX_ONLY=flux_only_opt, $
    INSTANTANEOUS_TO_AVERAGE=instantaneous_to_average_opt

;***********************************************************************
; Constants, inputs, and options

; Ensure legal combination of inputs
if keyword_set( hus_data ) + keyword_set( hus_file ) ne 1 then stop
if keyword_set( ua_data ) + keyword_set( ua_file ) ne 1 then stop
if keyword_set( va_data ) + keyword_set( va_file ) ne 1 then stop

; Various constants
constants, g_earth=g_earth, r_earth=r_earth
circum_earth = 2. * !pi * r_earth
degrad = !pi / 180.

; The default minimum pressure level
if n_elements( plev_min ) eq 0 then plev_min = 0.

; The option to just calculate the moisture flux
flux_only_opt = keyword_set( flux_only_opt )

;***********************************************************************
; Read data from file

; If we need to read humidity data from file
if keyword_set( hus_file ) then begin
  ; Read data from NetCDF file
  hus_data = netcdf_read_geo_multitime( hus_file, 'hus', lon=hus_lon, $
      lat=hus_lat, time=hus_time, units_time=hus_time_units, $
      calendar=hus_time_calendar, height=hus_plev )
  ; Get/check dimensions
  n_hus_lon = n_elements( hus_lon )
  n_hus_lat = n_elements( hus_lat )
  n_hus_plev = n_elements( hus_plev )
  n_hus_time = n_elements( hus_time )
  ; Reform to standard format
  hus_data = reform( hus_data, n_hus_lon, n_hus_lat, n_hus_plev, n_hus_time )
; Otherwise record data dimensions
endif else begin
  ; Get/check dimensions
  temp = size( hus_data, dimensions=1 )
  n_temp = n_elements( temp )
  n_hus_lon = temp[0]
  if n_hus_lon ne n_elements( hus_lon ) then stop
  if not( keyword_set( hus_lon ) ) then stop
  n_hus_lat = temp[1]
  if n_hus_lat ne n_elements( hus_lat ) then stop
  if not( keyword_set( hus_lat ) ) then stop
  if n_temp ge 3 then begin
    n_hus_plev = temp[2]
  endif else begin
    n_hus_plev = 1
  endelse
  if n_hus_plev ne n_elements( hus_plev ) then stop
  if not( keyword_set( hus_plev ) ) then stop
  if n_temp eq 4 then begin
    n_hus_time = temp[3]
  endif else begin
    n_hus_time = 1
  endelse
endelse
; Confirm pressure levels are in ascending order
if n_hus_plev gt 1 then begin
  if min( hus_plev[1:n_hus_plev-1] - hus_plev[0:n_hus_plev-2] ) lt 0 then stop
endif

; If we need to read surface pressure data from file
if keyword_set( ps_file ) then begin
  ; Read data from NetCDF file
  ps_data = netcdf_read_geo_multitime( ps_file, 'ps', lon=ps_lon, $
      lat=ps_lat, time=ps_time, units_time=ps_time_units, $
      calendar=ps_time_calendar )
  ; Get/check dimensions
  n_ps_lon = n_elements( ps_lon )
  n_ps_lat = n_elements( ps_lat )
  n_ps_time = n_elements( ps_time )
  ; Reform to standard format
  ps_data = reform( ps_data, n_ps_lon, n_ps_lat, n_ps_time )
; Otherwise record data dimensions
endif else if keyword_set( ps_data ) then begin
  ; Get/check dimensions
  temp = size( ps_data, dimensions=1 )
  n_temp = n_elements( temp )
  n_ps_lon = temp[0]
  if keyword_set( ps_lon ) then begin
    if n_ps_lon ne n_elements( ps_lon ) then stop
  endif
  n_ps_lat = temp[1]
  if keyword_set( ps_lat ) then begin
    if n_ps_lat ne n_elements( ps_lat ) then stop
  endif
  if n_temp eq 2 then begin
    n_ps_time = temp[2]
  endif else begin
    n_ps_time = 1
  endelse
endif
; Confirm pressure is consistent with humidity
if keyword_set( ps_data ) then begin
  if n_ps_lon ne n_hus_lon then stop
  if keyword_set( ps_lon ) then begin
    if max( abs( ps_lon - hus_lon ) ) ne 0 then stop
  endif
  if n_ps_lat ne n_hus_lat then stop
  if keyword_set( ps_lat ) then begin
    if max( abs( ps_lat - hus_lat ) ) ne 0 then stop
  endif
  if n_ps_time ne n_hus_time then stop
  if keyword_set( ps_time_units ) + keyword_set( hus_time_units ) eq 2 $
      then begin
    if ps_time_units ne hus_time_units then stop
    if ps_time_calendar ne hus_time_calendar then stop
    if max( abs( ps_time - hus_time ) ) ne 0 then stop
  endif
; Flag lack of surface pressure data
endif else begin
  n_ps_lon = 0
endelse

; If we need to read zonal wind data from file
if keyword_set( ua_file ) then begin
  ; Read data from NetCDF file
  ua_data = netcdf_read_geo_multitime( ua_file, 'ua', lon=ua_lon, lat=ua_lat, $
      time=uva_time, units_time=uva_time_units, calendar=uva_time_calendar, $
      height=uva_plev )
  ; Get/check dimensions
  n_ua_lon = n_elements( ua_lon )
  n_ua_lat = n_elements( ua_lat )
  n_uva_plev = n_elements( uva_plev )
  n_uva_time = n_elements( uva_time )
  ; Reform to standard format
  ua_data = reform( ua_data, n_ua_lon, n_ua_lat, n_uva_plev, n_uva_time )
; Otherwise record data dimensions
endif else begin
  ; Get/check dimensions
  temp = size( ua_data, dimensions=1 )
  n_temp = n_elements( temp )
  n_ua_lon = temp[0]
  if n_ua_lon ne n_elements( ua_lon ) then stop
  if not( keyword_set( ua_lon ) ) then stop
  n_ua_lat = temp[1]
  if n_ua_lat ne n_elements( ua_lat ) then stop
  if not( keyword_set( ua_lat ) ) then stop
  if n_temp ge 3 then begin
    n_uva_plev = temp[2]
  endif else begin
    n_uva_plev = 1
  endelse
  if ( n_uva_plev ne n_elements( uva_plev ) ) $
      and ( n_elements( uva_plev ) ne 0 ) then stop
  if not( keyword_set( uva_plev ) ) then stop
  if n_temp eq 4 then begin
    n_uva_time = temp[3]
  endif else begin
    n_uva_time = 1
  endelse
endelse
  
; If we need to read zonal wind data from file
if keyword_set( va_file ) then begin
  ; Read data from NetCDF file
  va_data = netcdf_read_geo_multitime( va_file, 'va', lon=va_lon, lat=va_lat, $
      time=va_time, units_time=va_time_units, calendar=va_time_calendar, $
      height=va_plev )
  ; Get/check dimensions
  n_va_lon = n_elements( va_lon )
  n_va_lat = n_elements( va_lat )
  if n_elements( va_plev ) ne n_uva_plev then stop
  if not( keyword_set( va_plev ) ) then stop
  if max( abs( va_plev - uva_plev ) ) ne 0 then stop
  if n_elements( va_time ) ne n_uva_time then stop
  if n_elements( uva_time ) ne 0 then begin
    if va_time_calendar ne uva_time_calendar then stop
    if va_time_units ne uva_time_units then stop
    if max( abs( va_time - uva_time ) ) ne 0 then stop
  endif
  ; Reform to standard format
  va_data = reform( va_data, n_va_lon, n_va_lat, n_uva_plev, n_uva_time )
; Otherwise record data dimensions
endif else begin
  ; Get/check dimensions
  temp = size( va_data, dimensions=1 )
  n_temp = n_elements( temp )
  n_va_lon = temp[0]
  if n_va_lon ne n_elements( va_lon ) then stop
  if not( keyword_set( va_lon ) ) then stop
  n_va_lat = temp[1]
  if n_va_lat ne n_elements( va_lat ) then stop
  if not( keyword_set( va_lat ) ) then stop
endelse

; Confirm matching pressure dimensions
if n_uva_plev ne n_hus_plev then stop
if keyword_set( uva_plev ) and keyword_set( hus_plev ) then begin
  if max( abs( uva_plev - hus_plev ) ) ne 0 then stop
endif else if keyword_set( hus_plev ) then begin
  uva_plev = hus_plev
endif else if keyword_set( uva_plev ) then begin
  hus_plev = uva_plev
endif else begin
  stop
endelse
; Confirm matching time dimensions
if n_uva_time ne n_hus_time then stop
if keyword_set( uva_time ) and keyword_set( hus_time ) then begin
  if uva_time_calendar ne hus_time_calendar then stop
  if uva_time_units ne hus_time_units then stop
  if max( abs( uva_time - hus_time ) ) ne 0 then stop
endif

; Ensure regular grid
d_hus_lon = hus_lon[1:n_hus_lon-1] - hus_lon[0:n_hus_lon-2]
if min( d_hus_lon ) ne max( d_hus_lon ) then stop
d_hus_lon = d_hus_lon[0]
d_hus_lat = hus_lat[1:n_hus_lat-1] - hus_lat[0:n_hus_lat-2]
;if min( d_hus_lat ) ne max( d_hus_lat ) then stop
d_hus_lat = d_hus_lat[0]
d_ua_lon = ua_lon[1:n_ua_lon-1] - ua_lon[0:n_ua_lon-2]
if min( d_ua_lon ) ne max( d_ua_lon ) then stop
d_ua_lon = d_ua_lon[0]
d_ua_lat = ua_lat[1:n_ua_lat-1] - ua_lat[0:n_ua_lat-2]
;if min( d_ua_lat ) ne max( d_ua_lat ) then stop
d_ua_lat = d_ua_lat[0]
d_va_lon = va_lon[1:n_va_lon-1] - va_lon[0:n_va_lon-2]
if min( d_va_lon ) ne max( d_va_lon ) then stop
d_va_lon = d_va_lon[0]
d_va_lat = va_lat[1:n_va_lat-1] - va_lat[0:n_va_lat-2]
;if min( d_va_lat ) ne max( d_va_lat ) then stop
d_va_lat = d_va_lat[0]

;; Remove NaNs from wind field (but keep in humidity field)
;id = where( finite( ua_data ) eq 0, n_id )
;if n_id gt 0 then ua_data[id] = 0.
;id = where( finite( va_data ) eq 0, n_id )
;if n_id gt 0 then va_data[id] = 0.

;***********************************************************************
; Calculate moisture flux

; Determine the mass of the pressure levels
if n_hus_plev eq 1 then begin
  hus_plev_bound = [ 0, hus_plev ]
  hus_plev_mass = hus_plev / g_earth
endif else begin
  hus_plev_bound = alog( hus_plev )
  hus_plev_bound = [ hus_plev_bound, $
      hus_plev_bound[n_hus_plev-1] $
      + ( hus_plev_bound[n_hus_plev-1] - hus_plev_bound[n_hus_plev-2] ) ]
  hus_plev_bound = ( hus_plev_bound[0:n_hus_plev-1] $
      + hus_plev_bound[1:n_hus_plev] ) / 2.
  hus_plev_bound = [ 0., exp( hus_plev_bound ) ]
  hus_plev_mass = $
      ( hus_plev_bound[1:n_hus_plev] - hus_plev_bound[0:n_hus_plev-1] ) $
      / g_earth
endelse

; Calculate the hus dimensions as indices of the ua and va dimension vectors
hus_lon_ua = ( hus_lon - ua_lon[0] ) / d_ua_lon
hus_lat_ua = ( hus_lat - ua_lat[0] ) / d_ua_lat
hus_lon_va = ( hus_lon - va_lon[0] ) / d_ua_lon
hus_lat_va = ( hus_lat - va_lat[0] ) / d_ua_lat
; Define the half-point hus dimension vectors
hus2_lon = [ hus_lon[0] - d_hus_lon / 2., hus_lon + d_hus_lon / 2. ]
hus2_lat = [ hus_lat[0] - d_hus_lat / 2., hus_lat + d_hus_lat / 2. ]
; Calculate the half-point hus dimensions as indices of the hus_lon and hus_lat 
; vectors
hus2_lon_hus = findgen( n_hus_lon + 1 ) - 0.5
hus2_lat_hus = findgen( n_hus_lat + 1 ) - 0.5
; Calculate the half-point hus dimensions as indices of the ua and va dimension 
; vectors
hus2_lon_ua = ( hus2_lon - ua_lon[0] ) / d_ua_lon
hus2_lat_ua = ( hus2_lat - ua_lat[0] ) / d_ua_lat
hus2_lon_va = ( hus2_lon - va_lon[0] ) / d_ua_lon
hus2_lat_va = ( hus2_lat - va_lat[0] ) / d_ua_lat

; Initialise the moisture flux convergence array
qflux_data = fltarr( n_hus_lon, n_hus_lat, n_hus_time )

; Iterate through time
for i_time = 0, n_hus_time - 1 do begin
  ; Initialise the moisture flux convergence array
  temp_qflux = fltarr( n_hus_lon, n_hus_lat, n_hus_plev )
  id = where( finite( hus_data[*,*,*,i_time] ) eq 0, n_id )
  if n_id gt 0 then temp_qflux[id] = !values.f_nan
  ; Iterate through pressure level
  for i_plev = 0, n_hus_plev - 1 do begin
    ; If we are calculating the moisture flux
    if flux_only_opt eq 1 then begin
      ; Calculate the wind speed
      temp_ua = interpolate( ua_data[*,*,i_plev,i_time], hus_lon_ua, $
          hus_lat_ua, grid=1 )
      temp_va = interpolate( va_data[*,*,i_plev,i_time], hus_lon_va, $
          hus_lat_va, grid=1 )
      ; Calculate the moisture flux (units of "kg kg^-1 m s^-1)
      temp_qflux[*,*,i_plev] = temp_qflux[*,*,i_plev] $
          + hus_data[*,*,i_plev,i_time] * sqrt( temp_ua ^ 2. + temp_va ^ 2. )
    ; If we are calculating the moisture flux convergence
    endif else begin
      ; Calculate the west-east advection term
      ; (units "kg kg^-1 m s^-1 m^-1")
      temp_ua = interpolate( ua_data[*,*,i_plev,i_time], hus_lon_ua, $
          hus_lat_ua, grid=1 )
      temp_dhus_dx = interpolate( hus_data[*,*,i_plev,i_time], hus2_lon_hus, $
          findgen( n_hus_lat ), grid=1 )
      ;id = where( finite( temp_dhus_dx ) eq 0, n_id )
      ;if n_id gt 0 then temp_dhus_dx[id] = 0.
      temp_dhus_dx $
          = ( temp_dhus_dx[1:n_hus_lon,*] - temp_dhus_dx[0:n_hus_lon-1,*] ) $
          / ( d_hus_lon / 360. * circum_earth $
          * cos( add_dim( hus_lat, 0, n_hus_lon ) * degrad ) )
      temp_qflux[*,*,i_plev] = temp_qflux[*,*,i_plev] - temp_ua * temp_dhus_dx
      temp_ua = 0
      temp_dhus_dx = 0
      ; Calculate the south-north advection term
      ; (units "kg kg^-1 m s^-1 m^-1")
      temp_va = interpolate( va_data[*,*,i_plev,i_time], hus_lon_va, $
          hus_lat_va, grid=1 )
      temp_dhus_dy = interpolate( hus_data[*,*,i_plev,i_time], $
          findgen( n_hus_lon ), hus2_lat_hus, grid=1 )
      ;id = where( finite( temp_dhus_dy ) eq 0, n_id )
      ;if n_id gt 0 then temp_dhus_dy[id] = 0.
      temp_dhus_dy $
          = ( temp_dhus_dy[*,1:n_hus_lat] - temp_dhus_dy[*,0:n_hus_lat-1] ) $
          / ( d_hus_lat / 360. * circum_earth )
      temp_qflux[*,*,i_plev] = temp_qflux[*,*,i_plev] - temp_va * temp_dhus_dy
      temp_va = 0
      temp_dhus_dy = 0
      ; Calculate the west-east convergence term
      ; (units "kg kg^-1 m s^-1 m^-1")
      temp_hus = hus_data[*,*,i_plev,i_time]
      temp_dua_dx = interpolate( ua_data[*,*,i_plev,i_time], hus2_lon_ua, $
          hus_lat_ua, grid=1 )
      id = where( finite( temp_dua_dx ) eq 0, n_id )
      if n_id gt 0 then temp_dua_dx[id] = 0.
      temp_dua_dx $
          = ( temp_dua_dx[1:n_hus_lon,*] - temp_dua_dx[0:n_hus_lon-1,*] ) $
          / ( d_hus_lon / 360. * circum_earth $
          * cos( add_dim( hus_lat, 0, n_hus_lon ) * degrad ) )
      temp_qflux[*,*,i_plev] = temp_qflux[*,*,i_plev] - temp_hus * temp_dua_dx
      temp_dua_dx = 0
      ; Calculate the south-north convergence term
      ; (units "kg kg^-1 m s^-1 m^-1")
      temp_dva_dy = interpolate( va_data[*,*,i_plev,i_time], hus_lon_va, $
          hus2_lat_va, grid=1 )
      id = where( finite( temp_dva_dy ) eq 0, n_id )
      if n_id gt 0 then temp_dva_dy[id] = 0.
      temp_dva_dy $
          = ( temp_dva_dy[*,1:n_hus_lat] - temp_dva_dy[*,0:n_hus_lat-1] ) $
          / ( d_hus_lat / 360. * circum_earth )
      temp_qflux[*,*,i_plev] = temp_qflux[*,*,i_plev] - temp_hus * temp_dva_dy
      temp_hus = 0
      temp_dva_dy = 0
    endelse
  endfor
  ; Calculate the atmospheric mass at each level
  temp_mass = add_dim( add_dim( hus_plev_mass, 0, n_hus_lat ), 0, n_hus_lon )
  if n_ps_lon gt 0 then begin
    ; If surface pressure data is input, determine the weight on the lowest 
    ; level(s)
    for i_lat = 0, n_hus_lat - 1 do begin
      for i_lon = 0, n_hus_lon - 1 do begin
        id = where( hus_plev_bound[1:n_hus_plev] $
            gt ps_data[i_lon,i_lat,i_time], n_id )
        if n_id gt 0 then begin
          if n_id gt 1 then temp_mass[i_lon,i_lat,id[1:n_id-1]] = 0.
          temp_mass[i_lon,i_lat,id[0]] = ( ps_data[i_lon,i_lat,i_time] $
              - hus_plev_bound[id[0]] ) / g_earth
        endif
      endfor
    endfor
  endif else if keyword_set( plev_max ) then begin
    ; If a maximum pressure is specified, determine the weight on the lowest 
    ; level(s)
    id = where( hus_plev_bound[1:n_hus_plev] gt plev_max, n_id )
    if n_id gt 0 then begin
      if n_id gt 1 then temp_mass[*,*,id[1:n_id-1]] = 0.
      temp_mass[*,*,id[0]] = ( plev_max - hus_plev_bound[id[0]] ) / g_earth
    endif
  endif
  if keyword_set( plev_min ) then begin
    ; If a minimum pressure is specified, determine the weight on the lowest 
    ; level(s)
    id = where( hus_plev_bound[0:n_hus_plev-1] le plev_min, n_id )
    if n_id gt 0 then begin
      if n_id gt 1 then temp_mass[*,*,id[0:n_id-2]] = 0.
      temp_mass[*,*,id[n_id-1]] = ( hus_plev_bound[id[n_id-1]+1] - plev_min ) $
          / g_earth
    endif
  endif
  ; Calculate and record the total flux convergence (units "kg m^-2 s^-1")
  ;   temp_mass is in units of "Pa m^-1 s^2" = "kg m^-2"
  ;   temp_qflux is in units of "kg kg^-1 m s^-1 m^-1"
  ; If flux_only_opt=1 then temp_qflux is in units of "kg kg-1 m s-1", so 
  ; the units here are "kg m^-1 s^-1".
  if n_hus_plev eq 1 then begin
    qflux_data[*,*,i_time] = temp_mass * temp_qflux
  endif else begin
    qflux_data[*,*,i_time] = total( temp_mass * temp_qflux, 3, nan=1 )
  endelse
  temp_mass = 0
  temp_qflux = 0
endfor

; Copy input time vector to output
if keyword_set( hus_time ) then qflux_time = hus_time

; Convert instantaneous data to time-step averages
if keyword_set( instantaneous_to_average_opt ) then begin
  qflux_data $
      = ( qflux_data[*,*,0:n_hus_time-2] + qflux_data[*,*,1:n_hus_time-1] ) / 2.
  if keyword_set( qflux_time ) then begin
    qflux_time = ( qflux_time[0:n_hus_time-2] + qflux_time[1:n_hus_time-1] ) $
        / 2.
  endif
endif

;***********************************************************************
; Output to a NetCDF file

; If we are to output to NetCDF
if keyword_set( qflux_file ) then begin
  if n_elements( qflux_file ) ne 1 then stop

  ; Define data variable attributes
  qflux_attribute_label = [ 'standard_name', 'long_name', 'units' ]
  ; Define dimension variable attributes
  lon_attribute_label = [ 'standard_name', 'long_name', 'units', 'axis' ]
  lat_attribute_label = [ 'standard_name', 'long_name', 'units', 'axis' ]
  time_attribute_label = [ 'standard_name', 'long_name', 'units', $
      'axis', 'calendar' ]
  time_attribute_value = [ '', '', hus_time_units, 'axis', hus_time_calendar ]

  ; The variable label
  if flux_only_opt eq 1 then begin
    qflux_label = 'qflux'
  endif else begin
    qflux_label = 'qflux-converge'
  endelse

  ; Write to NetCDF file
  netcdf_write, qflux_file, data_array=qflux_data, data_label=qflux_label, $
      data_attribute_label=qflux_attribute_label, $
      dim1_vector=hus_lon, dim1_label='lon', $
      dim1_attribute_label=lon_attribute_label, $
      dim2_vector=hus_lat, dim2_label='lat', $
      dim2_attribute_label=lat_attribute_label, $
      dim3_vector=qflux_time, dim3_label='time', $
      dim3_attribute_label=time_attribute_label, $
      dim3_attribute_value=time_attribute_value, $
      driver_name='moisture_flux.pro'

endif


;***********************************************************************
; The End

return, qflux_data
END
