;+
; NAME:
;    greatest_div
;
; PURPOSE:
;    This function calculates greatest common divisor of two or more numbers.
;
; CATEGORY:
;    Mathematics
;
; CALLING SEQUENCE:
;    Result = greatest_div( VALUES )
;
; INPUTS:
;    VALUES:  A vector of two or more positive integers for which to find the 
;        greatest common divisor.
;
; OUTPUT:
;    Result:  A scalar positive integer that is the greatest common divisor of 
;        the numbers in VALUES.
;
; USES:
;    factors.pro
;    isin.pro
;
; PROCEDURE:
;    This function uses factors.pro to determine the factors of each of the  
;    numbers in VALUES, then removes factors not in common, then multiplies 
;    together all surviving factors.
;
; EXAMPLE:
;    ; Calculate the greatest common divisor of 48 and 60.
;    result = greatest_div( [ 48, 60 ] )
;    ; It should return 12.
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2025-03-05
;-

;***********************************************************************

FUNCTION GREATEST_DIV, $
    VALUES

;***********************************************************************
; Variables

; The number of input values
n_values = n_elements( values )
if n_values lt 2 then stop

; Ensure positive values
if min( values ) lt 1 then stop

; Order values in increasing order (greatest efficiency)
id = sort( values )
values_sorted = values[id]

;***********************************************************************
; Calculate the greatest common divisor

; Determine the factors (including 1) of the first value
values_factors = [ 1, factors( values_sorted[0] ) ]
n_factors = n_elements( values_factors )

; Iterate through remaining values
for i_values = 1, n_values - 1 do begin
  ; If 1 is the only factor then we are done
  if n_factors eq 1 then begin
    ; Return 1
    return, 1
  ; Otherwise let us continue
  endif else begin
    ; Determine the factors of this value
    temp_values_factors = [ 1, factors( values_sorted[i_values] ) ]
    ; Iterate through current factors
    for i_factors = 0, n_factors - 1 do begin
      ; Determine if this factor is in common
      id = where( temp_values_factors eq values_factors[i_factors], n_id )
      ; If so then remove it from contention from the current number
      if n_id gt 0 then begin
        temp_values_factors[id[0]] = -1
      ; Otherwise omit this factor
      endif else begin
        values_factors[i_factors] = -1
      endelse
    endfor
    ; Remove invalid factors
    id = where( values_factors gt 0, n_factors )
    values_factors = values_factors[id]
  endelse
endfor

; Multiply all of the surviving factors
result = product( values_factors, integer=1 )

;***********************************************************************
; The End

return, result
END
