;+
; NAME:
;    gpd_cdf
;
; PURPOSE:
;    This function calculates the cumulative distribution function of the 
;    Generalised Pareto distribution for a given set of parameters.
;
; CATEGORY:
;    Statistics
;
; CALLING SEQUENCE:
;    gpd_cdf, values, scale=scale, shape=shape
;
; INPUTS:
;    VALUES:  A required floating vector of length N_VALUES specifying the 
;        values at which to calculate the cumulative distribution function.  
;        If INVERT is set, then it should specify the quantiles at which to 
;        to calculate the corresponding positions.
;    SCALE, SHAPE
;
; KEYWORD PARAMETERS:
;    INVERT:  If set, then rather than calculating the quantiles for the 
;        specified positions, the function calculates the positions for the 
;        specified quantiles.
;    SCALE:  A required float scalar or N_VALUES vector specifying the 
;        scale parameter of the GP distribution.  If an N_VALUES vector, 
;        then each of the N_VALUES elements is matched with the corresponding 
;        element in the VALUES input.
;    SHAPE:  A required float scalar or N_VALUES vector specifying the 
;        shape parameter of the GP distribution.  If an N_VALUES vector, 
;        then each of the N_VALUES elements is matched with the corresponding 
;        element in the VALUES input.
;
; OUTPUTS:
;    RESULT:  A floating vector of length N_VALUES containing the cumulative 
;        distribution function for each values specified in VALUES.
;
; USES:
;    -
;
; PROCEDURE:
;    This function calculates the cumulative distribution function of the 
;    Generalised Pareto distribution analytically.
;
; EXAMPLE:
;    ; Plot the cumulative distribution function for a given set of parameters
;    values = findgen( 201.) / 100.
;    cdf = gpd_cdf( values, scale=0.30, shape=-0.2 )
;    plot, values, cdf
;
; LICENSE:
;    This code was written as part of the Extreme Weather Event Real-Time 
;    Attribution Machine (EWERAM) project supported by the New Zealand Ministry 
;    of Business, Innovation and Employment.  The code is free for use under 
;    the terms of the Creative Commons License v2.0 
;    (http://creativecommons.org/licenses/by-nc-sa/2.0/).
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2019-11-06
;    Modified:  DAS, 2020-05-12 (Added INVERT keyword option)
;-

;***********************************************************************

FUNCTION GPD_CDF, $
    VALUES, $
    SCALE=scale, SHAPE=shape, $
    INVERT=invert_opt

;***********************************************************************
; Constants and checks

; The number of values for which to calculate the CDF
n_values = n_elements( values )
if n_values eq 0 then stop

; Check the GPD parameters
n_scale = n_elements( scale )
if max( n_scale eq [ 1, n_values ] ) ne 1 then stop
if min( scale ) lt 0. then stop
n_shape = n_elements( shape )
if max( n_shape eq [ 1, n_values ] ) ne 1 then stop
; The maximum number of parameters
n_param_max = max( [ n_scale, n_shape ] )

; The option to invert the calculation (i.e. quantiles to positions)
invert_opt = keyword_set( invert_opt )

;***********************************************************************
; Calculate the CDF

; Copy parameters to temporary variables, making all consistent in length
if n_param_max eq 1 then begin
  temp_scale = scale[0]
  temp_shape = shape[0]
endif else begin
  if n_scale eq 1 then begin
    temp_scale = scale + fltarr( n_param_max )
  endif else begin
    temp_scale = scale
  endelse
  if n_shape eq 1 then begin
    temp_shape = shape + fltarr( n_param_max )
  endif else begin
    temp_shape = shape
  endelse
endelse

; Initialise the output scalar/vector
result = 0. * values

; The case when the shape parameter is zero
id = where( temp_shape eq 0., n_id )
if n_id gt 0 then begin
  ; Calculate the CDF
  if n_param_max eq 1 then begin
    if invert_opt eq 0 then begin
      result = 1. - exp( -values / temp_scale )
    endif else begin
      result = -temp_scale * alog( 1. - values )
    endelse
  endif else begin
    if invert_opt eq 0 then begin
      result[id] = 1. - exp( -values[id] / temp_scale[id] )
    endif else begin
      result[id] = -temp_scale[id] * alog( 1. - values[id] )
    endelse
  endelse
endif

; The case when the shape parameter is not zero
id = where( temp_shape ne 0., n_id )
if n_id gt 0 then begin
  ; Calculate the CDF
  if n_param_max eq 1 then begin
    if invert_opt eq 0 then begin
      result = 1. $
          - ( 1. + temp_shape * values / temp_scale ) ^ ( -1. / temp_shape )
    endif else begin
      result = ( ( 1 - values ) ^ ( -temp_shape ) - 1. ) * temp_scale $
          / temp_shape
    endelse
  endif else begin
    if invert_opt eq 0 then begin
      result[id] = 1. $
          - ( 1. + temp_shape[id] * values[id] / temp_scale[id] ) $
          ^ ( -1. / temp_shape[id] )
    endif else begin
      result[id] = ( ( 1 - values[id] ) ^ ( -temp_shape[id] ) - 1. ) $
          * temp_scale[id] / temp_shape[id]
    endelse
  endelse
endif

;***********************************************************************
; The end

return, result
END
