;+
; NAME:
;    ebm_tune
;
; PURPOSE:
;    This procedure tunes the parameters of an energy balance model in order to 
;    achieve an optimal fit with target data.
;
; CATEGORY:
;    Physics
;
; CALLING SEQUENCE:
;    ebm_tune, FORCE, TEMPERATURE, D_TIME=d_time, HEATCAP=heatcap, LAMBDA=lambda
;
; INPUTS:
;    FORCE:  A floating point vector time series of external forcing values, of 
;        length N_TIME.  This can also be an array of the form N_TIME*N_FORCE 
;        for running multiple models in parallel.
;    TEMPERATURE:  A floating point vector of length N_TIME containing the time 
;        series of temperature data to which to fit the EBM.
;
; KEYWORD PARAMETERS:
;    D_TIME:  The required time step of the data in FORCE and to be used for 
;        RESULT.  In units of years.
;    DOUBLE:  If set, calculations are done in double precision.  The default 
;        is single precision.
;    HEATCAP:  A required scalar or N_FORCE vector float containing the initial 
;        guess of the value(s) of the heat capacity of the mixed layer of the 
;        ocean, in units of W*a*m^-2*K^-1.  If a scalar is specified then the 
;        fitting process will consider that a single value applies to all 
;        N_FORCE responses.  Also returns the N_FORCE heat capacity parameter 
;        value(s) of the best-fit EBM.
;    KV:  An optional scalar or N_FORCE vector float containing the initial 
;        guess of the value(s) of the vertical diffusivity parameter, used in 
;        representing diffusion across the mixed layer into the deep ocean.  In 
;        units of W*K^-1.  Only used, and required, when 
;        MODEL_TYPE='diffusive'.  If a scalar is specified then the fitting 
;        process will consider that a single value applies to all N_FORCE 
;        responses.  Also returns the N_FORCE diffusivity parameter value(s) of 
;        the best-fit EBM.
;    LAMBDA:  A required scalar or N_FORCE vector float containing the 
;        initial guess of the climate sensitivity parameter, in units of 
;        W*K^-1*m^-2.  If a scalar is specified then the fitting process will 
;        consider that a single value applies to all N_FORCE responses.  Also 
;        returns the N_FORCE climate sensitivity parameter value(s) of the 
;        best-fit EBM.
;    MAX_ROUND:  An optional integer specifying the maximum number of 
;        interative rounds to take when solving for the parameter values.  The 
;        default is 100.
;    MODEL_TYPE:  An optional scalar string specifying the type of energy 
;        balance model to use.  Supported values are:
;        * 'heatcap lambda':  The simplest EBM, with only the heatcap and 
;          lambda parameters.  HEATCAP*dT/dt=FORCE-LAMBDA*T.
;        * 'diffusive':  Same as the 'heatcap lambda' model but with the 
;          addition of a simple representative of ocean diffusivity.  The KV 
;          input is required.  HEATCAP*dT/dt=FORCE-LAMBDA*T-KV*d2T/dz2.
;        The default is 'heatcap lambda'.
;    N_THRESHOLD_CHECK:  An optional float specifying the number of iterations 
;        that THRESHOLD must be satisfied before deciding that we have a final 
;        solution.  The default is 5.
;    OUTPUT_TEMPERATURE:  Returns a float vector of length N_TIME containing 
;        the temperature time series produced by the EBM with the tuned 
;        parameters.
;    SOLVER_D_TIME_FACTOR:  The optional scalar integer specifying the number 
;        of time steps used by the solver within each time step of FORCE and 
;        RECORD.  In other words, the time step used by the solver is 
;        D_TIME/SOLVER_D_TIME_FACTOR.  The default is 1.
;    SOLVER_TYPE:  An optional scalar string specifying the numerical scheme to 
;        use to integrate the model.  Supported options are:
;        * 'first_difference':  The first difference scheme.
;        The default is 'first difference'.
;    THRESHOLD:  An optional float specifying the threshold ratio between the 
;        model fit and TEMPERATURE for exiting the iterative solving loop and 
;        deciding that we have a solution.  The ratio needs to be satisfied 
;        through N_THRESHOLD_CHECK iterations in order to exit.  The default is 
;        1.d-6.
;
; OUTPUTS:
;    HEATCAP, KV, LAMBDA
;    OUTPUT_TEMPERATURE
;
; USES:
;    ebm_calc.pro
;
; PROCEDURE:
;    This procedure iteratively runs the energy balance model calculation by 
;    compariing the input TEMPERATURE against the temperature produced by the 
;    model with from the previous round and the temperature produced by that 
;    model with separate positive and negative perturbations to each of the 
;    parameters, and shrinking the size of the perturbations if none of the 
;    perturbed models produces a better fit.
;
; EXAMPLE:
;    ; Create an EBM model and run it responding differently to two forcings, 
;    ; also adding some noise
;    n_time = 100
;    force = [ [ sin( findgen( n_time ) / n_time * 2. * !pi ) / 2. ], [ findgen( n_time ) / n_time ] ]
;    temperature = ebm_calc( force, d_time=0.1, heatcap=1., lambda=[1.,5.], init=0., total=1 ) + randomn( 1, n_time ) * 0.1
;    ; Try to fit parameters to the temperature time series
;    heatcap = 3.
;    lambda = [ 3., 1. ]
;    output_temperature = 0.
;    ebm_tune, force, temperature, d_time=0.1, heatcap=heatcap, lambda=lambda, output_temperature=output_temperature
;    ; The fit should produce approximately 1.0 for the heat capacity parameter 
;    ; (for both forcings) and 1.0 and 5.0 for the climate sensitivity 
;    ; parameter for the two forcings respectively.

;
; MODIFICATION HISTORY:
;    Written by:  Daithi Stone (dastone@runbox.com), 2025-04-09
;-

;***********************************************************************

PRO EBM_TUNE, $
    FORCE, TEMPERATURE, $
    D_TIME=d_time, $
    HEATCAP=heatcap, KV=kv, LAMBDA=lambda, $
    MAX_ROUND=max_round, $
    MODEL_TYPE=model_type, $
    OUTPUT_TEMPERATURE=output_temperature, $
    SOLVER_D_TIME_FACTOR=solver_d_time_factor, SOLVER_TYPE=solver_type, $
    THRESHOLD=threshold, N_THRESHOLD_CHECK=n_threshold_check, $
    DOUBLE=double_opt
 
;***********************************************************************
; Constants

; Number of forcings
n_force = n_elements( force[0,*] )
; Number of time steps
n_time = n_elements( time )
; The number of independent heat capacity parameters
n_heatcap = n_elements( heatcap )
if ( n_heatcap ne 1 ) and ( n_heatcap ne n_force ) then stop
; The number of independent climate sensitivity parameters
n_lambda = n_elements( lambda )
if ( n_lambda ne 1 ) and ( n_lambda ne n_force ) then stop
; The number of independent ocean diffusivity parameters
n_kv = n_elements( kv )
if keyword_set( model_type ) then begin
  if model_type eq 'diffusive' then begin
    if ( n_kv ne 1 ) and ( n_kv ne n_force ) then stop
  endif else begin
    if n_kv ne 0 then stop
  endelse
endif

; The default threshold for improved model fit for deciding on the final model
if not( keyword_set( threshold ) ) then threshold = 1.d-6
; The default number of rounds that the threshold needs to be passed in order 
; to fix the new model
; (This is required to ensure that we are not getting zero improvement just 
; because the parameter perturbations are too large.)
if not( keyword_set( n_threshold_check ) ) then n_threshold_check = 5
; The default maximum number of rounds before aborting
if not( keyword_set( max_round ) ) then max_round = 100

; Define the initial parameter deltas to sample (half of value)
pert_heatcap = 0.5 * heatcap
pert_lambda = 0.5 * lambda
if n_kv gt 0 then pert_kv = 0.5 * kv
; Define the factor by which to reduce perturbations in each non-progressive 
; step
pert_factor = 0.25

;***********************************************************************
; Find the Best Fit

; Initialise round counter
ctr_round = 0

; Iterate while we have not reached the tolerance for ending
flag_check = 0
while flag_check lt n_threshold_check do begin

  ; Run the EBM with the current parameter values
  temp_heatcap = heatcap
  temp_lambda = lambda
  if n_kv gt 0 then temp_kv=kv
  output_temperature = ebm_calc( force, d_time=d_time, heatcap=temp_heatcap, $
      kv=temp_kv, lambda=temp_lambda, model_type=model_type, $
      solver_d_time_factor=solver_d_time_factor, solver_type=solver_type, $
      total=1 )
  ; Calculate the RMS difference from the target
  rms_ref = sqrt( mean( ( output_temperature - temperature ) ^ 2. ) )
  ; Copy the RMS value as the initial best in this round
  rms_new = rms_ref

  ; Iterate through heat capacity parameters
  for i_heatcap = 0, n_heatcap - 1 do begin
    ; Run the EBM with positively perturbed heat capacity value
    heatcap_plus = heatcap
    heatcap_plus[i_heatcap] = heatcap_plus[i_heatcap] + pert_heatcap[i_heatcap]
    temp_heatcap = heatcap_plus
    temp_lambda = lambda
    if n_kv gt 0 then temp_kv = kv
    temp_plus = ebm_calc( force, d_time=d_time, heatcap=temp_heatcap, $
        kv=temp_kv, lambda=temp_lambda, model_type=model_type, $
        solver_d_time_factor=solver_d_time_factor, solver_type=solver_type, $
        total=1 )
    ; Calculate the RMS difference from target
    rms_plus = sqrt( mean( ( temp_plus - temperature ) ^ 2. ) )
    temp = 0.
    ; Run the EBM with negatively perturbed heat capacity value
    heatcap_minus = heatcap
    heatcap_minus[i_heatcap] = heatcap_minus[i_heatcap] $
        - pert_heatcap[i_heatcap]
    if heatcap_minus[i_heatcap] le 0. then begin
      ; A fix to ensure a positive value
      heatcap_minus[i_heatcap] = heatcap[i_heatcap] / 2.
    endif
    temp_heatcap = heatcap_minus
    temp_lambda = lambda
    if n_kv gt 0 then temp_kv = kv
    temp_minus = ebm_calc( force, d_time=d_time, heatcap=temp_heatcap, $
        kv=temp_kv, lambda=temp_lambda, model_type=model_type, $
        solver_d_time_factor=solver_d_time_factor, solver_type=solver_type, $
        total=1 )
    ; Calculate the RMS difference from target
    rms_minus = sqrt( mean( ( temp_minus - temperature ) ^ 2. ) )
    temp = 0.
    ; If the positively perturbed parameter produces the best fit
    if ( rms_plus lt rms_new ) and ( rms_plus lt rms_minus ) then begin
      ; Assign positively perturbed parameter value to updated model
      heatcap[i_heatcap] = heatcap_plus[i_heatcap]
      ; Update the RMS and output temperature for the best model in this round
      rms_new = rms_plus
      output_temperature = temp_plus
    ; If the negatively perturbed parameter produces the best fit
    endif else if ( rms_minus lt rms_new ) and ( rms_minus lt rms_plus ) $
        then begin
      ; Assign negatively perturbed parameter value to updated model
      heatcap[i_heatcap] = heatcap_minus[i_heatcap]
      ; Update the RMS and output temperature for the best model in this round
      rms_new = rms_minus
      output_temperature = temp_minus
    ; If the unperturbed parameter value produces the best fit
    endif else begin
      ; Decrease the perturbation for the next round with this parameter
      pert_heatcap[i_heatcap] = pert_heatcap[i_heatcap] * pert_factor
    endelse
  endfor

  ; Iterate through climate sensitivity parameters
  for i_lambda = 0, n_lambda - 1 do begin
    ; Run the EBM with positively perturbed climate sensitivity value
    lambda_plus = lambda
    lambda_plus[i_lambda] = lambda_plus[i_lambda] + pert_lambda[i_lambda]
    temp_heatcap = heatcap
    temp_lambda = lambda_plus
    if n_kv gt 0 then temp_kv = kv
    temp_plus = ebm_calc( force, d_time=d_time, heatcap=temp_heatcap, $
        kv=temp_kv, lambda=temp_lambda, model_type=model_type, $
        solver_d_time_factor=solver_d_time_factor, solver_type=solver_type, $
        total=1 )
    ; Calculate the RMS difference from target
    rms_plus = sqrt( mean( ( temp_plus - temperature ) ^ 2. ) )
    temp = 0.
    ; Run the EBM with negatively perturbed climate sensitivity value
    lambda_minus = lambda
    lambda_minus[i_lambda] = lambda_minus[i_lambda] - pert_lambda[i_lambda]
    if lambda_minus[i_lambda] le 0. then begin
      ; A fix to ensure a positive value
      lambda_minus[i_lambda] = lambda[i_lambda] / 2.
    endif
    temp_heatcap = heatcap
    temp_lambda = lambda_minus
    if n_kv gt 0 then temp_kv = kv
    temp_minus = ebm_calc( force, d_time=d_time, heatcap=temp_heatcap, $
        kv=temp_kv, lambda=temp_lambda, model_type=model_type, $
        solver_d_time_factor=solver_d_time_factor, solver_type=solver_type, $
        total=1 )
    ; Calculate the RMS difference from target
    rms_minus = sqrt( mean( ( temp_minus - temperature ) ^ 2. ) )
    temp = 0.
    ; If the positively perturbed parameter produces the best fit
    if ( rms_plus lt rms_new ) and ( rms_plus lt rms_minus ) then begin
      ; Assign positively perturbed parameter value to updated model
      lambda[i_lambda] = lambda_plus[i_lambda]
      ; Update the RMS and output temperature for the best model in this round
      rms_new = rms_plus
      output_temperature = temp_plus
    ; If the negatively perturbed parameter produces the best fit
    endif else if ( rms_minus lt rms_new ) and ( rms_minus lt rms_plus ) $
        then begin
      ; Assign negatively perturbed parameter value to updated model
      lambda[i_lambda] = lambda_minus[i_lambda]
      ; Update the RMS and output temperature for the best model in this round
      rms_new = rms_minus
      output_temperature = temp_minus
    ; If the unperturbed parameter value produces the best fit
    endif else begin
      ; Decrease the perturbation for the next round with this parameter
      pert_lambda[i_lambda] = pert_lambda[i_lambda] * pert_factor
    endelse
  endfor

  ; If we need to perturb a diffusivitiy parameter
  if n_kv gt 0 then begin
    ; Iterate through diffusivity parameters
    for i_kv = 0, n_kv - 1 do begin
      ; Run the EBM with positively perturbed diffusivity value
      kv_plus = kv
      kv_plus[i_kv] = kv_plus[i_kv] + pert_kv[i_kv]
      temp_heatcap = heatcap
      temp_lambda = lambda
      if n_kv gt 0 then temp_kv = kv_plus
      temp_plus = ebm_calc( force, d_time=d_time, heatcap=temp_heatcap, $
          kv=temp_kv, lambda=temp_lambda, model_type=model_type, $
          solver_d_time_factor=solver_d_time_factor, solver_type=solver_type, $
          total=1 )
      ; Calculate the RMS difference from target
      rms_plus = sqrt( mean( ( temp_plus - temperature ) ^ 2. ) )
      temp = 0.
      ; Run the EBM with negatively perturbed diffusivity value
      kv_minus = kv
      kv_minus[i_kv] = kv_minus[i_kv] - pert_kv[i_kv]
      if kv_minus[i_kv] le 0. then begin
        ; A fix to ensure a positive value
        kv_minus[i_kv] = kv[i_kv] / 2.
      endif
      temp_heatcap = heatcap
      temp_lambda = lambda
      if n_kv gt 0 then temp_kv = kv_minus
      temp_minus = ebm_calc( force, d_time=d_time, heatcap=temp_heatcap, $
          kv=temp_kv, lambda=temp_lambda, model_type=model_type, $
          solver_d_time_factor=solver_d_time_factor, solver_type=solver_type, $
          total=1 )
      ; Calculate the RMS difference from target
      rms_minus = sqrt( mean( ( temp_minus - temperature ) ^ 2. ) )
      temp = 0.
      ; If the positively perturbed parameter produces the best fit
      if ( rms_plus lt rms_new ) and ( rms_plus lt rms_minus ) then begin
        ; Assign positively perturbed parameter value to updated model
        kv[i_kv] = kv_plus[i_kv]
        ; Update the RMS and output temperature for the best model in this round
        rms_new = rms_plus
        output_temperature = temp_plus
      ; If the negatively perturbed parameter produces the best fit
      endif else if ( rms_minus lt rms_new ) and ( rms_minus lt rms_plus ) $
          then begin
        ; Assign negatively perturbed parameter value to updated model
        kv[i_kv] = kv_minus[i_kv]
        ; Update the RMS and output temperature for the best model in this round
        rms_new = rms_minus
        output_temperature = temp_minus
      ; If the unperturbed parameter value produces the best fit
      endif else begin
        ; Decrease the perturbation for the next round with this parameter
        pert_kv[i_kv] = pert_kv[i_kv] * pert_factor
      endelse
    endfor
  endif

  ; Check if we have reached the tolerance incremental improvements to the fit
  if ( rms_ref - rms_new ) / rms_ref lt threshold then begin
    flag_check = flag_check + 1
  ; If we have not reached the theshold
  endif else begin
    ; Re-set the check
    flag_check = 0
  endelse

  ; Increment the round counter
  ctr_round = ctr_round + 1
  ; If we have exceeded the maximum number of rounds
  if ctr_round gt max_round then flag_check = n_threshold_check

endwhile

;***********************************************************************
; The End

;stop
return
END
