;+
; NAME:
;    plot_compass
;
; PURPOSE:
;    The procedure plots a compass for use in maps.
;
; CATEGORY:
;    Geographical
;
; CALLING SEQUENCE:
;    compass_rose
;
; INPUTS:
;    ANGLE, BACKGROUND, BUFFER, CENTRE, CHARSIZE, CHARTHICK, COLOR, LABEL, 
;    N_MINOR, RADIUS, THICK, OVERPLOT
;
; KEYWORD PARAMETERS:
;    ANGLE:  An optional scalar float specifying the clockwise rotation of the 
;        compass rose in degrees.
;    BACKGROUND:  An optional scalar integer specifying the color for a 
;        background disk surrounding the compass rose.  If not set then no 
;        background is plotted.
;    BUFFER:  An optional scalar float specifying the size of the buffer around 
;        the compass rose in units of RADIUS, ignoring additional space for 
;       labels.  The default is 1.1.  This is used for generating new plots or 
;       plotting background disks.
;    CENTRE:  An optional 2-element float specifying the x and y position of 
;        the centre of the compass rose.  The default is [0,0].
;    CHARSIZE:  An optional scalar float specifying the size of the characters 
;        in units of RADIUS.  Minor labels are plotted at 0.7*CHARSIZE.  The 
;        default is 0.2.
;    CHARTHICK:  An optional scalar float specifying the line thickness for 
;        writing the labels.  The default is 1.
;    COLOR:  An optional 2-element vector specifying the indices of the primary 
;        and secondary colours to be used for plotting.  The default is 
;        [!p.color,!p.background].
;    LABEL:  An optional string vector specifying the cardinal directions to 
;        label.  Possible options are 'n', 'e', 's', 'w', 'ne', 'se', 'sw', and 
;        'nw'.
;    N_MINOR:  An optional scalar integer specifying the number of minor points 
;        for which to include pointers on the compass rose.  This must be a 
;        multiple of 4, including the option of 0 (none).  The default is 4 
;        (NE,SE,SW,NW).
;    RADIUS:  An optional scalar float specifying the radius of the compass 
;        rose.  The default is 1.
;    THICK:  An optional scalar integer specifying the line thickness of the 
;        outline.
;    OVERPLOT:  If set then the compass rose is plotted on top of the current 
;        plot.  The default is to start a new plot.
;
; OUTPUTS:
;    -
;
; USES:
;    -
;
; PROCEDURE:
;    This procedure draws the image to the current output device.
;
; EXAMPLE:
;    ; Plot a compass rose at a 20-degree angle with all directions labelled.
;    compass_rose, label=['n','e','s','w','ne','se','sw','nw'], angle=20, charthick=3
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone (dastone@runbox.com), 2019-12-18
;-

;***********************************************************************

PRO COMPASS_ROSE, $
    ANGLE=angle, $
    BACKGROUND=color_background, $
    BUFFER=buffer_scale, $
    CENTRE=centre, $
    CHARSIZE=charsize, CHARTHICK=charthick, $
    COLOR=color, $
    LABEL=label, $
    N_MINOR=n_point_minor, $
    RADIUS=radius, $
    THICK=thick, $
    OVERPLOT=overplot_opt

;***********************************************************************
; Constants and inputs

; The default centre
if n_elements( centre ) ne 2 then begin
  if n_elements( centre ) eq 0 then begin
    centre = [ 0., 0. ]
  endif else begin
    stop
  endelse
endif

; The default unit radius
if not( keyword_set( radius ) ) then radius = 1.

; The number of major (N,E,S,W) points
n_point_major = 4
; The number of minor (NE,SE,SW,NW,etc.) points
if n_elements( n_point_minor ) eq 0 then n_point_minor = 4
if n_point_minor mod n_point_major ne 0 then stop
; The number of minor points per major point
n_point_minor_per_major = n_point_minor / n_point_major
; The total number of points
n_point = n_point_major + n_point_minor
; The incremental angle between all points
delta_theta = 1. / n_point * 2. * !pi

; The length of major arrows
length_major = radius
; The length of minor arrows
if n_point_minor gt 0 then length_minor = 0.7 * length_major
; The length (distance from centre) of gaps between arrows
if n_point_minor eq 0 then begin
  length_gap = 0.3 * length_major
endif else begin
  length_gap = 0.5 * length_minor
endelse

; Number of points for drawing a circle
n_circle = 64

; The character size, in radius units
if not( keyword_set( charsize ) ) then charsize = 0.2
; The character size for minor labels
charsize_minor = 0.7 * charsize
; The default character thickness
if not( keyword_set( charthick ) ) then charthick = 1.
; The default character width-to-height ratio
char_ratio = 0.7

; The size of the buffer around the plotted compass rose, in units of RADIUS
if not( keyword_set( buffer_scale ) ) then begin
  if keyword_set( label ) then begin
    buffer_scale = 1.1 + charsize / radius
  endif else begin
    buffer_scale = 1.1
  endelse
endif

; The default primary and secondary colours
if not( keyword_set( color ) ) then color = [ !p.color, !p.background ]

; The default empty label
if not( keyword_set( label ) ) then label = ''

; Degrees to radians factor
degrad = !pi / 180.

;***********************************************************************
; Plot compass

; Generate points on a unit circle
x_circle = sin( findgen( n_circle + 1 ) / n_circle * 2. * !pi )
y_circle = cos( findgen( n_circle + 1 ) / n_circle * 2. * !pi )

; Plot setup if required
if not( keyword_set( overplot_opt ) ) then begin
  plot, x_circle, y_circle, isotropic=1, xstyle=5, ystyle=5, nodata=1, $
      xrange=centre+[-1,1]*radius*buffer_scale, $
      yrange=centre+[-1,1]*radius*buffer_scale
endif

; Plot background
if n_elements( color_background ) eq 1 then begin
  polyfill, centre[0]+buffer_scale*radius*x_circle, $
      centre[1]+buffer_scale*radius*y_circle, color=color_background
endif

; Plot ring
temp_x = [ 1.05 * length_minor * x_circle, length_minor * reverse( x_circle ) ]
temp_y = [ 1.05 * length_minor * y_circle, length_minor * reverse( y_circle ) ]
polyfill, centre[0]+temp_x, centre[1]+temp_y, color=color[0]

; Plot minor pointers.
; Interate through major pointers
for i_major = 0, n_point_major - 1 do begin
  ; Set angle of preceding major pointer
  temp_theta = i_major / float( n_point_major ) * 2. * !pi
  ; Iterate through minor pointers
  for i_minor = 0, n_point_minor_per_major - 1 do begin
    ; Increment angle
    temp_theta = temp_theta + delta_theta
    ; Determine points on counterclockwise half of pointer
    temp_x = [ length_gap * sin( temp_theta - 0.5 * delta_theta ), $
        length_minor * sin( temp_theta ), 0., $
        length_gap * sin( temp_theta - 0.5 * delta_theta ) ]
    temp_y = [ length_gap * cos( temp_theta - 0.5 * delta_theta ), $
        length_minor * cos( temp_theta ), 0., $
        length_gap * cos( temp_theta - 0.5 * delta_theta ) ]
    ; Rotate by angle
    if keyword_set( angle ) then begin
      temp_x_new = cos( angle * degrad ) * temp_x $
          + sin( angle * degrad ) * temp_y
      temp_y_new = -sin( angle * degrad ) * temp_x $
          + cos( angle * degrad ) * temp_y
      temp_x = temp_x_new
      temp_y = temp_y_new
    endif
    ; Plot pointers
    polyfill, centre[0]+temp_x, centre[1]+temp_y, color=color[0]
    plots, centre[0]+temp_x, centre[1]+temp_y, color=color[0], thick=thick
    ; Determine points on clockwise half of pointer
    temp_x = [ length_gap * sin( temp_theta + 0.5 * delta_theta ), $
        length_minor * sin( temp_theta ), 0., $
        length_gap * sin( temp_theta + 0.5 * delta_theta ) ]
    temp_y = [ length_gap * cos( temp_theta + 0.5 * delta_theta ), $
        length_minor * cos( temp_theta ), 0., $
        length_gap * cos( temp_theta + 0.5 * delta_theta ) ]
    ; Rotate by angle
    if keyword_set( angle ) then begin
      temp_x_new = cos( angle * degrad ) * temp_x $
          + sin( angle * degrad ) * temp_y
      temp_y_new = -sin( angle * degrad ) * temp_x $
          + cos( angle * degrad ) * temp_y
      temp_x = temp_x_new
      temp_y = temp_y_new
    endif
    ; Plot pointers
    polyfill, centre[0]+temp_x, centre[1]+temp_y, color=color[1]
    plots, centre[0]+temp_x, centre[1]+temp_y, color=color[0], thick=thick
  endfor
endfor

; Plot major pointers.
; Interate through major pointers
for i_major = 0, n_point_major - 1 do begin
  ; Set angle of major pointer
  temp_theta = i_major / float( n_point_major ) * 2. * !pi
  ; Determine points on counterclockwise half of pointer
  temp_x = [ length_gap * sin( temp_theta - 0.5 * delta_theta ), $
      length_major * sin( temp_theta ), 0., $
      length_gap * sin( temp_theta - 0.5 * delta_theta ) ]
  temp_y = [ length_gap * cos( temp_theta - 0.5 * delta_theta ), $
      length_major * cos( temp_theta ), 0., $
      length_gap * cos( temp_theta - 0.5 * delta_theta ) ]
  ; Rotate by angle
  if keyword_set( angle ) then begin
    temp_x_new = cos( angle * degrad ) * temp_x + sin( angle * degrad ) * temp_y
    temp_y_new = -sin( angle * degrad ) * temp_x $
        + cos( angle * degrad ) * temp_y
    temp_x = temp_x_new
    temp_y = temp_y_new
  endif
  ; Plot pointers
  polyfill, centre[0]+temp_x, centre[1]+temp_y, color=color[0]
  plots, centre[0]+temp_x, centre[1]+temp_y, color=color[0], thick=thick
  ; Determine points on clockwise half of pointer
  temp_x = [ length_gap * sin( temp_theta + 0.5 * delta_theta ), $
      length_major * sin( temp_theta ), 0., $
      length_gap * sin( temp_theta + 0.5 * delta_theta ) ]
  temp_y = [ length_gap * cos( temp_theta + 0.5 * delta_theta ), $
      length_major * cos( temp_theta ), 0., $
      length_gap * cos( temp_theta + 0.5 * delta_theta ) ]
  ; Rotate by angle
  if keyword_set( angle ) then begin
    temp_x_new = cos( angle * degrad ) * temp_x + sin( angle * degrad ) * temp_y
    temp_y_new = -sin( angle * degrad ) * temp_x $
        + cos( angle * degrad ) * temp_y
    temp_x = temp_x_new
    temp_y = temp_y_new
  endif
  ; Plot pointers
  polyfill, centre[0]+temp_x, centre[1]+temp_y, color=color[1]
  plots, centre[0]+temp_x, centre[1]+temp_y, color=color[0], thick=thick
endfor

; Generate label font
if keyword_set( label ) then begin
  ; Generate N
  pos_x_n = [ 0., 0., 1., 1. ]
  pos_y_n = [ 0., 1., 0., 1. ]
  ; Generate E
  pos_x_e = [ 1., 0., 0., 0.7, 0., 0., 1. ]
  pos_y_e = [ 1., 1., 0.5, 0.5, 0.5, 0., 0. ]
  ; Generate S
  temp_angle = ( 270. - findgen( n_circle * 3 / 4 ) * 360. / n_circle ) * degrad
  pos_x_s = 0.5 + 0.5 * sin( temp_angle )
  pos_y_s = 0.25 + 0.25 * cos( temp_angle )
  temp_angle = ( 180. + findgen( n_circle * 3 / 4 + 1 ) * 360. / n_circle ) $
      * degrad
  pos_x_s = [ pos_x_s, 0.5 + 0.5 * sin( temp_angle ) ]
  pos_y_s = [ pos_y_s, 0.75 + 0.25 * cos( temp_angle ) ]
  ; Generate W
  pos_x_w = [ 0., 0.25, 0.5, 0.75, 1. ]
  pos_y_w = [ 1., 0., 0.5, 0., 1. ]
endif

; Add north label
if max( label eq 'n' ) eq 1 then begin
  temp_x_n = char_ratio * ( pos_x_n - 0.5 )
  temp_y_n = pos_y_n
  if keyword_set( angle ) then begin
    temp_x = cos( angle * degrad ) * temp_x_n + sin( angle * degrad ) * temp_y_n
    temp_y = -sin( angle * degrad ) * temp_x_n $
        + cos( angle * degrad ) * temp_y_n
    temp_x = 1.05 * length_major * sin( angle * degrad ) + charsize * temp_x
    temp_y = 1.05 * length_major * cos( angle * degrad ) + charsize * temp_y
  endif else begin
    temp_x = 1.05 * charsize * temp_x_n
    temp_y = 1.05 * length_major + charsize * temp_y_n
  endelse
  plots, centre[0]+temp_x, centre[1]+temp_y, thick=charthick, color=color[0]
endif
; Add east label
if max( label eq 'e' ) eq 1 then begin
  temp_y_e = pos_y_e - 0.5
  temp_x_e = char_ratio * pos_x_e
  if keyword_set( angle ) then begin
    temp_x = cos( angle * degrad ) * temp_x_e + sin( angle * degrad ) * temp_y_e
    temp_y = -sin( angle * degrad ) * temp_x_e $
        + cos( angle * degrad ) * temp_y_e
    temp_x = 1.05 * length_major * cos( angle * degrad ) + charsize * temp_x
    temp_y = -1.05 * length_major * sin( angle * degrad ) + charsize * temp_y
  endif else begin
    temp_x = 1.05 * length_major + charsize * temp_x_e
    temp_y = 1.05 * charsize * temp_y_e
  endelse
  plots, centre[0]+temp_x, centre[1]+temp_y, thick=charthick, color=color[0]
endif
; Add south label
if max( label eq 's' ) eq 1 then begin
  temp_x_s = char_ratio * ( pos_x_s - 0.5 )
  temp_y_s = pos_y_s - 1.
  if keyword_set( angle ) then begin
    temp_x = cos( angle * degrad ) * temp_x_s + sin( angle * degrad ) * temp_y_s
    temp_y = -sin( angle * degrad ) * temp_x_s $
        + cos( angle * degrad ) * temp_y_s
    temp_x = -1.05 * length_major * sin( angle * degrad ) + charsize * temp_x
    temp_y = -1.05 * length_major * cos( angle * degrad ) + charsize * temp_y
  endif else begin
    temp_x = 1.05 * charsize * temp_x_s
    temp_y = -1.05 * length_major + charsize * temp_y_s
  endelse
  plots, centre[0]+temp_x, centre[1]+temp_y, thick=charthick, color=color[0]
endif
; Add west label
if max( label eq 'w' ) eq 1 then begin
  temp_x_w = char_ratio * (pos_x_w - 1. )
  temp_y_w = pos_y_w - 0.5
  if keyword_set( angle ) then begin
    temp_x = cos( angle * degrad ) * temp_x_w + sin( angle * degrad ) * temp_y_w
    temp_y = -sin( angle * degrad ) * temp_x_w $
        + cos( angle * degrad ) * temp_y_w
    temp_x = -1.05 * length_major * cos( angle * degrad ) + charsize * temp_x
    temp_y = 1.05 * length_major * sin( angle * degrad ) + charsize * temp_y
  endif else begin
    temp_x = -1.05 * length_major + charsize * temp_x_w
    temp_y = 1.05 * charsize * temp_y_w
  endelse
  plots, centre[0]+temp_x, centre[1]+temp_y, thick=charthick, color=color[0]
endif
; Add northeast label
if max( label eq 'ne' ) eq 1 then begin
  temp_angle = 45.
  if keyword_set( angle ) then temp_angle = temp_angle + angle
  temp_x_ne = char_ratio * pos_x_n - ( 1. + char_ratio ) / 2.
  temp_y_ne = pos_y_n
  temp_x = cos( temp_angle * degrad ) * temp_x_ne $
      + sin( temp_angle * degrad ) * temp_y_ne
  temp_y = -sin( temp_angle * degrad ) * temp_x_ne $
      + cos( temp_angle * degrad ) * temp_y_ne
  temp_x = 1.1 * length_minor * sin( temp_angle * degrad ) $
      + charsize_minor * temp_x
  temp_y = 1.1 * length_minor * cos( temp_angle * degrad ) $
      + charsize_minor * temp_y
  plots, centre[0]+temp_x, centre[1]+temp_y, thick=charthick, color=color[0]
  temp_x_ne = 1. + char_ratio * pos_x_e - ( 1. + char_ratio ) / 2.
  temp_y_ne = pos_y_e
  temp_x = cos( temp_angle * degrad ) * temp_x_ne $
      + sin( temp_angle * degrad ) * temp_y_ne
  temp_y = -sin( temp_angle * degrad ) * temp_x_ne $
      + cos( temp_angle * degrad ) * temp_y_ne
  temp_x = 1.1 * length_minor * sin( temp_angle * degrad ) $
      + charsize_minor * temp_x
  temp_y = 1.1 * length_minor * cos( temp_angle * degrad ) $
      + charsize_minor * temp_y
  plots, centre[0]+temp_x, centre[1]+temp_y, thick=charthick, color=color[0]
endif
; Add southeast label
if max( label eq 'se' ) eq 1 then begin
  temp_angle = 135.
  if keyword_set( angle ) then temp_angle = temp_angle + angle
  temp_x_se = char_ratio * pos_x_s - ( 1. + char_ratio ) / 2.
  temp_y_se = pos_y_s - 1.
  temp_x = cos( ( temp_angle + 180. ) * degrad ) * temp_x_se $
      + sin( ( temp_angle + 180. ) * degrad ) * temp_y_se
  temp_y = -sin( ( temp_angle + 180. ) * degrad ) * temp_x_se $
      + cos( ( temp_angle + 180. ) * degrad ) * temp_y_se
  temp_x = 1.1 * length_minor * sin( temp_angle * degrad ) $
      + charsize_minor * temp_x
  temp_y = 1.1 * length_minor * cos( temp_angle * degrad ) $
      + charsize_minor * temp_y
  plots, centre[0]+temp_x, centre[1]+temp_y, thick=charthick, color=color[0]
  temp_x_se = 1. + char_ratio * pos_x_e - ( 1. + char_ratio ) / 2.
  temp_y_se = pos_y_e - 1.
  temp_x = cos( ( temp_angle + 180. ) * degrad ) * temp_x_se $
      + sin( ( temp_angle + 180. ) * degrad ) * temp_y_se
  temp_y = -sin( ( temp_angle + 180. ) * degrad ) * temp_x_se $
      + cos( ( temp_angle + 180. ) * degrad ) * temp_y_se
  temp_x = 1.1 * length_minor * sin( temp_angle * degrad ) $
      + charsize_minor * temp_x
  temp_y = 1.1 * length_minor * cos( temp_angle * degrad ) $
      + charsize_minor * temp_y
  plots, centre[0]+temp_x, centre[1]+temp_y, thick=charthick, color=color[0]
endif
; Add southwest label
if max( label eq 'sw' ) eq 1 then begin
  temp_angle = 225.
  if keyword_set( angle ) then temp_angle = temp_angle + angle
  temp_x_sw = char_ratio * pos_x_s - ( 1. + char_ratio ) / 2.
  temp_y_sw = pos_y_s - 1.
  temp_x = cos( ( temp_angle + 180. ) * degrad ) * temp_x_sw $
      + sin( ( temp_angle + 180. ) * degrad ) * temp_y_sw
  temp_y = -sin( ( temp_angle + 180. ) * degrad ) * temp_x_sw $
      + cos( ( temp_angle + 180. ) * degrad ) * temp_y_sw
  temp_x = 1.1 * length_minor * sin( temp_angle * degrad ) $
      + charsize_minor * temp_x
  temp_y = 1.1 * length_minor * cos( temp_angle * degrad ) $
      + charsize_minor * temp_y
  plots, centre[0]+temp_x, centre[1]+temp_y, thick=charthick, color=color[0]
  temp_x_sw = 1. + char_ratio * pos_x_w - ( 1. + char_ratio ) / 2.
  temp_y_sw = pos_y_w - 1.
  temp_x = cos( ( temp_angle + 180. ) * degrad ) * temp_x_sw $
      + sin( ( temp_angle + 180. ) * degrad ) * temp_y_sw
  temp_y = -sin( ( temp_angle + 180. ) * degrad ) * temp_x_sw $
      + cos( ( temp_angle + 180. ) * degrad ) * temp_y_sw
  temp_x = 1.1 * length_minor * sin( temp_angle * degrad ) $
      + charsize_minor * temp_x
  temp_y = 1.1 * length_minor * cos( temp_angle * degrad ) $
      + charsize_minor * temp_y
  plots, centre[0]+temp_x, centre[1]+temp_y, thick=charthick, color=color[0]
endif
; Add northwest label
if max( label eq 'nw' ) eq 1 then begin
  temp_angle = 315.
  if keyword_set( angle ) then temp_angle = temp_angle + angle
  temp_x_nw = char_ratio * pos_x_n - ( 1. + char_ratio ) / 2.
  temp_y_nw = pos_y_n
  temp_x = cos( temp_angle * degrad ) * temp_x_nw $
      + sin( temp_angle * degrad ) * temp_y_nw
  temp_y = -sin( temp_angle * degrad ) * temp_x_nw $
      + cos( temp_angle * degrad ) * temp_y_nw
  temp_x = 1.1 * length_minor * sin( temp_angle * degrad ) $
      + charsize_minor * temp_x
  temp_y = 1.1 * length_minor * cos( temp_angle * degrad ) $
      + charsize_minor * temp_y
  plots, centre[0]+temp_x, centre[1]+temp_y, thick=charthick, color=color[0]
  temp_x_nw = 1. + char_ratio * pos_x_w - ( 1. + char_ratio ) / 2.
  temp_y_nw = pos_y_w
  temp_x = cos( temp_angle * degrad ) * temp_x_nw $
      + sin( temp_angle * degrad ) * temp_y_nw
  temp_y = -sin( temp_angle * degrad ) * temp_x_nw $
      + cos( temp_angle * degrad ) * temp_y_nw
  temp_x = 1.1 * length_minor * sin( temp_angle * degrad ) $
      + charsize_minor * temp_x
  temp_y = 1.1 * length_minor * cos( temp_angle * degrad ) $
      + charsize_minor * temp_y
  plots, centre[0]+temp_x, centre[1]+temp_y, thick=charthick, color=color[0]
endif

;***********************************************************************
; The end

return
END
