;+
; NAME:
;	CHOOSE_LEVELS
;
; PURPOSE:
;    This function chooses convenient values for contour levels.
;
; CATEGORY:
;    Graphics
;
; CALLING SEQUENCE:
;    Result = CHOOSE_LEVELS( DATA )
;
; INPUTS:
;    DATA:  The required array of data values, of type integer or floating 
;        point.
;
; KEYWORD PARAMETERS:
;    MAX_NLEVELS:  The maximum number of levels to be chosen.  The default is 
;        10.
;    MIN_NLEVELS:  The maximum number of levels to be chosen.  The default is 3.
;    NLEVELS:  The same as MAX_NLEVELS.  Retained for back compatibility.
;    RANGE:  A two-element vector containing the FIXED [minimum,maximum] level 
;        values.  The result returned will be a list of convenient levels 
;        within this range, which may not entirely extend to these limits.  The
;        default is to alter the limiting values such that they are convenient, 
;        and thus included in the levels.
;
; USES:
;    decimal_place.pro
;    first_digit.pro
;    interval_calc.pro
;    sign.pro
;    var_type.pro
;
; PROCEDURE:
;    This function adjusts the limiting values of a level scale and chooses the 
;    best number of levels such that the level values are convenient decimal 
;    numbers.
;
; EXAMPLE:
;    Create a random data set.
;      y = randomn( 1, 100 )
;    Choose convenient levels for partitioning the data.
;      result = choose_levels( y )
;
; MODIFICATION HISTORY:
;    Written by:  Daithi A. Stone, 2000-09-25
;    Modified:  DAS, 2001-01-12 (switched limit value-choosing algorithm)
;    Modified:  DAS, 2001-05-08 (fixed limit-choosing bug)
;    Modified:  DAS, 2002-02-06 (switched limit value-choosing algorithm again)
;    Modified:  DAS, 2002-02-13 (added ability to ignore NaNs)
;    Modified:  DAS, 2002-03-12 (added NLEVELS keyword, added checks to 
;        value-choosing)
;    Modified:  DAS, 2002-05-07 (fixed float-integer bug)
;    Modified:  DAS, 2002-05-19 (fixed round-off bug)
;    Modified:  DAS, 2002-06-02 (fine-tuning level-choosing)
;    Modified:  DAS, 2016-02-26 (updated formating;  added MAX_NLEVELS and 
;        MIN_NLEVELS keywords)
;-

;***********************************************************************

FUNCTION CHOOSE_LEVELS, $
    Data, $
    MAX_NLEVELS=max_nlevels, MIN_NLEVELS=min_nlevels, $
    NLEVELS=max_nlevels_old, $
    RANGE=range

;***********************************************************************
; Constants

; Data limiting values
min_datum = min( data, nan=1 )
max_datum = max( data, nan=1 )

; Pre-defined range limits
range_opt = keyword_set( range )
if ( range_opt eq 1 ) and ( n_elements( range ) lt 2 ) then range_opt = 0

; The maximum number of levels
nlevels_opt = keyword_set( max_nlevels )
if keyword_set( max_nlevels_old ) then begin
  if keyword_set( max_nlevels ) then stop
  max_nlevels = max_nlevels_old
  nlevels_opt = 1
endif
if nlevels_opt eq 0 then max_nlevels = 20
real_max_nlevels = 10
; The minimum number of levels
if not( keyword_set( min_nlevels ) ) then min_nlevels = 3
n_levels = indgen( max_nlevels - min_nlevels + 1 ) + min_nlevels

; Data variable type
vartype = var_type( data )
if ( vartype eq 2 ) or ( vartype eq 3 ) then begin
  one = 1
endif else begin
  one = 1.
endelse

; Correction factor for floating point numbers
if ( vartype eq 2 ) or ( vartype eq 3 ) then begin
  cfact = 1
endif else begin
  cfact = 1.00001
endelse

;***********************************************************************
; Calculate Convenient Limits

; Pre-set limits
if range_opt then begin
  if range[1] lt range[0] then begin
    temp = range[0]
    range[0] = range[1]
    range[0] = temp
  endif
endif

; Convenient limits
good_max = interval_calc( max_datum / cfact )
good_min = -interval_calc( -min_datum / cfact )

; Calculate the scale of change
scale_max = decimal_place( good_max / 2., greater=1 )
scale_min = decimal_place( good_min / 2., greater=1 )
if scale_max gt scale_min then good_max = one * 0
if scale_min gt scale_max then good_min = one * 0
scale = min( [ scale_max, scale_min ] )
scale = 10. ^ ( -scale )

; Improve maximum level
check = 0
if sign( max_datum ) eq 1 then begin
  temp = cfact
endif else begin
  temp = 1. / cfact
endelse
while check ne 1 do begin
  if scale eq 0 then check = 1
  if good_max - scale le max_datum / temp then begin
    check = 1
  endif else begin
    good_max = good_max - scale
  endelse
endwhile
if good_max lt max_datum / temp then good_max = good_max + scale

; Improve minimum level
check = 0
if sign( min_datum ) eq 1 then begin
  temp = cfact
endif else begin
  temp = 1. / cfact
endelse
while check ne 1 do begin
  if scale eq 0 then check = 1
  if good_min + scale ge min_datum * temp then begin
    check = 1
  endif else begin
    good_min = good_min + scale
  endelse
endwhile
if good_min gt min_datum * temp then good_min = good_min - scale

; Integer or real limits
if good_max - good_min ge 6. then begin
  good_max = round( good_max )
  good_min = round( good_min )
  one = round( one )
endif

;***********************************************************************
; Calculate Convenient Levels

; Number of legend values
dig = first_digit( cfact * ( good_max - good_min ) )
;decplace = max( decimal_place( [ good_min, good_max ] ) )
if dig le 3 then begin
;if good_max - good_min lt 2 then begin
  if dig lt 2 then begin
    decplace = decimal_place( good_max-good_min, greater=1 ) + 1
    dig = round( cfact * ( good_max - good_min ) * 10. ^ decplace )
  endif else begin
    dig = dig * 10
  endelse
endif
;;Update number of levels possibilities
;if range_opt then begin
;  if ( range[0] gt good_min ) or ( range[1] lt good_max ) then begin
;    nlevels = [3,4,5,6,7,8,9,10,11]
;  endif
;endif
n_levels_0 = round( 2 * dig / ( n_levels - 1 ) * 1. * ( n_levels - 1 ) )
id = where( n_levels_0 eq 2*dig, n_id )
n_levels = n_levels[id[n_id-1]]
if nlevels_opt then begin
  while n_levels le max_nlevels / 2 do n_levels = 2 * n_levels - 1
  one = 1.
endif
levels = good_min + findgen( n_levels ) / ( n_levels - 1 ) $
    * ( good_max - good_min )
; Remove excess levels
id = where( levels gt max_datum, n_id )
if n_id gt 1 then begin
  levels = levels[0:n_levels-n_id]
  n_levels = n_elements( levels )
endif
id = where( levels lt min_datum, n_id )
if n_id gt 1 then begin
  levels = levels[n_id-1:n_levels-1]
  n_levels = n_elements( levels )
endif
; If too many levels, remove every second one
if not( nlevels_opt ) then begin
  if n_levels gt real_max_nlevels then begin
    if not( odd( n_levels ) ) then begin
      if levels[0] eq 0 then begin
        levels = [ levels, 2 * levels[n_levels-1] - levels[n_levels-2] ]
      endif else begin
        levels = [ 2 * levels[0] - levels[1], levels ]
      endelse
    endif
    n_levels = n_levels / 2 + 1
    id = 2 * indgen( n_levels )
    levels = levels[id]
  endif
endif

; Correct if levels are beyond pre-defined limits
if range_opt then begin
  if range[0] lt 0 then begin
    id = where( levels ge range[0] * cfact )
  endif else begin
    id = where( levels ge range[0] / cfact )
  endelse
  levels = levels[id]
  if range[1] lt 0 then begin
    id = where( levels le range[1] / cfact )
  endif else begin
    id = where( levels le range[1] * cfact )
  endelse
  levels = levels[id]
endif

; Revert to integer if necessary
if one / 2 eq 0 then levels = round( levels )

;***********************************************************************
; The End

return, levels
END
