;+
; NAME:
;    BLOCK_CELL
;
; PURPOSE:
;    This procedure draws a boxed contour plot, where boxes are assigned a 
;    level instead of calculating smooth boundaries for the levels.
;
; CATEGORY:
;    Graphics
;
; CALLING SEQUENCE:
;    block_cell, data, xvec, yvec
;
; INPUTS:
;    DATA:  The data field array of size [N_XVEC,N_YVEC], or a vector of length 
;        N_SPACE.  Of type integer or floating point.
;    XVEC:  A vector or array of the X-axis coordinates of the data array, of 
;        type integer or floating point.  Of length N_XVEC or N_SPACE, 
;        depending on the format of Data.
;    YVEC:  A vector or array of the Y-axis coordinates of the data array, of 
;        type integer or floating point.  Of length N_YVEC or N_SPACE, 
;        depending on the format of Data.
;
; KEYWORD PARAMETERS:
;    BORDER:  If RETURN_BORDER is set, then this returns a 2-dimensional 
;        floating point number array containing the coordinates of the borders 
;        of the shapes.  The array is of the form [3,N_POINT], where N_POINT is 
;        the total number of points in all of the shapes.  Values in [0,*] and 
;        [1,*] are the coordinates in the X-axis and Y-axis respectively.  
;        Values in [2,*] are the indices of the shapes, running from 0 to 
;        N_SHAPE-1 where N_SHAPE is the total number of shapes.  If 
;        RETURN_BORDER is not set then nothing is returned.
;    C_COLORS:  A vector of colour indices for the contoured levels.  The 
;        default may not look great with certain colour tables.
;    CLIP:  If set to 0 then no clipping is performed.
;    COLOR:  The colour index of the various lines.  The default is set in 
;        !P.COLOR.
;    LEVELS:  A vector of values for the contour levels.  This can be 
;        determined automatically from the data.
;    LIMIT:  A four-element or eight-element vector specifying the limiting 
;        values of the [x,y] dimensions.  For a four-element vector, the format 
;        is [xmin,ymin,xmax,ymax].  For an eight-element vector the format is 
;        [x-left,y-left,x-top,y-top,x-right,y-right,x-bottom,y-bottom], for 
;        points on the left, top, right, and bottom edges of the plot, 
;        respectively.  If not set then the values of [X,Y]RANGE are used, or 
;        are taken as the minimum/maximum value minus/plus half the distance to 
;        the second-smallest/largest value.
;    LINESTYLE_OUTLINE:  The LINESTYLE keyword value to be used when plotting 
;        outlines.
;    NO_NAN:  If set then NaN areas are not filled in.  The default is to fill 
;        them in with the background colour.
;    NO_PLOT:  If set then no plotting is performed.  If RETURN_BORDER is set 
;        then the procedure can serve as a shape calculation tool.  The default 
;        is for plotting.
;    OUTLINE:  If set then contour lines for DATA are plotted instead of the 
;        filled colour-contours.
;    POINT_TICKS:  If contour lines are being drawn, then setting this 
;        keyword causes perpendicular ticks to be drawn on the contours lines.  
;        If set to -1 then the ticks point downhill on DATA, if set to 1 then 
;        they point uphill.
;    PROJECT_X:  An optional float array of size N_XVEC,N_YVEC containing the 
;        discrete transformation from each of the regular points along the XVEC 
;        and YVEC dimensions to a bespoke new x-dimension in an irregular 
;        projection.  The code interpolates using this descrete transformation 
;        data.  PROJECT_Y must be input too.  ROTATE_COORD and PROJECTION can 
;        input as well and are implemented after this bespoke projection.
;    PROJECT_Y:  Same as PROJECT_X except for a bespoke new y-dimension.
;    PROJECTION:  An optional scalar string naming a spatial transform to 
;        perform before plotting.  Available options are:
;        * 'hammer':  The Hammer map projection.  x-values range between 
;          +/-2*sqrt(2) and y-values range between +/-sqrt(2).
;        * 'orthographic=<lon>,<lat>';  The orthographic map projection.  
;          "<lon>" and "<lat>" are the longitude and latitude coordinates of 
;          the centre of the projection.
;    [X,Y]RANGE:  A 2-element vector containing the minimum and maximum 
;        [x,y]-coordinates to be plotted.
;    RETURN_BORDER:  If set then output is returned in BORDER.  The default 
;        is for no output to be returned in BORDER.
;    ROTATE_COORD:  An optional two-element float vector specifying the north 
;        pole for a grid rotation.  If input, then data a plotted according to 
;        the rotated grid.
;    THICK:  The line thickness used for drawing.
;    WRAF_X:  An optional two-element float vector specifying the plotting 
;        range in the x-dimension when that range wraps.  This prevents 
;        artefacts from plotting shapes that span the wrapping edge.  For 
;        instance, for geographic plots this might be longitudes [-180.,180.].  
;        Note that the values should be in units of PROJECT_X if that has been 
;        input.
;
; OUTPUTS:
;    BORDER
;
; USES:
;    choose_levels.pro
;    convert_rotated_grid.pro
;    dimension.pro
;    geo_dist.pro
;    sign.pro
;
; PROCEDURE:
;    This procedure plots a world map using map_set.pro and map_continents.pro 
;    and then plots a colour contour on top using contour.pro.
;
; EXAMPLE:
;    Contour an increasing-value 20*10 array over a map of Earth.
;      arr = findgen(20,10)
;      loadct, 5
;      contour_world, arr, blockcell=1
;
; MODIFICATION HISTORY:
;    Written by:  Daithi Stone (dstone@lbl.gov), 2012-08-19 (code extracted 
;        from contour_world.pro).
;    Modified:  DAS, 2016-10-20 (Extracted from contour_world.pro)
;    Modified:  DAS, 2017-01-24 (Added BORDER output.  Added NO_PLOT, 
;        RETURN_BORDER options)
;    Modified:  DAS, 2017-06-08 (Corrected error in calculation of minimum 
;        longitude deltas;  sorted plotting from larger shapes to smaller 
;        shapes in order to avoid overplotting)
;    Modified:  DAS, 2018-01-06 (Corrected bug in which no amalgamation of 
;        shapes was done in the x-dimension... not sure when that one appeared)
;    Modified:  DAS, 2018-05-02 (Fixed issue in which NaN-defined shapes were 
;        being lumped together and thus only the first shape was being plotted)
;    Modified:  DAS, 2018-09-25 (Fixed automatic selection of contour levels;  
;        Fixed errors in calculating node distances and limiting numbers of 
;        nearby nodes with irregular gridss)
;    Modified:  DAS, 2019-05-02 (Added PROJECTION keyword input)
;    Modified:  DAS, 2019-12-18 (Added streamlining of redundant points for 
;        plotting, which helps with GDL's numerics)
;    Modified:  DAS, 2020-02-24 (Fixed bug in reading orthographic projection 
;        input)
;    Modified:  DAS, 2021-01-21 (Added ROTATE_COORD keyword input)
;    Modified:  DAS, 2021-02-22 (Corrected issues with ROTATE_COORD)
;    Modified:  DAS, 2021-04-14 (Added NO_NAN keyword option)
;    Modified:  DAS, 2021-08-17 (Added LINESTYLE_OUTLINE keyword input)
;    Modified:  DAS, 2022-03-25 (Added CLIP keyword option)
;    Modified:  DAS, 2022-08-19 (Added bespoke projection mapping via the 
;        PROJECT_X and PROJECT_Y keyword inputs;  Added WRAP_X keyword input;  
;        Implemented clipping of far side for orthographic projection)
;    Modified:  DAS, 2022-08-24 (Fixed bug with wrapping for bespoke 
;        projections)
;
; TASKS:
;    Enact POINT_TICKS option
;-

;***********************************************************************

PRO BLOCK_CELL, $
    DATA, XVEC, YVEC, $
    C_COLORS=c_colors, $
    CLIP=clip_opt, $
    COLOR=color, $
    LEVELS=levels, $
    LIMIT=limit, $
    LINESTYLE_OUTLINE=outline_linestyle, $
    OUTLINE=outline_opt, $
    POINT_TICKS=point_ticks, $
    PROJECT_X=project_x, PROJECT_Y=project_y, $
    PROJECTION=projection, $
    XRANGE=xrange, YRANGE=yrange, $
    THICK=thick, $
    ROTATE_COORD=rotate_coord, $
    WRAP_X=wrap_x, $
    NO_NAN=no_nan_opt, $
    NO_PLOT=no_plot_opt, $
    RETURN_BORDER=return_border_opt, $
    BORDER=border_out

;***********************************************************************
; Constants and Options

; Option of no plot output
no_plot_opt = keyword_set( no_plot_opt )
; Option not to plot NaN values
no_nan_opt = keyword_set( no_nan_opt )
; The option to clip the plot
clip_opt = keyword_set( clip_opt )

; Option for line contours
outline_opt = keyword_set( outline_opt )

; Option to return border coorinates for the shapes
return_border_opt = keyword_set( return_border_opt )

; The default number of vertices for boxes when given irregularly gridded data
if outline_opt eq 1 then begin
  n_vertex_0 = 4
endif else begin
  n_vertex_0 = 4
endelse

; Determine if the data is gridded or irregular
if dimension( data ) eq 1 then begin
  irregular_opt = 1
endif else begin
  irregular_opt = 0
endelse

; Determine the input array dimensions
n_xvec = n_elements( xvec )
n_yvec = n_elements( yvec )
n_data = n_elements( data )
if irregular_opt eq 1 then begin
  if n_xvec ne n_data then stop
  if n_yvec ne n_data then stop
endif else begin
  if n_data ne n_xvec * n_yvec then stop
endelse

; Default contour levels
if not( keyword_set( levels ) ) then begin
  ;if not( keyword_set( n_levels ) ) then n_levels = 29
  levels = choose_levels( data )
  n_levels = n_elements( levels )
  ;levels = levels[0] + findgen( n_levels + 1 ) / n_levels $
  ;    * ( levels[n_levels-1] - levels[0] )
endif else begin
  n_levels = n_elements( levels )
endelse

; Colour scale
if not( keyword_set( c_colors ) ) and ( outline_opt eq 0 ) then begin
  c_colors = indgen( n_levels + 1 ) + 2
endif

; Degrees to radians conversion factor for use in map projections
if keyword_set( projection ) then degrad = !pi / 180.

;***********************************************************************
; Prepare data for analysis

; Copy input arrays
data_use = data
xvec_use = xvec
yvec_use = yvec

; Ensure monotonically increasing dimensions for regularly gridded data
if irregular_opt eq 0 then begin
  id = sort( xvec_use )
  if max( id[1:n_xvec-1] - id[0:n_xvec-2] ) gt 1 then begin
    xvec_use = xvec_use[id]
    data_use = data_use[id,*]
  endif
  id = sort( yvec_use )
  if max( id[1:n_yvec-1] - id[0:n_yvec-2] ) gt 1 then begin
    yvec_use = yvec_use[id]
    data_use = data_use[*,id]
  endif
  ; Get the minimum deltas
  d_xvec_use_min = min( xvec_use[1:n_xvec-1] - xvec_use[0:n_xvec-2] )
  d_yvec_use_min = min( yvec_use[1:n_yvec-1] - yvec_use[0:n_yvec-2] )
endif

; Set the default plotting limits
if not( keyword_set( limit ) ) then begin
  limit = fltarr( 4 )
  if keyword_set( xrange ) then begin
    limit[[0,2]] = xrange
  endif else begin
    id = sort( xvec )
    limit[[0,2]] = [ xvec[id[0]] - ( xvec[id[1]] - xvec[id[0]] ) / 2., $
        xvec[id[n_xvec-1]] + ( xvec[id[n_xvec-1]] - xvec[id[n_xvec-2]] ) / 2. ]
    id = -1
  endelse
  if keyword_set( yrange ) then begin
    limit[[1,3]] = yrange
  endif else begin
    id = sort( yvec )
    limit[[1,3]] = [ yvec[id[0]] - ( yvec[id[1]] - yvec[id[0]] ) / 2., $
        yvec[id[n_yvec-1]] + ( yvec[id[n_yvec-1]] - yvec[id[n_yvec-2]] ) / 2. ]
    id = -1
  endelse
endif
n_limit = n_elements( limit )

; Implement longitudinal shift of grid rotation
if ( irregular_opt eq 0 ) and keyword_set( rotate_coord ) then begin
  if min( xvec_use ) lt -180.1 then stop
  if max( xvec_use ) gt 180.1 then stop
  if rotate_coord[0] gt 0 then begin
    id_lon = where( xvec_use - rotate_coord[0] gt 180., n_id_lon )
    if n_id_lon gt 0 then begin
      if n_id_lon + id_lon[0] ne n_xvec then stop
      data_use = [ data_use[id_lon,*], data_use[0:id_lon[0]-1,*] ]
    endif
  endif else if rotate_coord[0] lt 0 then begin
    id_lon = where( xvec_use + rotate_coord[0] lt -180., n_id_lon )
    if n_id_lon gt 0 then begin
      if n_id_lon ne id_lon[n_id_lon-1] + 1 then stop
      data_use = [ data_use[id_lon,*], data_use[0:id_lon[0]-1,*] ]
    endif
  endif
  rotate_coord_use = [ 180., rotate_coord[1] ]
  ;rotate_coord_use = [ rotate_coord[0] - 180., rotate_coord[1] ]
  ;if rotate_coord_use[0] lt -180 then begin
  ;  rotate_coord_use[0] = rotate_coord_use[0] + 360.
  ;endif
endif

; Delete areas on opposite side of globe for orthographic projection
if keyword_set( projection ) then begin
  if strpos( projection, 'orthographic' ) eq 0 then begin
    ; Determine the centre of the projection
    temp_centre = strsplit( projection, '=,', count=n_temp_centre, extract=1 )
    temp_centre = float( temp_centre[1:2] )
    ; Delete all points on the far side of globe from this projection
    if keyword_set( project_x ) then begin
      temp_x = project_x
    endif else begin
      temp_x = xvec_use
    endelse
    if keyword_set( project_y ) then begin
      temp_y = project_y
    endif else begin
      temp_y = yvec_use
    endelse
    temp = geo_dist( temp_centre[0], temp_centre[1], temp_x, temp_y )
    id = where( temp gt 10000, n_id )
    if n_id gt 0 then data_use[id] = !values.f_nan
    temp = 0
    temp_x = 0
    temp_y = 0
    ; If there is nothing to plot then quit
    if max( finite( data_use ) ) eq 0 then return
  endif
endif

;***********************************************************************
; Plot Map

; Determine the longitude and latitude extent
if n_limit eq 4 then begin
  limit_xvec_min = limit[0]
  limit_xvec_max = limit[2]
  limit_yvec_min = limit[1]
  limit_yvec_max = limit[3]
endif else begin
  limit_xvec_min = min( limit[[0,2,4,6]] )
  limit_xvec_max = max( limit[[0,2,4,6]] )
  limit_yvec_min = min( limit[[1,3,5,7]] )
  limit_yvec_max = max( limit[[1,3,5,7]] )
endelse

; Contour over map.
; If we have irregularly gridded data
if irregular_opt eq 1 then begin
  ; Iterate through elements in the data
  for i_data = 0l, n_data - 1l do begin
    ; Calculate dimensional distances from this point
    near_xvec_delta = xvec_use - xvec_use[i_data]
    near_yvec_delta = yvec_use - yvec_use[i_data]
    ; Find n_vertex neighbouring points to this one
    near_delta = near_xvec_delta ^ 2 + near_yvec_delta ^ 2
    id_data = sort( near_delta )
    n_vertex = n_vertex_0
    id_data = id_data[1l:n_vertex]
    near_xvec_delta = near_xvec_delta[id_data]
    near_yvec_delta = near_yvec_delta[id_data]
    near_delta = near_delta[id_data]
    if outline_opt eq 1 then near_data = data_use[id_data]
    ; Sort vertices in counterclockwise order from the smallest angle
    near_ang = atan( near_yvec_delta, near_xvec_delta )
    id_near = sort( near_ang )
    near_xvec_delta = near_xvec_delta[id_near]
    near_yvec_delta = near_yvec_delta[id_near]
    near_ang = near_ang[id_near]
    near_delta = near_delta[id_near]
    if outline_opt eq 1 then near_data = near_data[id_near]
    ; If the largest angle is near or greater than pi
    temp_near_ang_diff = [ near_ang[1:n_vertex-1], near_ang[0] + 2. * !pi ] $
        - near_ang
    if max( temp_near_ang_diff ) gt 0.9 * !pi then begin
      ; Retain the closest points
      id_near = sort( near_delta )
      ; If more than 1.5*pi is missing then retain only half the points
      if max( temp_near_ang_diff ) gt 1.4 * !pi then begin
        n_vertex = n_vertex / 2
      ; Otherwise retain 3/4 of the points
      endif else begin
        n_vertex = 3 * n_vertex / 4
      endelse
      id_near = id_near[0:n_vertex-1]
      near_xvec_delta = near_xvec_delta[id_near]
      near_yvec_delta = near_yvec_delta[id_near]
      near_ang = near_ang[id_near]
      near_delta = near_delta[id_near]
      if outline_opt eq 1 then near_data = near_data[id_near]
      ; Create fake points reflected across our point
      near_xvec_delta = [ near_xvec_delta, -near_xvec_delta ]
      near_yvec_delta = [ near_yvec_delta, -near_yvec_delta ]
      near_delta = [ near_delta, near_delta ]
      near_ang = atan( near_yvec_delta, near_xvec_delta )
      if outline_opt eq 1 then begin
        ; Set data values according to nearest available point, rather than the 
        ; reflected point
        near_data = [ near_data, near_data ]
        for i_near = 0, n_vertex - 1 do begin
          temp = ( xvec_use - near_xvec_delta[n_vertex+i_near] $
              - xvec_use[i_data] ) ^ 2 $
              + ( yvec_use - near_yvec_delta[n_vertex+i_near] $
              - yvec_use[i_data] ) ^ 2
          id_sort = sort( temp )
          if id_sort[0] eq i_data then begin
            near_data[n_vertex+i_near] = data_use[id_sort[1]]
          endif else begin
            near_data[n_vertex+i_near] = data_use[id_sort[0]]
          endelse
        endfor
      endif
      n_vertex = n_vertex * 2
      ; Restrict to n_vertex_0 points
      if n_vertex gt n_vertex_0 then begin
        id_near = sort( near_ang )
        near_xvec_delta = near_xvec_delta[id_near]
        near_yvec_delta = near_yvec_delta[id_near]
        near_ang = near_ang[id_near]
        near_delta = near_delta[id_near]
        if outline_opt eq 1 then near_data = near_data[id_near]
        temp_near_ang_diff $
            = [ near_ang[1:n_vertex-1], near_ang[0] + 2. * !pi ] - near_ang
        id_near = sort( temp_near_ang_diff )
        id_near = id_near[n_vertex-n_vertex_0:n_vertex-1]
        near_xvec_delta = near_xvec_delta[id_near]
        near_yvec_delta = near_yvec_delta[id_near]
        near_ang = 0
        near_delta  = 0
        if outline_opt eq 1 then near_data = near_data[id_near]
        n_vertex = n_vertex_0
      endif
    endif
    ; Determine vertices of polygon
    vertex_xvec = fltarr( n_vertex )
    vertex_yvec = fltarr( n_vertex )
    vertex_xvec[0] = ( near_xvec_delta[n_vertex-1] + near_xvec_delta[0] ) / 2.
    vertex_yvec[0] = ( near_yvec_delta[n_vertex-1] + near_yvec_delta[0] ) / 2.
    for i_vertex = 1, n_vertex - 1 do begin
      vertex_xvec[i_vertex] = ( near_xvec_delta[i_vertex-1] $
          + near_xvec_delta[i_vertex] ) / 2.
      vertex_yvec[i_vertex] = ( near_yvec_delta[i_vertex-1] $
          + near_yvec_delta[i_vertex] ) / 2.
    endfor
    vertex_xvec = vertex_xvec + xvec_use[i_data]
    vertex_yvec = vertex_yvec + yvec_use[i_data]
    ; If we are drawing filled contours
    if ( outline_opt eq 0 ) and ( no_plot_opt eq 0 ) then begin
      ; Determine colour
      id = max( where( levels - data_use[i_data] lt 0 ) )
      if id eq -1 then id = 0
      ; Plot cell
      polyfill, vertex_xvec, vertex_yvec, color=c_colors[id], noclip=1-clip_opt
    ; If we are drawing outline contours
    endif else begin
      ; Determine the contour level of this cell
      id_level = max( where( levels - data_use[i_data] lt 0 ) )
      ; Iterate through neighbouring cells
      for i_near = 0, n_vertex - 1 do begin
        ; Determine the contour level of the cell to the east
        id_level_1 = max( where( levels - near_data[i_near] lt 0 ) )
        ; If the contour levels are different then plot a contour line
        if id_level_1 ne id_level then begin
          if i_near eq n_vertex - 1 then begin
            temp_xvec = vertex_xvec[[n_vertex-1,0]]
            temp_yvec = vertex_yvec[[n_vertex-1,0]]
          endif else begin
            temp_xvec = vertex_xvec[[i_near,i_near+1]]
            temp_yvec = vertex_yvec[[i_near,i_near+1]]
          endelse
          if no_plot_opt eq 0 then begin
            plots, temp_xvec, temp_yvec, color=color, thick=thick, $
                linestyle=outline_linestyle, noclip=1-clip_opt
          endif
          ; Plot a down-/up-hill tick if requested
          if keyword_set( point_ticks ) and ( no_plot_opt eq 0 ) then begin
            d_tick_xvec = [ 0, $
                ( near_xvec_delta[i_near] $
                - ( mean( vertex_xvec ) - xvec_use[i_data] ) ) / 4. ]
            d_tick_yvec = [ 0, $
                ( near_xvec_delta[i_near] $
                - ( mean( vertex_yvec ) - yvec_use[i_data] ) ) / 4. ]
            tick_xvec = [ 0, 0 ] + mean( temp_xvec )
            tick_yvec = [ 0, 0 ] + mean( temp_yvec )
            temp = point_ticks * sign( id_level_1 - id_level )
            oplot, tick_xvec+temp*d_tick_xvec, tick_yvec+temp*d_tick_yvec, $
                color=color, thick=thick
          endif
        endif
      endfor
    endelse
  endfor
; If we have regularly gridded data
endif else begin
  ; Initialise array identifying shape membership
  index_shape = lonarr( n_xvec, n_yvec )
  index_shape_max = 1
  ; Initialise the vector remembering removed (unused) shape counter
  id_shape_unused = 0l
  n_id_shape_unused = n_elements( id_shape_unused ) - 1l
  ; Iterate through rows
  for i_yvec = 0l, n_yvec - 1l do begin
    ; If this row is within the mapping range
    if ( yvec_use[i_yvec] ge limit_yvec_min ) $
        and ( yvec_use[i_yvec] le limit_yvec_max ) then begin
      ; Copy data for this row and the one below
      temp_data_use_row = data_use[*,i_yvec]
      temp_index_shape_row = index_shape[*,i_yvec]
      if i_yvec gt 0l then begin
        if yvec_use[i_yvec-1l] ge limit_yvec_min then begin
          temp_data_use_row_below = data_use[*,i_yvec-1l]
          temp_index_shape_row_below = index_shape[*,i_yvec-1l]
        endif
      endif
      ; Iterate through columns
      for i_xvec = 0l, n_xvec - 1l do begin
        ; If this column is within the mapping range
        if ( xvec_use[i_xvec] ge limit_xvec_min ) $
            and ( xvec_use[i_xvec] le limit_xvec_max ) then begin
          ; Copy data for this cell
          temp_data_use = temp_data_use_row[i_xvec]
          ; Determine if this cell has data
          cell_finite_opt = finite( temp_data_use )
          ; Note the level of this cell
          if cell_finite_opt eq 1 then begin
            id_level_here = max( where( levels - temp_data_use lt 0 ) )
            if id_level_here eq -1 then id_level_here = 0
          endif else begin
            id_level_here = -2
          endelse
          ; Initialise the value for this cell (which may already have been 
          ; calculated to something other than 0)
          temp_index = 0l
          id_level_left = -1
          id_level_below = -1
          ; Check if this matches the level of the cell to the left
          if i_xvec gt 0 then begin
            if xvec_use[i_xvec-1l] ge limit_xvec_min then begin
              temp_data_use_left = temp_data_use_row[i_xvec-1l]
              if cell_finite_opt eq 1 then begin
                if finite( temp_data_use_left ) eq 1 then begin
                  id_level_left = max( $
                      where( levels - temp_data_use_left lt 0 ) )
                  if id_level_left eq -1 then id_level_left = 0
                endif
              endif else begin
                if finite( temp_data_use_left ) eq 0 then id_level_left = -2
              endelse
              if id_level_left eq id_level_here then begin
                temp_index = temp_index_shape_row[i_xvec-1l]
              endif
            endif
          endif
          ; Check if this matches the level of the cell below
          ; (Check even if we already assigned to a shape above, in case we 
          ; find that we have now arrived at where the same shape was 
          ; assigned a different identifier in the previous row.)
          if i_yvec gt 0 then begin
            if yvec_use[i_yvec-1l] ge limit_yvec_min then begin
              temp_data_use_below = temp_data_use_row_below[i_xvec]
              if cell_finite_opt eq 1 then begin
                if finite( temp_data_use_below ) eq 1 then begin
                  id_level_below = max( $ 
                      where( levels - temp_data_use_below lt 0 ) )
                  if id_level_below eq -1 then id_level_below = 0
                endif
              endif else begin
                if finite( temp_data_use_below ) eq 0 then id_level_below = -2
              endelse
              if id_level_below eq id_level_here then begin
                ; Copy the index value for the cell below
                temp_index_below = temp_index_shape_row_below[i_xvec]
                ; If we should adopt the value for the cell below
                if temp_index eq 0 then begin
                  temp_index = temp_index_below
                ; If we need to revise the shape identifier on this cell
                endif else if temp_index ne temp_index_below then begin
                  ; If we need to revise the shape identifier on cells to the 
                  ; left as well
                  if id_level_left eq id_level_below then begin
                    if abs( temp_index_below ) gt abs( temp_index ) then begin
                      temp_index_use = temp_index
                      temp_index_discard = temp_index_below
                    endif else begin
                      temp_index_use = temp_index_below
                      temp_index_discard = temp_index
                    endelse
                    id = where( index_shape eq temp_index_discard, n_id )
                    index_shape[id] = temp_index_use
                    id = where( temp_index_shape_row eq temp_index_discard, $
                        n_id )
                    if n_id gt 0 then begin
                      temp_index_shape_row[id] = temp_index_use
                    endif
                    id = where( temp_index_shape_row_below $
                        eq temp_index_discard, n_id )
                    if n_id gt 0 then begin
                      temp_index_shape_row_below[id] = temp_index_use
                    endif
                    ; Change the shape identifier
                    temp_index = temp_index_use
                    ; Record the identifier that was going to be used
                    id_shape_unused = [ id_shape_unused, $
                        abs( temp_index_discard ) ]
                    if abs( temp_index ) eq index_shape_max then begin
                      index_shape_max = max( abs( index_shape ) )
                    endif
                  endif else begin
                    ; Change the shape identifier
                    temp_index = temp_index_below
                  endelse
                endif
              endif
            endif
          endif
          ; If we have not matched to a neighbouring cell then start a new
          ; shape
          if temp_index eq 0 then begin
            n_id_shape_unused = n_elements( id_shape_unused ) - 1l
            if n_id_shape_unused gt 0 then begin
              temp_index = id_shape_unused[n_id_shape_unused]
              id_shape_unused = id_shape_unused[0l:n_id_shape_unused-1l]
            endif else begin
              temp_index = max( [ 2, index_shape_max + 1 ] )
            endelse
            if temp_index gt index_shape_max then index_shape_max = temp_index
            ; Flag NaN shapes
            if cell_finite_opt eq 0 then temp_index = -abs( temp_index )
          endif
          ; Record the value
          index_shape[i_xvec,i_yvec] = temp_index
          temp_index_shape_row[i_xvec] = temp_index
        endif
      endfor
    endif
  endfor
  ; Count the number of shapes
  n_shape = max( abs( index_shape ) ) - 1
  ; Sort shapes by size (largest to smallest).  If a shape is nested in a 
  ; larger shape then it will be overwritten by the larger shape if the 
  ; larger is plotted afterward.
  area = fltarr( n_shape )
  for i_shape = 0l, n_shape - 1l do begin
    id = where( abs( index_shape ) eq i_shape + 2l, n_id )
    area[i_shape] = n_id
  endfor
  id_sort_shape = reverse( sort( area ) )
  ; Iterate through shapes
  for i_shape = 0l, n_shape - 1l do begin
    ; Find all cells in this shape
    temp_id_sort_shape = id_sort_shape[i_shape] + 2l
    id_shape = where( index_shape eq temp_id_sort_shape, n_id_shape )
    if n_id_shape eq 0 then begin
      temp_id_sort_shape = -1 * temp_id_sort_shape
      id_shape = where( index_shape eq temp_id_sort_shape, n_id_shape )
    endif
    if n_id_shape gt 0 then begin
      ; Determine the x and y coordinates of the first cell
      ; (Note we are assuming it is on the border, which it has to be.)
      id_shape_x = id_shape[0] mod n_xvec
      id_shape_y = id_shape[0] / n_xvec
      ; Initialise counter for border points
      i_point = 0l
      ; Initialise a vector for instructing the order of sides to examine in 
      ; the next cell.  Start with left, top, right, bottom ([0,1,2,3]).
      index_side = [ 0, 1, 2, 3 ]
      ; Iterate through cells until we return to the original spot
      flag_done_shape = 0
      while flag_done_shape eq 0 do begin
        ; Determine the shape values of this cell and the one to the left, top, 
        ; right, and bottom, as well as the distance to the edges with those 
        ; neighbouring cells from the centre of this cell
        temp_shape_here = index_shape[id_shape_x,id_shape_y]
        temp_shape_near = -1 + intarr( 4 )
        temp_shape_dist = fltarr( 4 )
        if id_shape_x ne 0 then begin
          temp_shape_near[0] = index_shape[id_shape_x-1,id_shape_y]
          ;id = id_shape_x + [ -1, 0 ]
          temp_shape_dist[0] = xvec_use[id_shape_x] $
              - ( xvec_use[id_shape_x] - xvec_use[id_shape_x-1] ) / 2.
        endif else begin
          temp_shape_dist[0] = xvec_use[0] - ( xvec_use[1] - xvec_use[0] ) / 2.
        endelse
        if id_shape_y ne n_yvec - 1 then begin
          temp_shape_near[1] = index_shape[id_shape_x,id_shape_y+1]
          temp_shape_dist[1] = yvec_use[id_shape_y] $
              + ( yvec_use[id_shape_y+1] - yvec_use[id_shape_y] ) / 2.
        endif else begin
          temp_shape_dist[1] = yvec_use[n_yvec-1] $
              + ( yvec_use[n_yvec-1] - yvec_use[n_yvec-2] ) / 2.
        endelse
        if id_shape_x ne n_xvec - 1 then begin
          temp_shape_near[2] = index_shape[id_shape_x+1,id_shape_y]
          temp_shape_dist[2] = xvec_use[id_shape_x] $
              + ( xvec_use[id_shape_x+1] - xvec_use[id_shape_x] ) / 2.
        endif else begin
          temp_shape_dist[2] = xvec_use[n_xvec-1] $
              + ( xvec_use[n_xvec-1] - xvec_use[n_xvec-2] ) / 2.
        endelse
        if id_shape_y ne 0 then begin
          temp_shape_near[3] = index_shape[id_shape_x,id_shape_y-1]
          temp_shape_dist[3] = yvec_use[id_shape_y] $
              - ( yvec_use[id_shape_y] - yvec_use[id_shape_y-1] ) / 2.
        endif else begin
          temp_shape_dist[3] = yvec_use[0] - ( yvec_use[1] - yvec_use[0] ) / 2.
        endelse
        ; Iterate through sides until we come to a non-border
        flag_done_sides = 0
        for i_side = 0, 3 do begin
          if flag_done_sides eq 0 then begin
            ; Determine the identity of the side to examine
            ; (0=left, 1=top, 2=right, 3=bottom)
            id_side = index_side[i_side]
            ; If this is on the border
            if temp_shape_here ne temp_shape_near[id_side] then begin
              ; If this is the first edge on the border then record the first 
              ; (counterclockwise-most) point on the edge
              if i_point eq 0l then begin
                id_before = ( id_side - 1 + 4 ) mod 4
                if ( id_side eq 0 ) or ( id_side eq 2 ) then begin
                  border = [ temp_shape_dist[id_side], $
                      temp_shape_dist[id_before] ]
                endif else begin
                  border = [ temp_shape_dist[id_before], $
                      temp_shape_dist[id_side] ]
                endelse
                i_point = 1l
              endif
              ; Record the second (clockwise-most) point on the edge
              border = [ [ border ], [ 0, 0 ] ]
              id_after = ( id_side + 1 ) mod 4
              if ( id_side eq 0 ) or ( id_side eq 2 ) then begin
                border[*,i_point] = [ temp_shape_dist[id_side], $
                   temp_shape_dist[id_after] ]
              endif else begin
                border[*,i_point] = [ temp_shape_dist[id_after], $
                   temp_shape_dist[id_side] ]
              endelse
              i_point = i_point + 1l
              ; Check if we have returned to the start point
              temp = abs( border[0,i_point-1l] - border[0,0] )
              if temp lt d_xvec_use_min / 2. then begin
                temp = abs( border[1,i_point-1l] - border[1,0] )
                if temp lt d_yvec_use_min / 2. then begin
                  ; If we have returned, then exit the loop for this shape
                  flag_done_sides = 1
                  flag_done_shape = 1
                endif
              endif
            ; If this is not a border then proceed to that cell
            endif else begin
              ; If this is the left border, then move to the left
              if id_side eq 0 then begin
                id_shape_x = id_shape_x - 1
                index_side = [ 3, 0, 1, 2 ]
              ; If this is the top border, then move up
              endif else if id_side eq 1 then begin
                id_shape_y = id_shape_y + 1
                index_side = [ 0, 1, 2, 3 ]
              ; If this is the right border, then move to the right
              endif else if id_side eq 2 then begin
                id_shape_x = id_shape_x + 1
                index_side = [ 1, 2, 3, 0 ]
              ; If this is the bottom border, then move down
              endif else if id_side eq 3 then begin
                id_shape_y = id_shape_y - 1
                index_side = [ 2, 3, 0, 1 ]
              endif
              ; Flag the exit from this cell
              flag_done_sides = 1
            endelse
          endif
        endfor
      endwhile
      n_point = n_elements( border[0,*] )
      ; If plotting is requested
      if no_plot_opt eq 0 then begin
        ; If a bespoke projection is requested
        if keyword_set( project_x ) and keyword_set( project_y ) then begin
          ; Interpolate border vertices through bespoke projection
          for i_point = 0, n_point - 1 do begin
            ; Determine where we are in the x dimension
            temp = min( abs( xvec_use - border[0,i_point] ), id_x )
            if id_x eq 0 then begin
              id_x = [ 0, 0, 1 ]
            endif else if id_x eq n_xvec - 1 then begin
              id_x = n_xvec - [ 2, 1, 1 ]
            endif else begin
              id_x = id_x + [ -1, 0, 1 ]
            endelse
            ; Determine where we are in the y dimension
            temp = min( abs( yvec_use - border[1,i_point] ), id_y )
            if id_y eq 0 then begin
              id_y = [ 0, 0, 1 ]
            endif else if id_y eq n_yvec - 1 then begin
              id_y = n_xvec - [ 2, 1, 1 ]
            endif else begin
              id_y = id_y + [ -1, 0, 1 ]
            endelse
            ; Determine the angle of rotation in the projection
            temp_project_x = project_x[id_x[2],id_y[1]] $
                - project_x[id_x[0],id_y[1]]
            if keyword_set( wrap_x ) then begin
              if project_x[id_x[2],id_y[1]] $
                  gt wrap_x[1] - 0.05 * ( wrap_x[1] - wrap_x[0] ) then begin
                if project_x[id_x[0],id_y[1]] $
                    lt wrap_x[0] + 0.05 * ( wrap_x[1] - wrap_x[0] ) then begin
                  temp_project_x = temp_project_x - 360.
                endif
              endif
            endif
            temp_xvec = xvec_use[id_x[2]] - xvec_use[id_x[0]]
            temp_project_xy = project_y[id_x[2],id_y[1]] $
                - project_y[id_x[0],id_y[1]]
            temp_yvec = yvec_use[id_y[2]] - yvec_use[id_y[0]]
            temp_ang = atan( $
                temp_project_xy / temp_yvec / ( id_y[2] - id_y[0] ), $
                temp_project_x / temp_xvec / ( id_x[2] - id_x[0] ) )
            ; Interpolate the projection
            temp_project_y = project_y[id_x[1],id_y[2]] $
                - project_y[id_x[1],id_y[0]]
            border[0,i_point] = project_x[id_x[1],id_y[1]] $
                + cos( temp_ang ) * temp_project_x / temp_xvec $
                * ( border[0,i_point] - xvec_use[id_x[1]] ) $
                + sin( 0 ) * temp_project_y / temp_yvec $
                * ( border[1,i_point] - yvec_use[id_y[1]] )
            border[1,i_point] = project_y[id_x[1],id_y[1]] $
                + sin( 0 ) * temp_project_x / temp_xvec $
                * ( border[0,i_point] - xvec_use[id_x[1]] ) $
                + cos( temp_ang ) * temp_project_y / temp_yvec $
                * ( border[1,i_point] - yvec_use[id_y[1]] )
          endfor
        endif
        ; If a coordinate rotation is requested
        if keyword_set( rotate_coord ) then begin
          ; Perform rotation
          convert_rotated_grid, in_lon=reform(border[0,*]), $
              in_lat=reform(border[1,*]), out_north_lon=rotate_coord_use[0], $
              out_north_lat=rotate_coord_use[1], out_lon=temp_lon, $
              out_lat=temp_lat
          ; Fix some longitude wrapping issues
          id = where( ( temp_lon gt 179. ) and ( border[0,*] lt -179. ), $
              n_id )
          if n_id gt 0 then temp_lon[id] = -180.
          id = where( ( temp_lon lt -179. ) and ( border[0,*] gt 179. ), $
              n_id )
          if n_id gt 0 then temp_lon[id] = 180.
          ; Record rotation
          border = transpose( [ [ temp_lon ], [ temp_lat ] ] )
        endif
        ; Fix values outside wrapping range
        if keyword_set( wrap_x ) then begin
          id_0 = where( border[0,*] lt wrap_x[0], n_id_0 )
          if n_id_0 gt 0 then begin
            border[0,id_0] = border[0,id_0] - wrap_x[0] + wrap_x[1]
          endif
          id_1 = where( border[0,*] gt wrap_x[1], n_id_1 )
          if n_id_1 gt 0 then begin
            border[0,id_1] = border[0,id_1] - wrap_x[1] + wrap_x[0]
          endif
        endif
        ; Split a shape crossing the wrapping boundary into multiple shapes
        ctr_border = 1
        if keyword_set( wrap_x ) then begin
          for i_point = 2, n_point - 3 do begin
            if border[0,i_point] $
                gt wrap_x[1] - 0.05 * ( wrap_x[1] - wrap_x[0] ) then begin
              if border[0,i_point-1] $
                  lt wrap_x[0] + 0.05 * ( wrap_x[1] - wrap_x[0] ) then begin
                if not( keyword_set( border_id ) ) then begin
                  border_id = 1 + intarr( n_point )
                endif
                ctr_border = ctr_border + 1
                border_id[i_point:n_point-1] = -1
              endif
            endif else if border[0,i_point] $
                lt wrap_x[0] + 0.05 * ( wrap_x[1] - wrap_x[0] ) then begin
              if border[0,i_point-1] $
                  gt wrap_x[1] - 0.05 * ( wrap_x[1] - wrap_x[0] ) then begin
                if not( keyword_set( border_id ) ) then begin
                  border_id = -1 + intarr( n_point )
                endif
                ctr_border = ctr_border + 1
                border_id[i_point:n_point-1] = 2
              endif
            endif
          endfor
          if ctr_border gt 1 then begin
            ctr_point = 0
            for i_point = 1, n_point - 2 do begin
              i_point_1 = i_point + 2 * ctr_point
              if border_id[i_point_1+1] - border_id[i_point_1] le -2 then begin
                border_id = [ border_id[0:i_point_1], border_id[i_point_1], $
                    border_id[i_point_1+1], $
                    border_id[i_point_1+1:n_point+2*ctr_point-1] ]
                border = [ [ border[*,0:i_point_1] ], $
                    [ wrap_x[0], border[1,i_point_1] ], $
                    [ wrap_x[1], border[1,i_point_1+1] ], $
                    [ border[*,i_point_1+1:n_point+2*ctr_point-1] ] ]
                ctr_point = ctr_point + 1
              endif else if border_id[i_point_1+1] - border_id[i_point_1] ge 2 $
                  then begin
                border_id = [ border_id[0:i_point_1], border_id[i_point_1], $
                    border_id[i_point_1+1], $
                    border_id[i_point_1+1:n_point+2*ctr_point-1] ]
                border = [ [ border[*,0:i_point_1] ], $
                    [ wrap_x[1], border[1,i_point_1] ], $
                    [ wrap_x[0], border[1,i_point_1+1] ], $
                    [ border[*,i_point_1+1:n_point+2*ctr_point-1] ] ]
                ctr_point = ctr_point + 1
              endif
            endfor
            n_point = n_point + 2 * ctr_point
            border_id = abs( border_id )
          endif
        endif
        ; If a geographic projection is requested
        if keyword_set( projection ) then begin
          ; Add numerical fix for edges of plot
          if keyword_set( wrap_x ) then begin
            temp = ( wrap_x[1] - wrap_x[0] ) * 0.00001
            id = where( border[0,*] gt wrap_x[1] - temp, n_id )
            if n_id gt 0 then border[0,id] = wrap_x[1] - temp
            id = where( border[0,*] lt wrap_x[0] + temp, n_id )
            if n_id gt 0 then border[0,id] = wrap_x[0] + temp
          endif
          ; The Hammer projection
          if projection eq "hammer" then begin
            temp = sqrt( 1. + cos( degrad * border[1,*] ) $
                * cos( degrad * border[0,*] / 2. ) )
            temp_border_x = 2. * cos( degrad * border[1,*] ) $
                * sin( degrad * border[0,*] / 2. ) / temp
            temp_border_y = sin( degrad * border[1,*] ) / temp
          ; The orthographic projection
          endif else if strpos( projection, 'orthographic' ) eq 0 then begin
            ; Determine the centre of the projection
            temp_centre = strsplit( projection, '=,', count=n_temp_centre, $
                extract=1 )
            temp_centre = float( temp_centre[1:2] )
            ; Perform the projection
            temp_border_x = cos( degrad * border[1,*] ) $
                * sin( degrad * ( border[0,*] - temp_centre[0] ) )
            temp_border_y = cos( degrad * temp_centre[1] ) $
                * sin( degrad * border[1,*] ) $
                - sin( degrad * temp_centre[1] ) * cos( degrad * border[1,*] ) $
                * cos( degrad * ( border[0,*] - temp_centre[0] ) )
          ; An unsupported projection
          endif else begin
            stop
          endelse
          temp_border_x = reform( temp_border_x )
          temp_border_y = reform( temp_border_y )
        ; If no projection is requested (i.e. rectangular cylindrical)
        endif else begin
          temp_border_x = reform( border[0,*] )
          temp_border_y = reform( border[1,*] )
        endelse
        ; Remove middle points in straight lines
        ; (deals with numerical issues with GDL's polyfill)
        ;n_point = i_point
        n_point = n_elements( temp_border_x )
        temp_remove = intarr( n_point )
        for i_point = 1l, n_point - 2l do begin
          if ( temp_border_x[i_point] eq temp_border_x[i_point-1] ) $
              and ( temp_border_x[i_point] eq temp_border_x[i_point+1] ) $
              then begin
            temp_remove[i_point] = 1
          endif else if ( temp_border_y[i_point] eq temp_border_y[i_point-1] ) $
              and ( temp_border_y[i_point] eq temp_border_y[i_point+1] ) $
              then begin
            temp_remove[i_point] = 1
          endif
        endfor
        id = where( temp_remove eq 0 )
        temp_border_x = temp_border_x[id]
        temp_border_y = temp_border_y[id]
        if ctr_border gt 1 then border_id = border_id[id]
        id = 0
        temp_remove = 0
        ; If we are only outlining the shape
        if ( outline_opt eq 1 ) and ( temp_id_sort_shape ge 0 ) then begin
          ; Plot the outline of the polygon
          if ctr_border eq 1 then begin
            ; Plot the one sub-shape
            plots, temp_border_x, temp_border_y, color=color, thick=thick, $
                linestyle=outline_linestyle, noclip=1-clip_opt
          endif else begin
            ; Plot the multiple sub-shapes
            for i_border = 0, ctr_border - 1 do begin
              id = where( border_id eq i_border + 1, n_id )
              if n_id ge 3 then begin
                plots, temp_border_x[id], temp_border_y[id], color=color, $
                    thick=thick, linestyle=outline_linestyle, noclip=1-clip_opt
              endif
            endfor
          endelse
        ; If we are filling
        endif else begin
          ; Determine the colour
          if temp_id_sort_shape ge 0 then begin
            id_shape_x = id_shape[0] mod n_xvec
            id_shape_y = id_shape[0] / n_xvec
            id_level = max( $
                where( levels - data_use[id_shape_x,id_shape_y] lt 0 ) )
            if id_level eq -1 then id_level = 0
            temp_color = c_colors[id_level]
          endif else begin
            if no_nan_opt eq 0 then begin
              temp_color = !p.background
            endif else begin
              temp_color = -1
            endelse
          endelse
          ; Plot the filled polygon
          if temp_color ge 0 then begin
            if ctr_border eq 1 then begin
              ; Plot the one sub-shape
              polyfill, temp_border_x, temp_border_y, color=temp_color, $
                  noclip=1-clip_opt
            endif else begin
              ; Plot the multiple sub-shapes
              for i_border = 0, ctr_border - 1 do begin
                id = where( border_id eq i_border + 1, n_id )
                if n_id ge 3 then begin
                  polyfill, temp_border_x[id], temp_border_y[id], $
                      color=temp_color, noclip=1-clip_opt
                endif
              endfor
            endelse
          endif
        endelse
        ; Delete sub-shape information
        border_id = 0
      endif
      ;print,i_shape,n_elements( border[0,*] )
      ; Record border coordinates for output, if requested
      if return_border_opt eq 1 then begin
        ; Determine the feature this belongs to (each feature potentially 
        ; containing multiple shapes)
        id_shape_x = id_shape[0] mod n_xvec
        id_shape_y = id_shape[0] / n_xvec
        id_level = max( $
            where( levels - data_use[id_shape_x,id_shape_y] lt 0 ) )
        if id_level eq -1 then id_level = 0
        ; Determine the number of vertices to add
        n_temp = n_elements( border[0,*] )
        ; Initialise the border information array for this shape, including 
        ; coordinates
        temp_border = [ border, fltarr( 2, n_temp ) ]
        ; Add feature identifier
        temp_border[2,*] = id_level
        ; Determine the shape identifier within the feature
        if not( keyword_set( border_out ) ) then begin
          id_subshape = 0
        endif else begin
          id = where( border_out[2,*] eq id_level, n_id )
          if n_id eq 0 then begin
            id_subshape = 0
          endif else begin
            id_subshape = max( border_out[3,id] ) + 1
          endelse
        endelse
        temp_border[3,*] = id_subshape
        ; Copy this shape to the output array
        if not( keyword_set( border_out ) ) then begin
          border_out = temp_border
        endif else begin
          border_out = [ [ border_out ], [ temp_border ] ]
        endelse
      endif
    endif
  endfor
endelse

;***********************************************************************
; The End

;stop
return
END
